/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
/*
 * Synopsis:
 *	int disrfst(int stream, size_t achars, char *value)
 *
 *	Gets a Data-is-Strings character string from <stream> and converts it
 *	into an ASCII NUL-terminated string, and puts the string into <value>,
 *	a pre-allocated string, <achars> long.  The character string in <stream>
 *	consists of an unsigned integer, followed by a number of characters
 *	determined by the unsigned integer.
 *
 *	Disrfst returns DIS_SUCCESS if everything works well.  It returns an
 *	error code otherwise.  In case of an error, the <stream> character
 *	pointer is reset, making it possible to retry with some other conversion
 *	strategy, and the first character of <value> is set to ASCII NUL.
 */

#include <pbs_config.h>   /* the master config generated by configure */

#include <assert.h>
#include <stddef.h>
#include <stdlib.h>

#ifndef NDEBUG
#include <string.h>
#endif

#include "dis.h"
#include "dis_.h"

int disrfst(stream, achars, value)
    int			stream;
    size_t		achars;
    char		*value;
    {
        int		locret;
        int		negate;
        unsigned	count;

	assert(value != NULL);
	assert(dis_gets != NULL);
	assert(disr_commit != NULL);

	locret = disrsi_(stream, &negate, &count, 1);
	if (locret == DIS_SUCCESS) {
		if (negate)
		        locret = DIS_BADSIGN;
		else if (count > achars)
		        locret = DIS_OVERFLOW;
		else if ((*dis_gets)(stream, value, (size_t)count) !=
						(size_t)count)
		        locret = DIS_PROTO;
#ifndef NDEBUG
		else if (memchr(value, 0, (size_t)count))
		        locret = DIS_NULLSTR;
#endif
		else
		        value[count] = '\0';
	}
	locret = (*disr_commit)(stream, locret == DIS_SUCCESS) ?
	    DIS_NOCOMMIT : locret;
	if (locret != DIS_SUCCESS)
	        *value = '\0';
	return (locret);
}
