/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
/*
 * Synopsis:
 * 	int diswf(int stream, float value)
 *
 *	Converts <value> into a Data-is-Strings floating point number and sends
 *	it to <stream>.  The converted number consists of two consecutive signed
 *	integers.  The first is the coefficient, at most <ndigs> long, with its
 *	implied decimal point at the low-order end.  The second is the exponent
 *	as a power of 10.
 *
 *	Returns DIS_SUCCESS if everything works well.  Returns an error code
 *	otherwise.  In case of an error, no characters are sent to <stream>.
 */

#include <pbs_config.h>   /* the master config generated by configure */

#include <assert.h>
#include <stddef.h>
#include <stdio.h>

#include "dis.h"
#include "dis_.h"
#undef diswf

int diswf(stream, value)
    int			stream;
    double		value;
    {
	int		c;
	int		expon;
	unsigned	ndigs;
	int		negate;
	int		retval;
	unsigned	pow2;
	char		*cp;
	char		*ocp;
	double		dval;

	assert(stream >= 0);
	assert(dis_puts != NULL);
	assert(disw_commit != NULL);

/* Make zero a special case.  If we don't it will blow exponent		*/
/* calculation.								*/
	if (value == 0.0) {
		retval = (*dis_puts)(stream, "+0+0", 4) != 4 ?
		    DIS_PROTO : DIS_SUCCESS;
		return (((*disw_commit)(stream, retval == DIS_SUCCESS) < 0) ?
		       DIS_NOCOMMIT : retval);
	}
/* Extract the sign from the coefficient.				*/
        dval = (negate = value < 0.0) ? -value : value;
/* Detect and complain about the infinite form.				*/
	if (dval > FLT_MAX)
	        return (DIS_HUGEVAL);
/* Compute the integer part of the log to the base 10 of dval.  As a	*/
/* byproduct, reduce the range of dval to the half-open interval,       */
/* [1, 10).								*/
	if (dis_dmx10 == 0)
	        disi10d_();
	expon = 0;
	pow2 = dis_dmx10 + 1;
	if (dval < 1.0) {
		do {
			if (dval < dis_dn10[--pow2]) {
				dval *= dis_dp10[pow2];
				expon += 1 << pow2;
			}
		} while (pow2);
		dval *= 10.0;
		expon = -expon - 1;
	} else {
		do {
			if (dval >= dis_dp10[--pow2]) {
				dval *= dis_dn10[pow2];
				expon += 1 << pow2;
			}
		} while (pow2);
	}
/* Round the value to the last digit					*/
	dval += 5.0 * disp10d_(-FLT_DIG);
	if (dval >= 10.0) {
		expon++;
		dval *= 0.1;
	}
/* Starting in the middle of the buffer, convert coefficient digits,	*/
/* most significant first.						*/
	ocp = cp = &dis_buffer[DIS_BUFSIZ - FLT_DIG];
	ndigs = FLT_DIG;
	do {
		c = dval;
		dval = (dval - c) * 10.0;
		*ocp++ = c + '0';
	} while (--ndigs);
/* Eliminate trailing zeros.						*/
	while (*--ocp == '0');
/* The decimal point is at the low order end of the coefficient		*/
/* integer, so adjust the exponent for the number of digits in the	*/
/* coefficient.								*/
	ndigs = ++ocp - cp;
	expon -= ndigs - 1;
/* Put the coefficient sign into the buffer, left of the coefficient.	*/
	*--cp = negate ? '-' : '+';
/* Insert the necessary number of counts on the left.			*/
	while (ndigs > 1)
	        cp = discui_(cp, ndigs, &ndigs);
/* The complete coefficient integer is done.  Put it out.		*/
	retval = (*dis_puts)(stream, cp, (size_t)(ocp - cp)) < 0 ?
	    DIS_PROTO : DIS_SUCCESS;
/* If that worked, follow with the exponent, commit, and return.	*/
	if (retval == DIS_SUCCESS)
		return (diswsi(stream, expon));
/* If coefficient didn't work, negative commit and return the error.	*/
	return (((*disw_commit)(stream, FALSE) < 0)  ? DIS_NOCOMMIT : retval);
}
