/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
 
static char ident[] = "@(#) $RCSfile: dec_rpyc.c,v $ $Revision: 2.1 $";

/*
 * decode_DIS_replyCmd() - decode a Batch Protocol Reply Structure for a Command
 * 
 *	This routine decodes a batch reply into the form used by commands.
 *	The only difference between this and the server version is on status
 *	replies.  For commands, the attributes are decoded into a list of
 *	attrl structure rather than the server's svrattrl.
 *
 * 	batch_reply structure defined in libpbs.h, it must be allocated
 *	by the caller.
 */
 
#include <pbs_config.h>   /* the master config generated by configure */

#include <sys/types.h>
#include <stdlib.h>
#include "libpbs.h"
#include "dis.h"

int decode_DIS_replyCmd(sock, reply)
	int		    sock;
	struct batch_reply *reply;
{
	int		      ct;
	int		      i;
	struct brp_select    *psel;
	struct brp_select   **pselx;
	struct brp_cmdstat   *pstcmd;
	struct brp_cmdstat  **pstcx;
	int		      rc = 0;

	/* first decode "header" consisting of protocol type and version */

	i = disrui(sock, &rc);
	if (rc != 0) return rc;
	if (i != PBS_BATCH_PROT_TYPE) return DIS_PROTO;
	i = disrui(sock, &rc);
	if (rc != 0) return rc;
	if (i != PBS_BATCH_PROT_VER) return DIS_PROTO;

	/* next decode code, auxcode and choice (union type identifier) */

	reply->brp_code    = disrsi(sock, &rc);
	if (rc) return rc;
	reply->brp_auxcode = disrsi(sock, &rc);
	if (rc) return rc;
	reply->brp_choice  = disrui(sock, &rc);
	if (rc) return rc;


	switch (reply->brp_choice) {

	    case BATCH_REPLY_CHOICE_NULL:
		break;	/* no more to do */

	    case BATCH_REPLY_CHOICE_Queue:
	    case BATCH_REPLY_CHOICE_RdytoCom:
	    case BATCH_REPLY_CHOICE_Commit:
		if (rc = disrfst(sock, PBS_MAXSVRJOBID+1,reply->brp_un.brp_jid))
			return (rc);
		break;

	    case BATCH_REPLY_CHOICE_Select:

		/* have to get count of number of strings first */

		reply->brp_un.brp_select = (struct brp_select *)0;
		pselx = &reply->brp_un.brp_select;
		ct = disrui(sock, &rc);
		if (rc) return rc;

		while (ct--) {
			psel = (struct brp_select *)malloc(sizeof (struct brp_select));
			if (psel == 0) return DIS_NOMALLOC;
			psel->brp_next = (struct brp_select *)0;
			psel->brp_jobid[0] = '\0';
			rc = disrfst(sock, PBS_MAXSVRJOBID+1, psel->brp_jobid);
			if (rc) {
				(void)free(psel);
				return rc;
			}
			*pselx = psel;
			pselx  = &psel->brp_next;
		}
		break;

	    case BATCH_REPLY_CHOICE_Status:

		/* have to get count of number of status objects first */

		reply->brp_un.brp_statc = (struct brp_cmdstat *)0;
		pstcx = &reply->brp_un.brp_statc;
		ct = disrui(sock, &rc);
		if (rc) return rc;

		while (ct--) {
			pstcmd = (struct brp_cmdstat *)malloc(sizeof (struct brp_cmdstat));
			if (pstcmd == 0) return DIS_NOMALLOC;

			pstcmd->brp_stlink = (struct brp_cmdstat *)0;
			pstcmd->brp_objname[0] = '\0';
			pstcmd->brp_attrl = (struct attrl *)0;

			pstcmd->brp_objtype = disrui(sock, &rc);
			if (rc == 0) {
				rc = disrfst(sock, PBS_MAXSVRJOBID+1,
					     pstcmd->brp_objname);
			}
			if (rc) {
				(void)free(pstcmd);
				return rc;
			}
			rc = decode_DIS_attrl(sock, &pstcmd->brp_attrl);
			if (rc) {
				(void)free(pstcmd);
				return rc;
			}
			*pstcx = pstcmd;
			pstcx  = &pstcmd->brp_stlink;
		}
		break;
		
	    case BATCH_REPLY_CHOICE_Text:
		
		/* text reply */

		reply->brp_un.brp_txt.brp_str = disrcs(sock,
				(size_t*)&reply->brp_un.brp_txt.brp_txtlen,
				&rc);
		break;

	    case BATCH_REPLY_CHOICE_Locate:

		/* Locate Job Reply */

		rc = disrfst(sock, PBS_MAXDEST+1, reply->brp_un.brp_locate);
		break;

	    case BATCH_REPLY_CHOICE_RescQuery:

		/* Resource Query Reply */

		reply->brp_un.brp_rescq.brq_avail = NULL;
		reply->brp_un.brp_rescq.brq_alloc = NULL;
		reply->brp_un.brp_rescq.brq_resvd = NULL;
		reply->brp_un.brp_rescq.brq_down  = NULL;
		ct = disrui(sock, &rc);
		if (rc) break;
		reply->brp_un.brp_rescq.brq_number = ct;
		reply->brp_un.brp_rescq.brq_avail  = 
					  (int *)malloc(ct * sizeof (int));
		reply->brp_un.brp_rescq.brq_alloc  = 
					  (int *)malloc(ct * sizeof (int));
		reply->brp_un.brp_rescq.brq_resvd  = 
					  (int *)malloc(ct * sizeof (int));
		reply->brp_un.brp_rescq.brq_down   = 
					  (int *)malloc(ct * sizeof (int));

		for (i=0; (i < ct) && (rc == 0); ++i) 
		    *(reply->brp_un.brp_rescq.brq_avail+i) = disrui(sock, &rc);
		for (i=0; (i < ct) && (rc == 0); ++i) 
		    *(reply->brp_un.brp_rescq.brq_alloc+i) = disrui(sock, &rc);
		for (i=0; (i < ct) && (rc == 0); ++i) 
		    *(reply->brp_un.brp_rescq.brq_resvd+i) = disrui(sock, &rc);
		for (i=0; (i < ct) && (rc == 0); ++i) 
		    *(reply->brp_un.brp_rescq.brq_down+i)  = disrui(sock, &rc);
		break;

	    default:
		return -1;
	}

	return rc;
}
