/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
/*
 *	This file contines two main library entries:
 *		pbs_selectjob()
 *		pbs_selstat()
 *
 *
 *	pbs_selectjob() - the SelectJob request
 *		Return a list of job ids that meet certain selection criteria.
*/

#include <pbs_config.h>   /* the master config generated by configure */

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include "libpbs.h"
#include "dis.h"

static char ident[] = "@(#) $RCSfile: pbsD_selectj.c,v $ $Revision: 2.1 $";

static int PBSD_select_put A_((int, int, struct attropl *, char *));
static char **PBSD_select_get A_((int));
static int PBSD_select_put A_((int, int, struct attropl *, char *));

char ** pbs_selectjob(c, attrib, extend)
	int c;
	struct attropl *attrib;
	char *extend;
{

	if (PBSD_select_put(c, PBS_BATCH_SelectJobs, attrib, extend) == 0)
		return ( PBSD_select_get(c) );
	else
		return ((char **)0);
}

/*
 * 	pbs_selstat() - Selectable status
 *		Return status information for jobs that meet certain selection
 *		criteria.  This is a short-cut combination of pbs_selecljob()
 *		and repeated pbs_statjob().
 */

struct batch_status * pbs_selstat(c, attrib, extend)
	int c;
	struct attropl *attrib;
	char *extend;
{
	extern struct batch_status *PBSD_status_get A_((int c));

	if (PBSD_select_put(c, PBS_BATCH_SelStat, attrib, extend) == 0) 
		return ( PBSD_status_get(c) );
	else
		return ((struct batch_status *)0);
}
	

static int PBSD_select_put(c, type, attrib, extend)
	int c;
	int type;
	struct attropl *attrib;
	char * extend;
{
	int rc;
	int sock;

	sock = connection[c].ch_socket;

	/* setup DIS support routines for following DIS calls */

	DIS_tcp_setup(sock);

	if ( (rc = encode_DIS_ReqHdr(sock, type, pbs_current_user)) ||
	     (rc = encode_DIS_attropl(sock, attrib)) ||
	     (rc = encode_DIS_ReqExtend(sock, extend)) ) {
		connection[c].ch_errtxt = strdup(dis_emsg[rc]);
		return (pbs_errno = PBSE_PROTOCOL);
	}

	/* write data */

	if (DIS_tcp_wflush(sock)) {
		return (pbs_errno = PBSE_PROTOCOL);
	}

	return 0;
}


static char ** PBSD_select_get(c)
	int c;
{
	int   i;
	struct batch_reply *reply;
	int   njobs;
	char *sp;
	int   stringtot;
	size_t totsize;
	struct brp_select *sr;
	char **retval = (char **)NULL;

	/* read reply from stream */

	reply = PBSD_rdrpy(c);
	if ( reply == NULL) {
                pbs_errno = PBSE_PROTOCOL;
	} else if ( reply->brp_choice != NULL &&
	            reply->brp_choice != BATCH_REPLY_CHOICE_Text &&
	            reply->brp_choice != BATCH_REPLY_CHOICE_Select) {
		pbs_errno = PBSE_PROTOCOL;
	} else if ( connection[c].ch_errno == 0 ) {
		/* process the reply -- first, build a linked 
		  list of the strings we extract from the reply, keeping 
		  track of the amount of space used...
		*/
		stringtot = 0;
		njobs = 0;
		sr = reply->brp_un.brp_select;
		while( sr != (struct brp_select *)NULL ) {
			stringtot += strlen(sr->brp_jobid) + 1;
			njobs++;
			sr = sr->brp_next;
		}
		/* ...then copy all the strings into one of "Bob's
		   structures", freeing all strings we just allocated...
		*/
		
		totsize = stringtot + (njobs+1) * (sizeof (char *));
		retval = (char **)malloc( totsize );
		if( retval == (char **)NULL ) {
			pbs_errno = PBSE_SYSTEM;
			return (char **)NULL;
		}
		sr = reply->brp_un.brp_select;
		sp = (char *)retval + (njobs + 1) * sizeof (char *);
		for( i=0; i<njobs; i++ ) {
			retval[i] = sp;
			strcpy(sp,sr->brp_jobid);
			sp += strlen(sp) + 1;
			sr = sr->brp_next;
		}
		retval[i] = (char *)NULL;
	}
	
	PBSD_FreeReply(reply);

	return retval;
}
