/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
/*
**	Routines to communicate with UDP packets - Reliable Packet Protocol.
**
**	This package provides for sending information in "messages"
**	which are complete blocks of data which will either arrive
**	complete or not at all.
*/
static	char	ident[] = "@(#) $RCSfile: rpp.c,v $ $Revision: 2.3 $";

#if !defined(_BSD) && defined(_AIX)   /* this is needed by AIX */
#define	_BSD	1
#endif

#include <pbs_config.h>   /* the master config generated by configure */

#include	<stdio.h>
#include	<stddef.h>
#include	<stdlib.h>
#include	<string.h>
#include	<unistd.h>
#include	<fcntl.h>
#include	<netdb.h>
#include	<errno.h>
#include	<assert.h>
#include	<math.h>
#include	<sys/types.h>
#include	<sys/file.h>
#include	<sys/stat.h>
#include	<sys/socket.h>
#include	<sys/time.h>
#include	<netinet/in.h>
#include	"rpp.h"

#if !defined(H_ERRNO_DECLARED)
extern int h_errno;
#endif

/*---------Boolean Constants-------------------------------------------*/
#ifndef	FALSE
#define	FALSE	0
#endif
#ifndef	TRUE
#define	TRUE	1
#endif


/*---------Size-Constants for the various parts of RPP packet------------*/
#define	RPP_PKT_SIZE	4*1024 /*max packet length, including header     */

#define	RPP_PKT_HEAD	26     /*Size of packet header, includes CRC     */

#define	RPP_HDR_SID	2      /*position of stream id in packet header  */

#define	RPP_HDR_SEQ	10     /*position of "sequence" in pkt header    */

#define	RPP_CRC_LEN	8      /*Length of CRC field in pkt header       */

#define RPP_PKT_CRC	(RPP_PKT_HEAD - RPP_CRC_LEN)

#define	RPP_PKT_DATA	(RPP_PKT_SIZE - RPP_PKT_HEAD)


/*---RPP packet-header fields; listed in order; field lengths in bytes---*/
/*
**   length             field
**  ========          =========
**     2		type
**     8		stream id
**     8		sequence
**     8		crc
*/


/*---------Integer codes for all the valid RPP message types--------------*/
#define	RPP_ACK		1
#define	RPP_DATA	2
#define	RPP_EOD		3
#define	RPP_HELLO1	4
#define	RPP_HELLO2	5
#define	RPP_GOODBYE	6


/*---------Integer codes for all the valid RPP state values---------------*/
#define RPP_DEAD	-1
#define	RPP_FREE 	 0
#define	RPP_OPEN_PEND	 1
#define	RPP_OPEN_WAIT	 2
#define	RPP_CONNECT	 3
#define	RPP_CLOSE_PEND	 4
#define	RPP_LAST_ACK	 5
#define	RPP_CLOSE_WAIT1	 6
#define	RPP_CLOSE_WAIT2	 7
#define	RPP_STALE	99


/*---------Constants governing the sending of RPP packets-----------------*/
#define	RPP_TIMEOUT	2        /*time in seconds; packet on the master  */
			         /*send queue is not sent more often than */
			         /*every RPP_TIMEOUT seconds              */


#define	RPP_RETRY	20       /*after this many sendto attempts on a   */
			         /*packet, give up and declare the stream */
                                 /*to be in the "stale" state             */


#define	RPP_HIGHWATER	24       /*Max allowed number of outstanding pkts */



/*---------Data Structure Definitions--------------------------------------*/

struct	send_packet
{
 /*Several kinds of linked lists "hang" from each  "RPP stream structure." */
 /*In particular, each stram structure has a list of "send_packet"         */
 /*structures.  These structures record information that's necessary for    */
 /*managing a piece of data that's being sent to the other end of the      */
 /*connection.  Besides having a pointer that links it to the next         */
 /*"send_packet on the stream, a "send_packet" posseses a pair of pointers */
 /*(up,down) which link the  send_packet to the "master send list" when    */
 /*connection sequencing has reached the point where it is now proper to   */
 /*attach it to the list (stream is fully opened).  Once on the master send*/
 /*list, any invocation of the "rpp_send_out" routine will attempt to      */
 /*transfer the "send_packet's" data buffer to the other end of the stream */
 /*"connection" if all of the following hold:                              */
 /*1) packet hasn't reached its maximum-transfer-attempt limit (RPP_RETRY) */
 /*2) RPP_TIMEOUT or more seconds have elapsed since the last attempt      */
 /*3) the transfer has yet to be ACK'd by the other side.                  */
 /*4) less than RPP_HIGHWATER number of non-duplicate packets are un-ACK'd */
 /**/
 /*Those "send_packets" that deal with stream control (RPP_ACK, RPP_HELLO1,*/
 /*RPP_HELLO2, RPP_GOODBYE) have no associated data, the "send_packet's"   */
 /*"data" buffer is comprised of only the header.                          */



	u_char	*data;               /*points to a buffer to be transferred*/

	u_short	type;		     /*RPP_ACK, RPP_HELLO1,RPP_DATA,etc    */

	u_short	sent_out;            /*number of times sent; <=RPP_RETRY   */

	int	len;                 /*size *data not counting the header  */

	int	index;               /*other end's stream id, i.e. which   */
                                     /*stream on the other is supposed to  */
                                     /*receive the data                    */

	int	sequence;            /*sequential value that is placed into*/
                                     /*data buffer's header and is returned*/
                                     /*in the RPP_ACK; identifies the data */
                                     /*that was transferred and now needs  */
                                     /*to be removed from master send list */

	time_t	time_sent;	     /*time packet was last sent; zero if  */
				     /*it has yet to be sent               */

	struct	send_packet	*next;   /*nxt packet on stream's send list*/

	struct	send_packet	*up;     /*used when the "send_packet" is  */
                                         /*on RPP's "master send *list";   */
                                         /*pointer to preceeding packet    */

	struct	send_packet	*down;   /*similar to "up", but pointing to*/
                                         /*packet following this one       */
};


struct	recv_packet
{
 /*Hanging from an "RPP stream structure" is a linked list of recv_packets*/
 /*Each "recv_packet" records the information necessary for managing a    */
 /*piece of data that was received from the other end of a connection.    */
 /*With the exception of RPP_GOODBYE (a surrogate RPP_EOD), pkts that deal*/
 /*with stream control (RPP_ACK,RPP_HELLO1,RPP_HELLO2,) don't ever make it*/
 /*to the stream's receive list, they are handled directly as they come in*/
 /*So, the only types of packets on a stream's receive list should be ones*/
 /*of type RPP_DATA, RPP_EOD, and RPP_GOODBYE                             */

	u_char	*data;         /*pointer to byte string that was sent from*/
                               /*the other end of the stream              */

	u_short	type;          /*RPP_DATA, RPP_EOD or RPP_GOODBYE; i.e. a */
                               /*piece of the message or the end of the   */
                               /*message (either RPP_EOD or RPP_GOODBYE   */

	int	len;           /*size in bytes of the received string;    */
                               /*does not include the length of the header*/

	int	sequence;      /*decoded sequential number; it describes  */
                               /*the buffer's sequential output order     */
                               /*relative to the other buffers being sent */
                               /*this end of the connection. This sequence*/
                               /*is put in the buffer by the sender and   */
                               /*pulled out and stored in this element    */

	struct	recv_packet  *next;    /*pointer to next recv_packet in   */
                                       /*the linked  "receive" list for   */
                                       /*this stream                      */
};


struct	pending
{
 /*Each "RPP stream structure" has a linked list of "pend structs" hanging*/
 /*from it.  This list of structs is used to manage a set of data buffers */
 /*that may or may not all get 'packetized' and sent over to the other end*/
 /*of the connection.  Subsequent to the creation of the list of data     */
 /*buffers, the creating program decides to either "commit or not commit" */
 /*the sending of this data to the other side-- done by calling the inter-*/
 /*face function, rpp_wcommit( stream_id, (Yes|No) ).  The interface func-*/
 /*tion rpp_write(stream_id,bfr,len) doesn't send the data, it merely     */
 /*attaches the data buffer to a "pending" struct and links this struct to*/
 /*the stream's list of pending structs-- no transfer to the "other end"  */
 /*gets set in motion by calling rpp_write(...)                           */
 /**/
 /*Doing an rpp_wcommit(stream id, Yes), on the other hand, entails adjoin*/
 /*ing a header to each pending struct's data buffer, attaching the buffer*/
 /*to a new send_packet struct, linking the send_packet to the master send*/
 /*list, freeing the pend struct, update of the stream's "pend_commit"    */
 /*variable--  a running sum of the number of bytes sent to the other side*/
 /**/
 /*Dcommiting data that was written for transfer to the other end entails */
 /*removing and freeing the "pending structs" and their associated data   */
 /*buffer from the stream's pend list and, updating the stream's          */
 /*"pend_attempt" varable back to the byte count that is stored in its    */
 /*"pend_commit" variable.  Refer to relevant fields in "struct stream"   */
 /**/

	u_char	*data;           /*pointer to a buffer of data without any*/
                                 /*header information adjoined            */

	struct	pending	*next;   /*pointer to the next "pending" struct on*/
                                 /*the list                               */
};


struct	stream
{
 /*Every RPP stream that comes into existence during the life of the      */
 /*process gets realized on each end of the "connection" by a stream      */
 /*struct.  The struct governs communications with the "other end" of the */
 /*stream.  All the stream structs generated by a process are in a        */
 /*dynamic array in the process' heap area.                               */
 /*Each RPP stream is mechanized as a finite state machine.               */

	int			state;        /*state of this end of the  */
                                              /*connection; RPP_OPEN, etc */

	struct	sockaddr_in	addr;         /*address of the other end; */
                                              /*port/family/IPadrs        */

	struct	in_addr		*addr_array;  /*array of alternate network*/
                                              /*addresses for other end   */
                                              /*of the connection         */

	int			fd;           /*socket descriptor         */

	int			stream_id;    /*id of other end of the    */
                                              /*connection; array position*/
                                              /*of stream struct on the   */
                                              /*other end                 */

	int			open_key;     /*unique bit pattern created*/
                                              /*by the end issuing the    */
                                              /*rpp_open().  It's the same*/
                                              /*same for each end of the  */
                                              /*connecton; used in setting*/
                                              /*up the stream connection  */

	int			msg_cnt;      /*size in bytes of current  */
                                              /*DATA/EOD/GOODBYE message; */
                                              /*integral number of packet */
                                              /*lengths                   */

	int			send_sequence;/*initialized to value of 1 */
                                              /*and incremented by 1 for  */
                                              /*each packet that's added  */
                                              /*to the "master send list" */

	struct	pending		*pend_head;   /*head and tail pointers for*/
	struct	pending		*pend_tail;   /*stream's pend list; see   */
                                              /*struct pend definition    */

	int			pend_commit;  /*total number of data bytes*/
                                              /*sent to other end connect.*/
	int			pend_attempt; /*total number bytes that   */
                                              /*reside in the list of     */
                                              /*pending struct buffers.   */
                                              /*relationship:             */
                                              /*pend_commit<=pend_attempt */

	struct	send_packet	*send_head;   /*head and tail pointers for*/
	struct	send_packet	*send_tail;   /*stream's "master send list*/
                                              /*see struct send_packet    */

	int			recv_sequence;/*monotonic,increasing, by 1*/
                                              /*starts from zero; A Packet*/
                                              /*on the stream's recv list */
                                              /*having a sequence number  */
                                              /*less than this value is a */
                                              /*packet of an earlier mesg */

	struct	recv_packet	*recv_head;   /*head and tail pointers for*/
	struct	recv_packet	*recv_tail;   /*the stream's "recv list"; */
                                              /*see struct recv_packet    */

	int			recv_commit;  /*number bytes,from start of*/
                                              /*current message,that have */
                                              /*gotten accepted by the    */
                                              /*reader on this end        */

	int			recv_attempt; /*number bytes, from start  */
                                              /*of current message, that  */
                                              /*gotten read by the reader */
};


/*---------------------Static Variables-----------------------------------*/
static	struct	stream	*stream_array = NULL;  /*pointer to 1-st stream  */
                                               /*struct in  dynamic array*/

static	int		stream_num = 0;        /*current nmbr of stream   */
                                               /*structs in dynamic array */

static	int		pkts_sent = 0;         /*range: 0 - RPP_HIGHWATER;*/
                                               /*incremented with each new*/
                                               /*pkt sent; retransmissions*/
                                               /*are not counted          */
                                             
static	int		open_key = 0;          /*dynamic, monotonic value */
                                               /*stored in stream struct; */
                                               /*created on rpp_open and  */
                                               /*passed to other end to be*/
                                               /*recorded in corresponding*/
                                               /*stream struct on that end*/

static	struct	send_packet	*top = NULL;   /*ptrs to beginning and end*/
static	struct	send_packet	*bottom = NULL;/*of "master send list";   */
                                               /*No send_packet's data    */
                                               /*is sent to the other end */
                                               /*unless that send_packet  */
                                               /*is linked on to this list*/

/*---------------------Global Variables-----------------------------------*/
int		rpp_dbprt = 0;                 /*controls debug printing  */

int		rpp_fd = -1;                   /*dynamic; records return  */
                                               /*value of last successful */
                                               /*socket call; -1,no socket*/
                                               /*has been created for the */
                                               /*process thus far         */ 

int		*rpp_fd_array = NULL;          /*dynamic array of socket  */
                                               /*descriptors having a bound*/
                                               /*network address;  array  */
                                               /*is usually 1 element long*/

int		rpp_fd_num = 0;                /*nmbr of elements in above*/
                                               /*dynamic array, usually 1.*/


/*--------------------Global Constant Data--------------------------------*/
/*--Convert 2 and 8 digit hexidecimal strings to integer and conversely---*/
/*--using the Macros: I2TOH(i,h) HTOI2(h,i) I8TOH(i,h) HTOI8(h,i)---------*/
/*--Refer to the Macros section that follows this section-----------------*/

char	cval[] = {
	-1, -1, -1, -1, -1, -1, -1, -1,		/* nul .. bel */
	-1, -1, -1, -1, -1, -1, -1, -1,		/* bs .. si */
	-1, -1, -1, -1, -1, -1, -1, -1,		/* dle .. etb */
	-1, -1, -1, -1, -1, -1, -1, -1,		/* can .. us */
	-1, -1, -1, -1, -1, -1, -1, -1,		/* sp .. ' */
	-1, -1, -1, -1, -1, -1, -1, -1,		/* ( .. / */
	 0,  1,  2,  3,  4,  5,  6,  7,		/* 0 .. 7 */
	 8,  9, -1, -1, -1, -1, -1, -1,		/* 8 .. ? */
	-1, 10, 11, 12, 13, 14, 15, -1,		/* @ .. G */
	-1, -1, -1, -1, -1, -1, -1, -1,		/* H .. O */
	-1, -1, -1, -1, -1, -1, -1, -1,		/* P .. W */
	-1, -1, -1, -1, -1, -1, -1, -1,		/* X .. _ */
	-1, 10, 11, 12, 13, 14, 15, -1,		/* ` .. g */
	-1, -1, -1, -1, -1, -1, -1, -1,		/* h .. o */
	-1, -1, -1, -1, -1, -1, -1, -1,		/* p .. w */
	-1, -1, -1, -1, -1, -1, -1, -1		/* x .. del */
};

char	ival[] = {
	'0', '1', '2', '3', '4', '5', '6', '7',
	'8', '9', 'A', 'B', 'C', 'D', 'E', 'F'
};



/*-------------Conversion  and other Macros-------------------------------*/

#define	I2TOH(i, h) \
{ \
	int	num = i; \
	char	*str = h; \
	str[1] = ival[num & 0xF]; num >>= 4 ; \
	str[0] = ival[num & 0xF]; \
}

#define	I8TOH(i, h) \
{ \
	u_long	num = i; \
	char	*str = h; \
	str[7] = ival[num & 0xF]; num >>= 4; \
	str[6] = ival[num & 0xF]; num >>= 4; \
	str[5] = ival[num & 0xF]; num >>= 4; \
	str[4] = ival[num & 0xF]; num >>= 4; \
	str[3] = ival[num & 0xF]; num >>= 4; \
	str[2] = ival[num & 0xF]; num >>= 4; \
	str[1] = ival[num & 0xF]; num >>= 4; \
	str[0] = ival[num & 0xF]; \
}

#define	HTOI2(h, i) \
{ \
	char	*str = h; \
	int	num = 0; \
	num = cval[str[0] & 0xFF]; num <<= 4; \
	num |= cval[str[1] & 0xFF]; \
	i = num; \
}

#define	HTOI8(h, i) \
{ \
	char	*str = h; \
	u_long	num; \
	num  = (long)cval[str[0] & 0x7F]; num <<= 4; \
	num |= (long)cval[str[1] & 0x7F]; num <<= 4; \
	num |= (long)cval[str[2] & 0x7F]; num <<= 4; \
	num |= (long)cval[str[3] & 0x7F]; num <<= 4; \
	num |= (long)cval[str[4] & 0x7F]; num <<= 4; \
	num |= (long)cval[str[5] & 0x7F]; num <<= 4; \
	num |= (long)cval[str[6] & 0x7F]; num <<= 4; \
	num |= (long)cval[str[7] & 0x7F]; \
	i = num; \
}


/*
** A print macro for use in debugging.
*/

#ifdef  DEBUG
#define	DBPRT(x) \
	if (rpp_dbprt) { \
		int	err = errno; \
		printf("%X: ", time(0)); \
       		printf x; \
		errno = err; \
	}
#else
#define	DBPRT(x)
#endif


#ifndef	MIN
#define	MIN(x, y)	(((x) < (y)) ? (x) : (y))
#endif
#ifndef	MAX
#define	MAX(x, y)	(((x) > (y)) ? (x) : (y))
#endif




#ifdef lesiak
#ifndef	FALSE
#define	FALSE	0
#endif
#ifndef	TRUE
#define	TRUE	1
#endif

int	rpp_dbprt = 0;

/*
** Set up a debug print macro.
*/
#ifdef  DEBUG
#define	DBPRT(x) \
	if (rpp_dbprt) { \
		int	err = errno; \
		printf("%X: ", time(0)); \
       		printf x; \
		errno = err; \
	}
#else
#define	DBPRT(x)
#endif
#endif


#ifndef	NO_CRC
/*
**	BEGIN included source
*/
/*-
 * Copyright (c) 1991, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * This code is derived from software contributed to Berkeley by
 * James W. Williams of NASA Goddard Space Flight Center.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgment:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static char sccsid[] = "@(#)crc.c	8.1 (Berkeley) 6/17/93";
#endif /* not lint */

static u_long crctab[] = {
	0x0,
	0x04c11db7, 0x09823b6e, 0x0d4326d9, 0x130476dc, 0x17c56b6b,
	0x1a864db2, 0x1e475005, 0x2608edb8, 0x22c9f00f, 0x2f8ad6d6,
	0x2b4bcb61, 0x350c9b64, 0x31cd86d3, 0x3c8ea00a, 0x384fbdbd,
	0x4c11db70, 0x48d0c6c7, 0x4593e01e, 0x4152fda9, 0x5f15adac,
	0x5bd4b01b, 0x569796c2, 0x52568b75, 0x6a1936c8, 0x6ed82b7f,
	0x639b0da6, 0x675a1011, 0x791d4014, 0x7ddc5da3, 0x709f7b7a,
	0x745e66cd, 0x9823b6e0, 0x9ce2ab57, 0x91a18d8e, 0x95609039,
	0x8b27c03c, 0x8fe6dd8b, 0x82a5fb52, 0x8664e6e5, 0xbe2b5b58,
	0xbaea46ef, 0xb7a96036, 0xb3687d81, 0xad2f2d84, 0xa9ee3033,
	0xa4ad16ea, 0xa06c0b5d, 0xd4326d90, 0xd0f37027, 0xddb056fe,
	0xd9714b49, 0xc7361b4c, 0xc3f706fb, 0xceb42022, 0xca753d95,
	0xf23a8028, 0xf6fb9d9f, 0xfbb8bb46, 0xff79a6f1, 0xe13ef6f4,
	0xe5ffeb43, 0xe8bccd9a, 0xec7dd02d, 0x34867077, 0x30476dc0,
	0x3d044b19, 0x39c556ae, 0x278206ab, 0x23431b1c, 0x2e003dc5,
	0x2ac12072, 0x128e9dcf, 0x164f8078, 0x1b0ca6a1, 0x1fcdbb16,
	0x018aeb13, 0x054bf6a4, 0x0808d07d, 0x0cc9cdca, 0x7897ab07,
	0x7c56b6b0, 0x71159069, 0x75d48dde, 0x6b93dddb, 0x6f52c06c,
	0x6211e6b5, 0x66d0fb02, 0x5e9f46bf, 0x5a5e5b08, 0x571d7dd1,
	0x53dc6066, 0x4d9b3063, 0x495a2dd4, 0x44190b0d, 0x40d816ba,
	0xaca5c697, 0xa864db20, 0xa527fdf9, 0xa1e6e04e, 0xbfa1b04b,
	0xbb60adfc, 0xb6238b25, 0xb2e29692, 0x8aad2b2f, 0x8e6c3698,
	0x832f1041, 0x87ee0df6, 0x99a95df3, 0x9d684044, 0x902b669d,
	0x94ea7b2a, 0xe0b41de7, 0xe4750050, 0xe9362689, 0xedf73b3e,
	0xf3b06b3b, 0xf771768c, 0xfa325055, 0xfef34de2, 0xc6bcf05f,
	0xc27dede8, 0xcf3ecb31, 0xcbffd686, 0xd5b88683, 0xd1799b34,
	0xdc3abded, 0xd8fba05a, 0x690ce0ee, 0x6dcdfd59, 0x608edb80,
	0x644fc637, 0x7a089632, 0x7ec98b85, 0x738aad5c, 0x774bb0eb,
	0x4f040d56, 0x4bc510e1, 0x46863638, 0x42472b8f, 0x5c007b8a,
	0x58c1663d, 0x558240e4, 0x51435d53, 0x251d3b9e, 0x21dc2629,
	0x2c9f00f0, 0x285e1d47, 0x36194d42, 0x32d850f5, 0x3f9b762c,
	0x3b5a6b9b, 0x0315d626, 0x07d4cb91, 0x0a97ed48, 0x0e56f0ff,
	0x1011a0fa, 0x14d0bd4d, 0x19939b94, 0x1d528623, 0xf12f560e,
	0xf5ee4bb9, 0xf8ad6d60, 0xfc6c70d7, 0xe22b20d2, 0xe6ea3d65,
	0xeba91bbc, 0xef68060b, 0xd727bbb6, 0xd3e6a601, 0xdea580d8,
	0xda649d6f, 0xc423cd6a, 0xc0e2d0dd, 0xcda1f604, 0xc960ebb3,
	0xbd3e8d7e, 0xb9ff90c9, 0xb4bcb610, 0xb07daba7, 0xae3afba2,
	0xaafbe615, 0xa7b8c0cc, 0xa379dd7b, 0x9b3660c6, 0x9ff77d71,
	0x92b45ba8, 0x9675461f, 0x8832161a, 0x8cf30bad, 0x81b02d74,
	0x857130c3, 0x5d8a9099, 0x594b8d2e, 0x5408abf7, 0x50c9b640,
	0x4e8ee645, 0x4a4ffbf2, 0x470cdd2b, 0x43cdc09c, 0x7b827d21,
	0x7f436096, 0x7200464f, 0x76c15bf8, 0x68860bfd, 0x6c47164a,
	0x61043093, 0x65c52d24, 0x119b4be9, 0x155a565e, 0x18197087,
	0x1cd86d30, 0x029f3d35, 0x065e2082, 0x0b1d065b, 0x0fdc1bec,
	0x3793a651, 0x3352bbe6, 0x3e119d3f, 0x3ad08088, 0x2497d08d,
	0x2056cd3a, 0x2d15ebe3, 0x29d4f654, 0xc5a92679, 0xc1683bce,
	0xcc2b1d17, 0xc8ea00a0, 0xd6ad50a5, 0xd26c4d12, 0xdf2f6bcb,
	0xdbee767c, 0xe3a1cbc1, 0xe760d676, 0xea23f0af, 0xeee2ed18,
	0xf0a5bd1d, 0xf464a0aa, 0xf9278673, 0xfde69bc4, 0x89b8fd09,
	0x8d79e0be, 0x803ac667, 0x84fbdbd0, 0x9abc8bd5, 0x9e7d9662,
	0x933eb0bb, 0x97ffad0c, 0xafb010b1, 0xab710d06, 0xa6322bdf,
	0xa2f33668, 0xbcb4666d, 0xb8757bda, 0xb5365d03, 0xb1f740b4
};

/*
 * Compute a POSIX 1003.2 checksum.  This routine has been broken out so that
 * other programs can use it.  It takes a char pointer and length.
 * It ruturns the crc value for the data in buf.
 */

u_long
crc(buf, clen)
	u_char	*buf;
	u_long	clen;
{
	register u_char *p;
	register u_long crc, len;

#define	COMPUTE(var, ch) (var) = (((var) << 8) ^ \
		crctab[(((var) >> 24) & 0xff) ^ (ch)]) & 0xffffffff

	for (crc=0, len=clen, p = buf; len--; ++p) {
		COMPUTE(crc, *p);
	}

	/* Include the length of the file. */
	for (; clen != 0; clen >>= 8) {
		COMPUTE(crc, clen & 0xff);
	}

	return (~crc & 0xffffffff);
}
/*
**	END of included source
*/
#else
#define	crc(x, y)	0
#endif

/*
**	Generate a sequence number for a packet.
*/
static
int
next_seq(seq)
    int		*seq;
{
	(*seq)++;
	if (*seq < 0) {		/* had a rollover */
		errno = EFBIG;
		return -1;
	}

	return 0;
}

char *
netaddr(ap)
    struct sockaddr_in *ap;
{
	static	char	out[80];
	u_long		ipadd;

	ipadd = ntohl(ap->sin_addr.s_addr);

	sprintf(out, "%ld.%ld.%ld.%ld:%d",
		(ipadd & 0xff000000) >> 24,
		(ipadd & 0x00ff0000) >> 16,
		(ipadd & 0x0000ff00) >> 8,
		(ipadd & 0x000000ff),
		ntohs(ap->sin_port));
	return out;
}

/*
**	Create a packet of the given type, fill in the sequence and
**	index number.  If buf is NULL, malloc an area for just
**	a header.  If buf is not NULL, it should contain space
**	for len+RPP_PKT_HEADER bytes.
*/
static
void
rpp_form_pkt(index, type, seq, buf, len)
    int		index;
    int		type;
    int		seq;
    u_char	*buf;
    int		len;
{
	static	char		id[] = "form_pkt";
	struct	send_packet	*pktp;
	struct	stream		*sp;

	DBPRT(("%s: index %d type %d seq %u len %d\n",
		id, index, type, seq, len))
	sp = &stream_array[index];
	pktp = (struct send_packet *)malloc(sizeof(struct send_packet));
	assert(pktp != NULL);

	pktp->type = type;
	pktp->sequence = seq;
	pktp->time_sent = 0;
	pktp->sent_out = 0;
	pktp->len = len;
	pktp->index = index;

	if (buf)
		pktp->data = (u_char *)realloc(buf, len+RPP_PKT_HEAD);
	else
		pktp->data = (u_char *)malloc(RPP_PKT_HEAD);
	assert(pktp->data != NULL);
	/*
	** Put on stream send list
	*/
	if (sp->send_head == NULL)
		sp->send_head = pktp;
	else
		sp->send_tail->next = pktp;
	sp->send_tail = pktp;
	pktp->next = NULL;
	pktp->down = NULL;

	/*
	** if open has not completed yet, hold off putting on send queue
	*/
	if (sp->stream_id == -1) {
		pktp->up = NULL;
		return;
	}
	/*
	** if the stream is fully open, format and put on the send queue
	*/
	DBPRT(("%s: idx %d link %d seq %d len %d to sendq\n", id,
		index, type, seq, len))

	I2TOH(type, (char *)&pktp->data[len])
	I8TOH(sp->stream_id, (char *)&pktp->data[len+RPP_HDR_SID])
	I8TOH(seq, (char *)&pktp->data[len+RPP_HDR_SEQ])
	I8TOH(crc(pktp->data, (u_long)(len+RPP_PKT_CRC)),
		(char *)&pktp->data[len+RPP_PKT_CRC])

	if (bottom)
		bottom->down = pktp;
	pktp->up = bottom;
	if (top == NULL)		/* first one */
		top = pktp;
	bottom = pktp;
	return;
}

/*
**	Check to make sure an incoming packet goes with one of the
**	streams we have.
*/
static
struct	stream	*
rpp_check_pkt(index, addrp)
    int		index;
    struct	sockaddr_in	*addrp;
{
	static	char	id[] = "check_pkt";
	struct	stream	*sp;
	struct	in_addr	*addrs;
	int		i;

	if (index < 0 || index >= stream_num) {
		DBPRT(("%s: BAD INDEX %d outside limit %d\n",
			id, index, stream_num))
		return NULL;
	}

	sp = &stream_array[index];
	if (sp->state <= RPP_FREE) {
		DBPRT(("%s: FREE STREAM\n", id))
		return NULL;
	}

	if (addrp->sin_port != sp->addr.sin_port)
		goto bad;
	if (addrp->sin_family != sp->addr.sin_family)
		goto bad;
	if (addrp->sin_addr.s_addr == sp->addr.sin_addr.s_addr)
		return sp;

	if ((addrs = sp->addr_array) != NULL) {
		for (i=0; addrs[i].s_addr; i++) {
			if (addrs[i].s_addr == addrp->sin_addr.s_addr)
				return sp;
		}
	}

  bad:
	DBPRT(("%s: ADDRESS MISMATCH\n", id))
	DBPRT(("\tstream %d addr %s\n", index, netaddr(&sp->addr)))
	DBPRT(("\tpkt addr %s\n", netaddr(addrp)))
	return NULL;
}

/*
**	Send outstanding information starting with top and working
**	down to bottom.  Will not cause state change.
*/
static
void
rpp_send_out()
{
	static	char		id[] = "send_out";
	struct	send_packet	*pp;
	struct	stream		*sp;
	time_t			curr;

	DBPRT(("%s: entered out %d\n", id, pkts_sent))
	curr = time(NULL);
	for (pp = top; pp; pp = pp->down) {
		int	sitting = curr - pp->time_sent;

		if (sitting < RPP_TIMEOUT)
			continue;
		if (pp->time_sent == 0 &&
				pkts_sent >= RPP_HIGHWATER)
			break;

		sp = &stream_array[pp->index];
		DBPRT(("%s index %d type %d sent %d seq %d to %s crc %8.8s\n",
				id, pp->index, pp->type,
				pp->sent_out, pp->sequence, netaddr(&sp->addr),
				(char *)&pp->data[pp->len+RPP_PKT_CRC]))
		if (sendto(sp->fd, (char *)pp->data, RPP_PKT_HEAD+pp->len,
				0, (struct sockaddr *)&sp->addr,
				sizeof(struct sockaddr_in)) == -1) {
			DBPRT(("%s: SENDTO errno %d\n", id, errno))
			pp->sent_out++;
			continue;
		}

		if (pp->time_sent == 0)		/* new one */
			pkts_sent++;

		pp->time_sent = curr;
		pp->sent_out++;
	}
	return;
}

/*
**	Create or reuse a possition in stream_array.
*/
static
int
rpp_create_sp()
{
	int	i;
	struct	stream	*sp;

	if (stream_array == NULL) {
		stream_array = (struct stream *)malloc(sizeof(struct stream));
		if (stream_array == NULL)
			return -1;
		memset(stream_array, '\0', sizeof(struct stream));
		stream_num = 1;
	}
	for (i=0; i<stream_num; i++) {
		sp = &stream_array[i];

		if (sp->state == RPP_FREE)
			break;
	}
	if (i == stream_num) {
		for (i=0; i<stream_num; i++) {
			sp = &stream_array[i];

			if (sp->state == RPP_DEAD)
				break;
		}
	}
	if (i == stream_num) {		/* none available */
		sp = (struct stream *)realloc((void *)stream_array,
					(stream_num*2)*sizeof(struct stream));
		if (sp == NULL) {
			sp = (struct stream *)realloc((void *)stream_array,
					(stream_num+1)*sizeof(struct stream));
			if (sp == NULL)
				return -1;
			stream_num++;
		}
		else
			stream_num *= 2;
		stream_array = sp;
		memset((void *)&stream_array[i], '\0',
			(stream_num-i)*sizeof(struct stream));
	}
	else
		memset((void *)&stream_array[i], '\0', sizeof(struct stream));
	DBPRT(("rpp_create_sp: new index %d\n", i))
	return i;
}

/*
**	Look up the "canonical" name for the host by
**	calling gethostbyaddr with an IP address.
*/
static
struct	hostent		*
rpp_get_cname(addr)
    struct	sockaddr_in	*addr;
{
	static	char	id[] = "get_cname";
	struct	hostent		*hp;
	char			*hname;

	if ((hp = gethostbyaddr((void *)&addr->sin_addr,
				sizeof(struct in_addr),
				addr->sin_family)) == NULL) {
		DBPRT(("%s: addr not found, h_errno=%d errno=%d\n",
				id, h_errno, errno))
		return NULL;
	}
	if ((hname = (char *)strdup(hp->h_name)) == NULL)
		return NULL;

	if ((hp = gethostbyname(hname)) == NULL) {
		DBPRT(("%s: canonical name %s not found, h_errno=%d errno=%d\n",
				id, hname, h_errno, errno))
	}
	free(hname);
	return hp;
}

/*
**	Allocate a list of alternate address for a host and save
**	them in the stream structure.
*/
static
void
rpp_alist(hp, sp)
    struct	hostent		*hp;
    struct	stream		*sp;
{
	int	i, j;

	for (i=1; hp->h_addr_list[i]; i++);
	if (i == 1)
		return;

	sp->addr_array = (struct in_addr *)calloc(i, sizeof(struct in_addr));
	for (j=i=0; hp->h_addr_list[i]; i++) {
		if (memcmp(&sp->addr.sin_addr,
				hp->h_addr_list[i], hp->h_length) == 0)
			continue;
		memcpy(&sp->addr_array[j++], hp->h_addr_list[i], hp->h_length);
	}
	sp->addr_array[j].s_addr = 0;
	return;
}

static
int
rpp_send_ack(sp, seq)
    struct	stream	*sp;
    int		seq;
{
	static	char	id[] = "send_ack";
	char	buf[RPP_PKT_HEAD];
	u_long	xcrc;

	if (sp->stream_id < 0) {		/* can't send yet */
		DBPRT(("%s: STREAM NOT OPEN seq %d\n", id, seq))
		return 0;
	}

	I2TOH(RPP_ACK, buf)
	I8TOH(sp->stream_id, &buf[2])
	I8TOH(seq, &buf[10])
	xcrc = crc((u_char *)buf, (u_long)RPP_PKT_CRC);
	I8TOH(xcrc, &buf[RPP_PKT_CRC])

	DBPRT(("%s: seq %d to %s crc %lX\n",
			id, seq, netaddr(&sp->addr), xcrc))
	if (sendto(sp->fd, buf, RPP_PKT_HEAD, 0, (struct sockaddr *)&sp->addr,
			sizeof(struct sockaddr_in)) == -1) {
		DBPRT(("%s: ACK error %d\n", id, errno))
		if (errno != EWOULDBLOCK && errno != ENOBUFS)
			return -1;
	}
	return 0;
}

/*
**	Take a packet off the send queue and free it.
*/
static
void
dqueue(pp)
    struct	send_packet	*pp;
{
	if (pp->down == NULL)
		bottom = pp->up;
	else
		pp->down->up = pp->up;
	if (pp->up == NULL)
		top = pp->down;
	else
		pp->up->down = pp->down;

	if (--pkts_sent < 0)
		pkts_sent = 0;
	free(pp->data);
	free(pp);
	return;
}

/*
**	Get rid of anything on the pend and send queue for a stream.
*/
static
void
clear_send(sp)
    struct	stream		*sp;
{
	struct	pending		*ppp, *pprev;
	struct	send_packet	*spp, *sprev;

	for (ppp=sp->pend_head; ppp; ppp=pprev) {
		pprev=ppp->next;
		free(ppp->data);
		free(ppp);
	}
	sp->pend_head = NULL;
	sp->pend_tail = NULL;
	sp->pend_commit = 0;
	sp->pend_attempt = 0;

	for (spp=sp->send_head; spp; spp=sprev) {
		sprev=spp->next;

		if (sp->stream_id == -1) { 	    	/* not open yet */
			struct	send_packet	*look;	/* might not be */
							/* on send queue */
			for (look = top; look; look = look->down) {
				if (look == spp)
					break;
			}
			if (look == NULL) {
				free(spp->data);
				free(spp);
				continue;
			}
		}
		dqueue(spp);
	}
	sp->send_head = NULL;
	sp->send_tail = NULL;
}

/*
**	Remove packets from receive, pending and send queues for
**	a stream, free all the memory and zero the stream_array
**	entry.
*/
static
void
clear_stream(sp)
    struct	stream		*sp;
{
	struct	recv_packet	*rpp, *rprev;

	DBPRT(("CLEAR stream %ld\n",
			((long)sp - (long)stream_array)/sizeof(struct stream)))
	for (rpp=sp->recv_head; rpp; rpp=rprev) {
		rprev=rpp->next;
		if (rpp->data)
			free(rpp->data);
		free(rpp);
	}
	sp->recv_head = NULL;
	sp->recv_tail = NULL;

	clear_send(sp);
	if (sp->addr_array) {
		free(sp->addr_array);
		sp->addr_array = NULL;
	}
	sp->state = RPP_DEAD;
}

/*
**	Do a recvfrom call to get a packet off of all file descriptors.
**	Return the index of the stream the packet belonged to
**	or -2 if it was not data, or -1 if there was an error.
**	Return -3 if there was no data to read.
**	MAY CAUSE STATE CHANGE!
*/
static
int
rpp_recv_pkt(fd)
	int	fd;
{
	static	char		id[] = "recv_pkt";
	int			len, flen;
	struct	sockaddr_in	addr;
	struct	hostent		*hp;
	int			i, streamid;
	struct	send_packet	*spp, *sprev;
	struct	recv_packet	*rpp, *rprev;
	struct	recv_packet	*pkt;
	struct	stream		*sp;
	char			*data;
	int		type;
	int		sequence;
	u_long		pktcrc;

	data = malloc(RPP_PKT_SIZE);
	assert(data != NULL);

	flen = sizeof(struct sockaddr_in);
	len = recvfrom(fd, data, RPP_PKT_SIZE, 0,
			(struct sockaddr *)&addr, &flen);
	if (len == -1) {
		free(data);
		if (errno == EWOULDBLOCK ||
		    errno == EAGAIN      ||
		    errno == ECONNREFUSED) {
			errno = 0;
			return -3;
		}
		return -1;
	}

	DBPRT(("%s: addr %s len %d\n", id, netaddr(&addr), len))

	if (len < RPP_PKT_HEAD)		/* less than minimum size */
		goto err_out;

	HTOI8(&data[len-RPP_CRC_LEN], pktcrc)
	if (pktcrc != crc((u_char *)data, (u_long)(len-RPP_CRC_LEN))) {
		DBPRT(("%s: packet crc %08lX failed\n", id, pktcrc))
		goto err_out;
	}
	HTOI2(&data[len-RPP_PKT_HEAD], type)
	HTOI8(&data[len-RPP_PKT_HEAD+RPP_HDR_SID], streamid)
	HTOI8(&data[len-RPP_PKT_HEAD+RPP_HDR_SEQ], sequence)

	switch (type) {

	case RPP_ACK:
		DBPRT(("%s: ACK stream %d sequence %d crc %08lX\n",
				id, streamid, sequence, pktcrc))
		free(data);
		if ((sp = rpp_check_pkt(streamid, &addr)) == NULL)
			return -2;

		if (sp->state == RPP_OPEN_PEND) {
			if (sequence != sp->open_key) {
				DBPRT(("%s: WILD ACK in RPP_OPEN_PEND %d\n",
					id, streamid))
				return -2;
			}
			spp = sp->send_head;
			assert(spp->type == RPP_HELLO2);
			assert(spp->next == NULL);

			sp->state = RPP_CONNECT;
			sp->send_head = NULL;
			sp->send_tail = NULL;
			dqueue(spp);
			return streamid;
		}
		else if (sp->stream_id == -1) {
			DBPRT(("%s: ACK for closed stream %d\n",
				id, streamid))
			return -2;
		}

		for (spp=sp->send_head, sprev=NULL; spp;
				sprev=spp, spp=spp->next) {
			if (spp->sequence == sequence)
				break;
		}
		if (spp) {
			int	took = time(NULL) - spp->time_sent;

			DBPRT(("%s: stream %d seq %d took %d\n",
				id, streamid, sequence, took))

			if (sp->state == RPP_CLOSE_WAIT1 &&
					spp->type == RPP_GOODBYE)
				sp->state = RPP_CLOSE_WAIT2;

			if (sprev == NULL)
				sp->send_head = spp->next;
			else
				sprev->next = spp->next;
			if (sp->send_tail == spp)
				sp->send_tail = sprev;
			dqueue(spp);

			if (sp->state == RPP_LAST_ACK &&
					sp->send_head == NULL) {
				clear_stream(sp);
				return -2;
			}
		}
		return streamid;

	case RPP_GOODBYE:
		DBPRT(("%s: GOODBYE stream %d sequence %d crc %08lX\n",
				id, streamid, sequence, pktcrc))
		free(data);
		if ((sp = rpp_check_pkt(streamid, &addr)) == NULL)
			return -2;
		if (rpp_send_ack(sp, sequence) == -1)
			return -1;

		switch (sp->state) {

		case RPP_OPEN_PEND:
		case RPP_OPEN_WAIT:
		case RPP_CLOSE_PEND:
		case RPP_LAST_ACK:
			return -2;

		case RPP_CLOSE_WAIT1:
			sp->state = RPP_LAST_ACK;
			return -2;

		case RPP_CLOSE_WAIT2:
			clear_stream(sp);
			return -2;
		
		default:
			break;
		}

		sp->state = RPP_CLOSE_PEND;
		clear_send(sp);		/* other side not reading now */

		for (rpp=sp->recv_head, rprev=NULL; rpp;
				rprev=rpp, rpp=rpp->next) {
			if (rpp->sequence >= sequence)
				break;
		}
		if (rpp == NULL || rpp->sequence > sequence) {
			DBPRT(("%s: GOOD seq %d\n", id, sequence))
			pkt = (struct recv_packet *)
				malloc(sizeof(struct recv_packet));
			assert(pkt != NULL);
			pkt->type = type;
			pkt->sequence = sequence;
			pkt->len = 0;
			pkt->data = NULL;
			if (rprev == NULL) {
				pkt->next = sp->recv_head;
				sp->recv_head = pkt;
			}
			else {
				pkt->next = rprev->next;
				rprev->next = pkt;
			}
			if (sp->recv_tail == rprev)
				sp->recv_tail = pkt;
		}
		else {
			DBPRT(("%s: DUPLICATE seq %d MAX seen %d\n",
				id, sequence, rpp->sequence))
		}
		return -2;

	case RPP_DATA:
	case RPP_EOD:
		DBPRT(("%s: DATA stream %d sequence %d crc %08lX len %d\n",
				id, streamid, sequence, pktcrc, len))
		if ((sp = rpp_check_pkt(streamid, &addr)) == NULL)
			goto err_out;
		if (rpp_send_ack(sp, sequence) == -1) {
			free(data);
			return -1;
		}

		switch (sp->state) {
		case RPP_OPEN_WAIT:
			DBPRT(("INPUT on unconnected stream %d\n", streamid))
			free(data);
			return -2;
		case RPP_CLOSE_WAIT1:
		case RPP_CLOSE_WAIT2:
		case RPP_LAST_ACK:
			DBPRT(("INPUT on closed stream %d\n", streamid))
			free(data);
			return -2;
		default:
			break;
		}

		if (sequence < sp->recv_sequence) {
			DBPRT(("%s: OLD seq %d\n", id, sequence))
			free(data);
			return -2;
		}

		for (rpp=sp->recv_head, rprev=NULL; rpp;
				rprev=rpp, rpp=rpp->next) {
			if (rpp->sequence >= sequence)
				break;
		}
		if (rpp == NULL || rpp->sequence > sequence) {
			DBPRT(("%s: GOOD seq %d\n", id, sequence))
			data = realloc(data, len);
			assert(data != NULL);
			pkt = (struct recv_packet *)
				malloc(sizeof(struct recv_packet));
			assert(pkt != NULL);
			pkt->type = type;
			pkt->sequence = sequence;
			pkt->len = len-RPP_PKT_HEAD;
			pkt->data = (u_char *)data;
			if (rprev == NULL) {
				pkt->next = sp->recv_head;
				sp->recv_head = pkt;
			}
			else {
				pkt->next = rprev->next;
				rprev->next = pkt;
			}
			if (sp->recv_tail == rprev)
				sp->recv_tail = pkt;

			if (sp->state == RPP_OPEN_PEND)
				return -2;
			else
				return streamid;
		}
		else {
			DBPRT(("%s: DUPLICATE seq %d MAX seen %d\n",
				id, sequence, rpp->sequence))
			free(data);
		}
		break;

	case RPP_HELLO1:
		/*
		** HELLO1 packets have the remote side's stream index
		** in the "streamid" field and open key in the sequence.
		*/
		DBPRT(("%s: HELLO1 stream %d sequence %d\n",
				id, streamid, sequence))
		free(data);
		for (i=0; i<stream_num; i++) {
			sp = &stream_array[i];

			if (sp->state <= RPP_FREE)
				continue;
			if (memcmp(&sp->addr, &addr, sizeof(addr)))
				continue;
			if (sp->open_key == sequence) {
				rpp_send_out();
				return -2;
			}
			DBPRT(("OLD STREAM state %d reopened %d %d\n",
				sp->state, sp->open_key, sequence))
			clear_stream(sp);	/* old stream */
		}

		i = rpp_create_sp();
		if (i == -1)
			return -1;
		sp = &stream_array[i];
		sp->state = RPP_OPEN_PEND;
		sp->fd = fd;
		memcpy(&sp->addr, &addr, sizeof(addr));
		if ((hp = rpp_get_cname(&addr)) != NULL)
			rpp_alist(hp, sp);
		sp->stream_id = streamid;
		sp->open_key = sequence;
		open_key = MAX(open_key, sequence);
		rpp_form_pkt(i, RPP_HELLO2, i, NULL, 0);
		rpp_send_out();
		break;

	case RPP_HELLO2:
		/*
		** HELLO2 packet has this side's stream index in
		** "streamid" as usual and the remote side's
		** stream index overloaded in the "sequence" field.
		*/
		DBPRT(("%s: HELLO2 stream %d sequence %d\n",
				id, streamid, sequence))
		free(data);
		if ((sp = rpp_check_pkt(streamid, &addr)) == NULL)
			return -2;

		switch (sp->state) {

		case RPP_OPEN_WAIT:
			sp->state = RPP_CONNECT;
			break;

		case RPP_CLOSE_WAIT1:	/* called close before open done */
		case RPP_LAST_ACK:
			break;

		default:
			if (sp->stream_id == sequence) {
				DBPRT(("%s: stream %d got DUP HELLO2 %d\n",
						id, streamid, sp->state))
				if (rpp_send_ack(sp, sp->open_key) == -1)
					return -1;
			}
			else {
				DBPRT(("%s: NON-DUP HELLO2\n", id))
			}
			return -2;
		}

		sp->stream_id = sequence;
		if (rpp_send_ack(sp, sp->open_key) == -1)
			return -1;

		if ((spp = sp->send_head) == NULL) {
			DBPRT(("%s: stream %d got HELLO2 but sendq NULL\n",
					id, streamid))
			return -2;
		}
		if (spp->type != RPP_HELLO1) {
			DBPRT(("%s: stream %d sendq %d rather than HELLO1\n",
					id, streamid, spp->type))
			return -2;
		}
		sp->send_head = spp->next;	/* remove HELLO1 pkt */
		if (sp->send_tail == spp)
			sp->send_tail = NULL;
		dqueue(spp);

		/*
		** Put any waitting packets onto the send queue
		*/
		for (spp=sp->send_head; spp; spp=spp->next) {
			int	len = spp->len;

			DBPRT(("%s: idx %d link %d seq %d len %d to sendq\n",
				id, streamid, spp->type, spp->sequence, len))
			I2TOH(spp->type, (char *)&spp->data[len])
			I8TOH(sp->stream_id,
					(char *)&spp->data[len+RPP_HDR_SID])
			I8TOH(spp->sequence,
					(char *)&spp->data[len+RPP_HDR_SEQ])
			I8TOH(crc(spp->data, (u_long)(len+RPP_PKT_CRC)),
					(char *)&spp->data[len+RPP_PKT_CRC])

			if (bottom)
				bottom->down = spp;
			spp->up = bottom;
			spp->down = NULL;
			if (top == NULL)		/* first one */
				top = spp;
			bottom = spp;
		}

		break;

	default:
		DBPRT(("%s: UNKNOWN packet type %d stream %d sequence %d\n",
				id, type, streamid, sequence))
		free(data);
		break;
	}
	return -2;

err_out:
	free(data);
	return -2;
}

/*
**	Do recv calls until there is one that shows data.
*/
static
int
rpp_recv_all()
{
	int	i, ret;
	int	rc = -3;

	for (i=0; i<rpp_fd_num; i++) {
		ret = rpp_recv_pkt(rpp_fd_array[i]);
		rc = MAX(ret, rc);
		if (ret == -1)
			break;
	}
	return rc;
}

/*
**	Check to see if any packet being sent out on a stream has
**	been sent more than a reasonable number of times.
*/
static
void
rpp_stale(sp)
    struct	stream	*sp;
{
	struct	send_packet	*pp;

	if (sp->state <= RPP_FREE || sp->state == RPP_STALE)
		return;
	for (pp = sp->send_head; pp; pp = pp->next) {
		if (pp->sent_out >= RPP_RETRY)
			break;
	}
	if (pp) {
		DBPRT(("STALE PACKET seq %d sent %d\n",
			pp->sequence, pp->sent_out))
		switch (sp->state) {
		case RPP_OPEN_PEND:
		case RPP_CLOSE_WAIT1:
		case RPP_CLOSE_WAIT2:
		case RPP_LAST_ACK:
			clear_stream(sp);
			break;
		default:
			sp->state = RPP_STALE;
			break;
		}
	}
}

/*
**	Form data packets for any pending data.  If flag is true,
**	create an EOD packet too.
*/
static
int
rpp_dopending(index, flag)
    int		index;
    int		flag;
{
	static	char	id[] = "dopending";
	struct	stream		*sp;
	struct	pending		*pp;

	DBPRT(("%s: entered index %d\n", id, index))
	sp = &stream_array[index];

	for (pp=sp->pend_head; pp != sp->pend_tail; pp=sp->pend_head) {
		rpp_form_pkt(index, RPP_DATA, sp->send_sequence,
				pp->data, RPP_PKT_DATA);
		sp->pend_head = pp->next;
		free(pp);
		sp->pend_attempt -= RPP_PKT_DATA;
		if (next_seq(&sp->send_sequence) == -1)
			return -1;
	}
	if (flag) {
		rpp_form_pkt(index, RPP_EOD, sp->send_sequence,
			pp ? pp->data : NULL, sp->pend_attempt);
		if (pp) {
			free(pp);
			sp->pend_head = NULL;
			sp->pend_tail = NULL;
		}
		sp->pend_attempt = 0;
		if (next_seq(&sp->send_sequence) == -1)
			return -1;
	}
	sp->pend_commit = sp->pend_attempt;
	return 0;
}

/*
**	Flush all data out of a stream -- do an end of message.
**	Return 0 if it all went well, -1 on error.
*/
int
rpp_flush(index)
    int		index;
{
	static	char	id[] = "flush";
	struct	stream	*sp;

	DBPRT(("%s: entered index %d\n", id, index))
	if (index < 0 || index >= stream_num) {
		errno = EINVAL;
		return -1;
	}
	sp = &stream_array[index];

	switch (sp->state) {

	case RPP_CLOSE_PEND:
		errno = EPIPE;
		return -1;

	case RPP_DEAD:
	case RPP_FREE:
	case RPP_OPEN_PEND:
	case RPP_CLOSE_WAIT1:
	case RPP_CLOSE_WAIT2:
	case RPP_LAST_ACK:
		errno = ENOTCONN;
		return -1;

	default:
		break;
	}

/*
**	if something is pending or we need to return a zero len EOM,
**	call rpp_dopending().
*/
	if (sp->pend_head != NULL || sp->send_head == NULL) {
		if (rpp_dopending(index, TRUE))
			return -1;
	}

	if (rpp_recv_all() == -1)
		return -1;
	rpp_send_out();
	return 0;
}

int
rpp_bind(port)
    uint	port;
{
	struct	sockaddr_in	from;
	int			flags;

	if (rpp_fd == -1) {
		if ((rpp_fd = socket(PF_INET, SOCK_DGRAM, 0)) == -1)
			return -1;

		/* set close on exec */
		if ((flags = fcntl(rpp_fd, F_GETFD)) == -1) {
			close(rpp_fd);
			rpp_fd = -1;
			return -1;
		}
		flags |= FD_CLOEXEC;
		if (fcntl(rpp_fd, F_SETFD, flags) == -1) {
			close(rpp_fd);
			rpp_fd = -1;
			return -1;
		}

		/* set no delay */
		if ((flags = fcntl(rpp_fd, F_GETFL)) == -1) {
			close(rpp_fd);
			rpp_fd = -1;
			return -1;
		}
#if defined(FNDELAY) && !defined(__hpux)
		flags |= FNDELAY;
#else
		flags |= O_NONBLOCK;
#endif
		if (fcntl(rpp_fd, F_SETFL, flags) == -1) {
			close(rpp_fd);
			rpp_fd = -1;
			return -1;
		}
	}

	if (rpp_fd_array != NULL) {
		int	i;

		for (i=0; i<rpp_fd_num; i++) {
			if (rpp_fd_array[i] == rpp_fd)
				return rpp_fd;
		}
	}
        memset(&from, '\0', sizeof(from));
        from.sin_family = AF_INET;
        from.sin_addr.s_addr = htonl(INADDR_ANY);
        from.sin_port = htons((u_short)port);

	if (bind(rpp_fd, (struct sockaddr *)&from, sizeof(from)) == -1)
		return -1;

	DBPRT(("bind to port %d\n", ntohs(from.sin_port)))
	if (rpp_fd_array == NULL) {
		rpp_fd_array = (int *)malloc(sizeof(int));
		rpp_fd_num = 1;
#if defined(HAVE_ATEXIT)
		(void)atexit(rpp_shutdown);
#elif defined(HAVE_ON_EXIT)
		(void)atexit(rpp_shutdown, 0);
#else
		/* atexit() or on_exit() must be defined */
		abort compile
#endif
	}
	else {
		rpp_fd_num++;
		rpp_fd_array = (int *)realloc(rpp_fd_array,
				sizeof(int)*rpp_fd_num);
	}
	assert(rpp_fd_array);
	rpp_fd_array[rpp_fd_num-1] = rpp_fd;
	return rpp_fd;
}

/*
**	Allocate a communication stream.
*/
int
rpp_open(name, port)
    char	*name;
    uint	port;
{
	static	char	id[] = "open";
	int			i, stream;
	struct	hostent		*hp;
	struct	stream		*sp;

	DBPRT(("%s: entered %s:%d\n", id, name, port))

	if (rpp_bind(0) == -1)	/* bind if we need to */
		return -1;

	/*
	** First, we look up the IP address for this name.
	*/
	if ((hp = gethostbyname(name)) == NULL) {
		DBPRT(("%s: host %s not found\n", id, name))
		errno = ENOENT;
		return -1;
	}

	for (i=0; i<stream_num; i++) {
		sp = &stream_array[i];
		if (sp->state <= RPP_FREE)
			continue;
		if (memcmp(&sp->addr.sin_addr, hp->h_addr, hp->h_length))
			continue;
		if (sp->addr.sin_port != htons((unsigned short)port))
			continue;
		if (sp->addr.sin_family != hp->h_addrtype)
			continue;
		DBPRT(("%s: DUP STREAM state %d reopened %d\n",
			id, sp->state, sp->open_key))

		if (sp->state > RPP_CLOSE_PEND)
			clear_stream(sp);	/* old stream */
		else
			return i;
	}

	stream = rpp_create_sp();
	if (stream == -1)
		return -1;
	sp = &stream_array[stream];

	if (open_key == 0)
		open_key = (int)time(0) & 0x0fff;
	/*
	** We save the address returned for the name given so we
	** can send out on the perfered interface.
	*/
	memcpy(&sp->addr.sin_addr, hp->h_addr, hp->h_length);
	sp->addr.sin_port = htons((unsigned short)port);
	sp->addr.sin_family = hp->h_addrtype;
	sp->fd = rpp_fd;


	if (hp->h_addr_list[1] == NULL) {
		if ((hp = rpp_get_cname(&sp->addr)) == NULL) {
			errno = ENOENT;
			return -1;
		}
	}
	rpp_alist(hp, sp);
	sp->stream_id = stream;	/* use my streamid for HELLO1 */
	sp->state = RPP_OPEN_WAIT;
	sp->open_key = open_key++;
	rpp_form_pkt(stream, RPP_HELLO1, sp->open_key, NULL, 0);
	sp->stream_id = -1;	/* don't know his stream id yet */

	if (rpp_recv_all() == -1)
		return -1;
	rpp_send_out();
	return stream;
}

/*
**	Return the network address for a stream.
*/
struct	sockaddr_in*
rpp_getaddr(index)
    int		index;
{
	static	char	id[] = "getaddr";
	struct	stream	*sp;

	DBPRT(("%s: entered index %d\n", id, index))
	if (index < 0 || index >= stream_num) {
		errno = EINVAL;
		return NULL;
	}

	sp = &stream_array[index];
	if (sp->state <= RPP_FREE) {
		errno = ENOTCONN;
		return NULL;
	}

	return &sp->addr;
}

/*
**	Free all memory and close the socket.
*/
void
rpp_terminate()
{
	struct	stream		*sp;
	struct	send_packet	*spp;
	struct	pending		*ppp;
	struct	recv_packet	*rpp;
	int			i;

	for (i=0; i<rpp_fd_num; i++)
		(void)close(rpp_fd_array[i]);
	if (rpp_fd_array) {
		free(rpp_fd_array);
		rpp_fd_array = NULL;
		rpp_fd_num = 0;
	}
	for (i=0; i<stream_num; i++) {
		sp = &stream_array[i];

		if (sp->state == RPP_DEAD)
			continue;

		for (ppp = sp->pend_head; ppp; ppp = sp->pend_head) {
			free(ppp->data);
			sp->pend_head = ppp->next;
			free(ppp);
		}
		for (rpp = sp->recv_head; rpp; rpp = sp->recv_head) {
			if (rpp->data)
				free(rpp->data);
			sp->recv_head = rpp->next;
			free(rpp);
		}
		for (spp = sp->send_head; spp; spp = sp->send_head) {
			free(spp->data);
			sp->send_head = spp->next;
			free(spp);
		}
	}
	top = NULL;
	bottom = NULL;
	if (stream_array)
		free(stream_array);
	stream_num = 0;
	stream_array = NULL;
	rpp_fd = -1;
}

/*
**	Shutdown the library.  Flush and close all open streams
**	and call rpp_terminate().
*/
void
rpp_shutdown()
{
	int			timeouts, num, i;
	fd_set			fdset;
	struct	timeval		tv;

	FD_ZERO(&fdset);
	for (i=0; i<stream_num; i++)
		(void)rpp_close(i);

	for (timeouts = 0; timeouts < 3;) {
		for (i=0; i<stream_num; i++) {
			if (stream_array[i].state > RPP_FREE)
				break;
		}
		if (i == stream_num)
			break;

		DBPRT(("shutdown: stream %d state %d\n",
				i, stream_array[i].state))
		if ((num = rpp_recv_all()) == -1)
			break;
		rpp_send_out();
		if (num == -3) {        /* got nothing -- wait a bit */
			tv.tv_sec = RPP_TIMEOUT;
			tv.tv_usec = 0;
			for (i=0; i<rpp_fd_num; i++)
				FD_SET(rpp_fd_array[i], &fdset);
			i = select(FD_SETSIZE, &fdset, NULL, NULL, &tv);
			if (i == 0)
				timeouts++;
			if (i == -1)
				break;
		}
	}
	rpp_terminate();
}

/*
**	Terminate a connection stream.
**	Return 0 if it all went well, -1 on error.
*/
int
rpp_close(index)
    int		index;
{
	static	char		id[] = "close";
	struct	stream		*sp;

	DBPRT(("%s: entered index %d\n", id, index))
	if (index < 0 || index >= stream_num) {
		errno = EINVAL;
		return -1;
	}
	sp = &stream_array[index];

	switch (sp->state) {

	case RPP_STALE:
		clear_stream(sp);
		return 0;

	case RPP_CLOSE_PEND:
		sp->state = RPP_LAST_ACK;
		break;

	case RPP_OPEN_WAIT:
	case RPP_CONNECT:
		if (sp->pend_head != NULL) {
			if (rpp_dopending(index, TRUE))
				return -1;
		}
		sp->state = RPP_CLOSE_WAIT1;
		break;

	default:
		errno = ENOTCONN;
		return -1;		/* stream closed */
	}

	rpp_form_pkt(index, RPP_GOODBYE,
			sp->send_sequence, NULL, 0);
	if (rpp_recv_all() == -1)
		return -1;
	rpp_send_out();
	return 0;
}

/*
**	Add information to the stream given by index.
**	Return -1 on error, otherwise number of bytes written.
*/
int
rpp_write(index, buf, len)
    int		index;
    void	*buf;
    int		len;
{
	static	char	id[] = "write";
	struct	stream	*sp;
	struct	pending	*pp;
	int		hold, residual, more;

	DBPRT(("%s: entered index %d\n", id, index))
	if (index < 0 || index >= stream_num || len < 0) {
		errno = EINVAL;
		return -1;
	}
	if (len == 0)
		return 0;
	sp = &stream_array[index];
	rpp_stale(sp);			/* check freshness */

	switch (sp->state) {

	case RPP_STALE:
		errno = ETIMEDOUT;
		return -1;

	case RPP_CLOSE_PEND:
		errno = EPIPE;
		return -1;

	case RPP_OPEN_PEND:			/* shouldn't happen */
	case RPP_DEAD:
	case RPP_FREE:
	case RPP_CLOSE_WAIT1:			/* stream closed */
	case RPP_CLOSE_WAIT2:
	case RPP_LAST_ACK:
		errno = ENOTCONN;
		return -1;

	default:
		break;
	}

	residual = 0;
	while (residual < len) {
		hold = sp->pend_attempt % RPP_PKT_DATA;
		if ((pp = sp->pend_tail) == NULL || hold == 0) {
			pp = (struct pending *)malloc(sizeof(struct pending));
			if (sp->pend_tail == NULL)
				sp->pend_head = pp;
			else
				sp->pend_tail->next = pp;
			sp->pend_tail = pp;
			pp->data = (u_char *)malloc(RPP_PKT_SIZE);
			assert(pp->data != NULL);
			pp->next = NULL;
		}
		more = MIN(len - residual, RPP_PKT_DATA - hold);
		memcpy(&pp->data[hold], (char *)buf + residual, more);
		residual += more;
		sp->pend_attempt += more;
	}

	if (rpp_recv_all() == -1)
		return -1;
	rpp_send_out();
	return residual;
}

/*
**	Check a stream to see if it needs attention.
*/
static
int
rpp_attention(index)
    int		index;
{
	static	char		id[] = "attention";
	int			mesg, count;
	int			seq;
	struct	stream		*sp;
	struct	recv_packet	*pp;

	sp = &stream_array[index];
	DBPRT(("%s: stream %d in state %d addr %s\n",
			id, index, sp->state, netaddr(&sp->addr)))
	rpp_stale(sp);

	switch (sp->state) {

	case RPP_STALE:			/* need to report error */
		return TRUE;

	case RPP_CLOSE_PEND:		/* we haven't closed yet */
	case RPP_CONNECT:		/* check for message */
		break;

	default:
		return FALSE;

	}

	if (sp->msg_cnt > 0 && sp->recv_attempt <= sp->msg_cnt)
		return TRUE;		/* message to read */

	mesg = FALSE;
	count = 0;
	for (pp=sp->recv_head, seq=sp->recv_sequence; pp; pp=pp->next, seq++) {
		count += pp->len;
		if (pp->sequence != seq)
			break;
		if (pp->type != RPP_DATA) {	/* end of message */
			mesg = TRUE;
			break;
		}
	}
	if (mesg)
		sp->msg_cnt = count;
	return mesg;
}

/*
**	Check some state before reading or skipping.  If it is
**	okay to continue, return 1.  Otherwise, return <= 0.
*/
static
int
rpp_okay(index)
    int	index;
{
	struct	stream		*sp;
	fd_set			fdset;
	struct	timeval		tv;

	FD_ZERO(&fdset);
	while (rpp_attention(index) == FALSE) {
		int	i;

		tv.tv_sec = RPP_TIMEOUT;
		tv.tv_usec = 0;
		for (i=0; i<rpp_fd_num; i++)
			FD_SET(rpp_fd_array[i], &fdset);
		i = select(FD_SETSIZE, &fdset, NULL, NULL, &tv);
		if (i == -1) 
			return -1;
		if (rpp_recv_all() == -1)
			return -1;
		rpp_send_out();
	}
  
	sp = &stream_array[index];
	if (sp->state == RPP_STALE) {		/* stale output */
		errno = ETIMEDOUT;
		return -1;
	}
	if (sp->recv_attempt == sp->msg_cnt) {	/* end of message */
		if (sp->state == RPP_CLOSE_PEND)
			return -2;
		else
			return 0;
	}
	return 1;
}

/*
**	Read a message.  Return data up to the end of a message
**	or the end of the provided buffer.
**	Return -1 on error, -2 if other side has closed, otherwise
**	number of bytes read.
*/
int
rpp_read(index, buf, len)
    int		index;
    void	*buf;
    int		len;
{
	static	char		id[] = "read";
	int			hiwater, cpylen, hold, ret, xlen;
	struct	recv_packet	*pp;
	struct	stream		*sp;

	DBPRT(("%s: entered index %d\n", id, index))

	if (index < 0 || index >= stream_num || len < 0) {
		errno = EINVAL;
		return -1;
	}
	if (len == 0)
		return 0;
	sp = &stream_array[index];

	switch (sp->state) {

	case RPP_DEAD:
	case RPP_FREE:
	case RPP_CLOSE_WAIT1:
	case RPP_CLOSE_WAIT2:
	case RPP_LAST_ACK:
		errno = ENOTCONN;
		return -1;	/* stream closed */

	default:
		break;
	}

	if ((ret = rpp_okay(index)) <= 0)
		return ret;

	sp = &stream_array[index];
	cpylen = 0;				/* find packet to copy from */
	for (pp=sp->recv_head; pp; pp=pp->next) {
		int	bump = cpylen + pp->len;

		if (sp->recv_attempt < bump)
			break;
		cpylen = bump;
	}
	hiwater = 0;
	xlen = MIN(len, sp->msg_cnt);
	hold = sp->recv_attempt - cpylen;	/* start point in pkt data */
	while (pp && xlen > hiwater) {		/* got room */
		cpylen = MIN(pp->len-hold, xlen-hiwater);
		memcpy((char *)buf + hiwater, &pp->data[hold], cpylen);
		hiwater += cpylen;
		sp->recv_attempt += cpylen;
		hold = 0;
		pp = pp->next;
	}
	return hiwater;
}

/*
**	Commit data which has been read up to recv_attempt if flag
**	is TRUE.  Otherwise, set recv_attempt back to the previous
**	commit point recv_commit.
**	Return -1 on error, FALSE on decommit or if end-of-message has
**	not been reached, TRUE if end-of-message has been reached.
*/
int
rpp_rcommit(index, flag)
    int		index;
    int		flag;
{
	static	char		id[] = "rcommit";
	struct	stream		*sp;

	DBPRT(("%s: entered index %d\n", id, index))

	if (index < 0 || index >= stream_num) {
		errno = EINVAL;
		return -1;
	}
	sp = &stream_array[index];

	switch (sp->state) {

	case RPP_CLOSE_WAIT1:			/* stream closed */
	case RPP_CLOSE_WAIT2:
	case RPP_LAST_ACK:
	case RPP_OPEN_PEND:			/* shouldn't happen */
	case RPP_FREE:
	case RPP_DEAD:
		errno = ENOTCONN;
		return -1;

	default:
		break;
	}

	if (flag == FALSE) {			/* no commit */
		sp->recv_attempt = sp->recv_commit;
		return 0;
	}
	sp->recv_commit = sp->recv_attempt;
	return (sp->recv_commit == sp->msg_cnt);
}

/*
**	Reset end-of-message condition on a stream.  Any packets
**	on the receive queue are freed.
**	Return -1 on error, 0 otherwise.
*/
int
rpp_eom(index)
    int		index;
{
	static	char	id[] = "eom";
	struct	stream		*sp;
	struct	recv_packet	*pp;

	DBPRT(("%s: entered index %d\n", id, index))

	if (index < 0 || index >= stream_num) {
		errno = EINVAL;
		return -1;
	}
	sp = &stream_array[index];
	switch (sp->state) {

	case RPP_CLOSE_WAIT1:			/* stream closed */
	case RPP_CLOSE_WAIT2:
	case RPP_LAST_ACK:
	case RPP_OPEN_PEND:			/* shouldn't happen */
	case RPP_FREE:
	case RPP_DEAD:
		errno = ENOTCONN;
		return -1;

	default:
		break;
	}
/*
**	work though recv packets
*/
	for (pp=sp->recv_head; pp; pp=sp->recv_head) {
		if (pp->type == RPP_GOODBYE)	/* stream finished */
			break;
		if (sp->msg_cnt < pp->len)
			break;
		sp->recv_sequence++;
		sp->msg_cnt -= pp->len;
		if (pp->data)
			free(pp->data);
		sp->recv_head = pp->next;
		free(pp);
	}
	if (sp->recv_head == NULL)
		sp->recv_tail = NULL;
	sp->recv_attempt = 0;
	sp->recv_commit = 0;
	return 0;
}

/*
**	Commit data which has been written up to pend_attempt if flag
**	is TRUE.  Otherwise, set pend_attempt back to the previous
**	commit point pend_commit.
**	Return -1 on error, 0 otherwise.
*/
int
rpp_wcommit(index, flag)
    int		index;
    int		flag;
{
	static	char		id[] = "wcommit";
	struct	pending		*pp, *next;
	struct	stream		*sp;

	DBPRT(("%s: entered index %d\n", id, index))
	if (index < 0 || index >= stream_num) {
		errno = EINVAL;
		return -1;
	}
	sp = &stream_array[index];
	switch (sp->state) {

	case RPP_CLOSE_PEND:
		errno = EPIPE;
		return -1;

	case RPP_STALE:
		errno = ETIMEDOUT;
		return -1;

	case RPP_CLOSE_WAIT1:			/* stream closed */
	case RPP_CLOSE_WAIT2:
	case RPP_LAST_ACK:
	case RPP_OPEN_PEND:			/* shouldn't happen */
	case RPP_FREE:
	case RPP_DEAD:
		errno = ENOTCONN;
		return -1;

	default:
		break;
	}

	if (flag) {			/* commit */
		if (rpp_dopending(index, FALSE))
			return -1;
		if (rpp_recv_all() == -1)
			return -1;
		rpp_send_out();
		return 0;
	}

	sp->pend_attempt = sp->pend_commit;
	if (sp->pend_head == NULL)
		return 0;
	for (pp=sp->pend_head->next; pp; pp=next) {
		free(pp->data);
		next = pp->next;
		free(pp);
	}
	sp->pend_head->next = NULL;
	sp->pend_tail = sp->pend_head;
	return 0;
}

/*
**	Skip len characters of a message.
*/
int
rpp_skip(index, len)
    int		index;
    int		len;
{
	static	char		id[] = "skip";
	struct	stream		*sp;
	struct	recv_packet	*pp;
	int			ret, hiwater;

	DBPRT(("%s: entered index %d\n", id, index))
	if (index < 0 || index >= stream_num) {
		errno = EINVAL;
		return -1;
	}
	sp = &stream_array[index];

	switch (sp->state) {

	case RPP_DEAD:
	case RPP_FREE:
	case RPP_CLOSE_WAIT1:
	case RPP_CLOSE_WAIT2:
	case RPP_LAST_ACK:
		errno = ENOTCONN;
		return -1;	/* stream closed */

	default:
		break;
	}

	if ((ret = rpp_okay(index)) <= 0)
		return ret;

	sp = &stream_array[index];
	hiwater = MIN(sp->msg_cnt - sp->recv_attempt, len);
	sp->recv_attempt += hiwater;
	return hiwater;
}

/*
**	Check for any stream with a message waiting and
**	return the stream number or a -1 if there are none.
*/
int
rpp_poll()
{
	static	char	id[] = "poll";
	int			i;

	DBPRT(("%s: entered streams %d\n", id, stream_num))
	/*
	** Read socket to get any packets
	*/
	for (;;) {
		i = rpp_recv_all();
		if (i == -1 || i == -3)
			break;
	}
	if (i == -1)
		return -1;

	/*
	** See if any stream has a message waiting.
	*/
	for (i=0; i<stream_num; i++) {
		if (rpp_attention(i))
			break;
	}
	if (i < stream_num)	/* found one */
		return i;

	rpp_send_out();
	return -2;
}

/*
**	Process any stream i/o.
**	Return 0 or a -1 if there was an error.
*/
int
rpp_io()
{
	static	char	id[] = "io";
	int			i;

	DBPRT(("%s: entered streams %d\n", id, stream_num))
	/*
	** Read socket to get any packets
	*/
	for (;;) {
		i = rpp_recv_all();
		if (i == -1 || i == -3)
			break;
	}
	if (i == -1)
		return -1;

	rpp_send_out();
	return 0;
}

/*
**	Read a character.
**	Returns  >=0	the char read
**		  -1	error or EOD
**		  -2	EOF
*/
int
rpp_getc(index)
    int		index;
{
	int	ret;
	u_char	c;

	if ((ret = rpp_read(index, &c, 1)) == 1)
		return ((int)c);
	return ((ret == -2) ? -2 : -1);
}

/*
**	Write a character.
*/
int
rpp_putc(index, c)
    int		index;
    int		c;
{
	u_char	x = (u_char)c;

	if (rpp_write(index, &x, 1) != 1)
		return -1;
	return 0;
}
