/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/

#include <pbs_config.h>   /* the master config generated by configure */

#include	<stdio.h>
#include	<stdlib.h>
#include	<limits.h>
#include	<fcntl.h>
#include	<netdb.h>
#include	<string.h>
#include	<errno.h>
#include	<assert.h>
#include	<sys/types.h>
#include	<sys/socket.h>
#include	<sys/time.h>
#include	<netinet/in.h>
#include	<netdb.h>

#include	"dis.h"
#include	"dis_init.h"
#include	"tm.h"
#include	"net_connect.h"

static	char	ident[] = "@(#) $RCSfile: tm.c,v $ $Revision: 2.1 $";

/*
** Set up a debug print macro.
*/
#ifdef  DEBUG
#define DBPRT(x) \
{ \
	int	err = errno; \
	printf x; \
	errno = err; \
}
#else
#define DBPRT(x)
#endif

#ifndef	MIN
#define	MIN(a, b)	(((a) < (b)) ? (a) : (b))
#endif

/*
**	Allocate some string space to hold the values passed in the
**	enviornment from MOM.
*/
static	char		*tm_jobid = NULL;
static	int		tm_jobid_len = 0;
static	char		*tm_jobcookie = NULL;
static	int		tm_jobcookie_len = 0;
static	tm_task_id	tm_jobtid = TM_NULL_TASK;
static	tm_node_id	tm_jobndid = TM_ERROR_NODE;
static	int		tm_momport = 0;
static	int		local_conn = -1;

/*
**	Events are the central focus of this library.  They are tracked
**	in a hash table.  Many of the library calls return events.  They
**	are recorded and as information is received from MOM's, the
**	event is updated and marked so tm_poll() can return it to the user.
*/
#define	EVENT_HASH	128
struct	event_info {
	tm_event_t		e_event;	/* event number */
	tm_node_id		e_node;		/* destination node */
	int			e_mtype;	/* message type sent */
	void			*e_info;	/* possible returned info */
	struct	event_info	*e_next;	/* link to next event */
	struct	event_info	*e_prev;	/* link to prev event */
}	typedef event_info;
static	event_info	*event_hash[EVENT_HASH];
static	int		event_count = 0;

/*
**	Find an event number or return a NULL.
*/
static event_info *
find_event(x)
    tm_event_t	x;
{
	event_info	*ep;

	for (ep=event_hash[x % EVENT_HASH]; ep; ep=ep->e_next) {
		if (ep->e_event == x)
			break;
	}
	return ep;
}

/*
**	Delete an event.
*/
static void
del_event(ep)
    event_info	*ep;
{

	/* unlink event from hash list */
	if (ep->e_prev)
		ep->e_prev->e_next = ep->e_next;
	else
		event_hash[ep->e_event % EVENT_HASH] = ep->e_next;
	if (ep->e_next)
		ep->e_next->e_prev = ep->e_prev;

	/*
	**	Free any memory saved with the event.  This depends
	**	on whay type of event it is.
	*/
	switch (ep->e_mtype) {

	case TM_INIT:
	case TM_SPAWN:
	case TM_SIGNAL:
	case TM_OBIT:
	case TM_POSTINFO:
		break;

	case TM_TASKS:
	case TM_GETINFO:
	case TM_RESOURCES:
		free(ep->e_info);
		break;

	default:
		DBPRT(("del_event: unknown event command %d\n", ep->e_mtype))
		break;
	}
	free(ep);

	if (--event_count == 0) {
		close(local_conn);
		local_conn = -1;
	}
	return;
}

/*
**	Create a new event number.
*/
static tm_event_t
new_event()
{
	static	tm_event_t	next_event = TM_NULL_EVENT+1;
	event_info		*ep;
	tm_event_t		ret;

	if (next_event == INT_MAX)
		next_event = TM_NULL_EVENT+1;
	for (;;) {
		ret = next_event++;

		for (ep=event_hash[ret % EVENT_HASH]; ep; ep=ep->e_next) {
			if (ep->e_event == ret)
				break;	/* innter loop: this number is in use */
		}
		if (ep == NULL)
			break;		/* this number is not in use */
	}
	return ret;
}

/*
**	Link new event number into the above hash table.
*/
static void
add_event(event, node, type, info)
    tm_event_t		event;
    tm_node_id		node;
    int			type;
    void		*info;
{
	event_info		*ep, **head;

	ep = (event_info *)malloc(sizeof(event_info));
	assert(ep != NULL);

	head = &event_hash[event % EVENT_HASH];
	ep->e_event = event;
	ep->e_node = node;
	ep->e_mtype = type;
	ep->e_info = info;
	ep->e_next = *head;
	ep->e_prev = NULL;
	if (*head)
		(*head)->e_prev = ep;
	*head = ep;

	event_count++;
	return;
}

/*
**	Sessions must be tracked by the library so tm_taskid objects
**	can be resolved into real tasks on real nodes.
**	We will use a hash table.
*/
#define	TASK_HASH	256
struct	task_info {
	char			*t_jobid;	/* jobid */
	tm_task_id		 t_task;	/* task id */
	tm_node_id		 t_node;	/* node id */
	struct	task_info	*t_next;	/* link to next task */
}	typedef	task_info;
static	task_info	*task_hash[TASK_HASH];

/*
**	Find a task table entry for a given task number or return a NULL.
*/
static task_info *
find_task(x)
    tm_task_id	x;
{
	task_info	*tp;

	for (tp=task_hash[x % TASK_HASH]; tp; tp=tp->t_next) {
		if (tp->t_task == x)
			break;
	}
	return tp;
}

/*
**	Create a new task entry and link it into the above hash
**	table.
*/
static tm_task_id
new_task(jobid, node, task)
    char	*jobid;
    tm_node_id	node;
    tm_task_id	task;
{
	static	char		id[] = "new_task";
	task_info		*tp, **head;

	DBPRT(("%s: jobid=%s node=%d task=%lu\n", id, jobid, node, task))
	if (jobid != tm_jobid && strcmp(jobid, tm_jobid) != 0) {
		DBPRT(("%s: task job %s not my job %s\n",
				id, jobid, tm_jobid))
		return TM_NULL_TASK;
	}
	if (node == TM_ERROR_NODE) {
		DBPRT(("%s: called with TM_ERROR_NODE\n", id))
		return TM_NULL_TASK;
	}

	if ((tp = find_task(task)) != NULL) {
		DBPRT(("%s: task %lu found with node %d should be %d\n",
			id, task, tp->t_node, node))
		return task;
	}

	if ((tp = (task_info *)malloc(sizeof(task_info))) == NULL)
		return TM_NULL_TASK;

	head = &task_hash[task % TASK_HASH];
	tp->t_jobid = tm_jobid;
	tp->t_task  = task;
	tp->t_node  = node;
	tp->t_next = *head;
	*head = tp;

	return task;
}

/*
**	Delete a task.
*/
static void
del_task(x)
    tm_task_id	x;
{
	task_info	*tp, *prev;

	prev = NULL;
	for (tp=task_hash[x % TASK_HASH]; tp; prev=tp, tp=tp->t_next) {
		if (tp->t_task == x)
			break;
	}
	if (tp) {
		if (prev)
			prev->t_next = tp->t_next;
		else
			task_hash[x % TASK_HASH] = tp->t_next;
		tp->t_next = NULL;
		if (tp->t_jobid != tm_jobid)
			free(tp->t_jobid);
		free(tp);
	}
	return;
}

/*
**	The nodes are tracked in an array.
*/
static	tm_node_id	*node_table = NULL;


/*
**	localmom() - make a connection to the local pbs_mom
**
**	The connection will remain open as long as there is an
**	outstanding event.
*/
#define PBS_NET_RC_FATAL -1
#define PBS_NET_RC_RETRY -2

static int
localmom()
{
	static  int	       have_addr = 0;
	static  struct in_addr hostaddr;
	struct	hostent	*hp;
	int		 i;
	struct sockaddr_in remote;
	int              sock;


	if (local_conn >= 0)
		return local_conn;	/* already have open connection */

	if (have_addr == 0) {
		/* lookup "localhost" and save address */
		if ((hp = gethostbyname("localhost")) == NULL) {
			DBPRT(("tm_init: localhost not found\n"))
                	return -1;
        	}
		assert(hp->h_length <= sizeof(hostaddr));
		memcpy(&hostaddr, hp->h_addr_list[0], hp->h_length);
		have_addr = 1;
	}

	for (i=0; i<5; i++) {

		/* get socket */

		sock = socket(AF_INET, SOCK_STREAM, 0);
		if (sock < 0)
			return -1;
			
		/* connect to specified local pbs_mom and port */

		remote.sin_addr = hostaddr;
		remote.sin_port = htons((unsigned short)tm_momport);
		remote.sin_family = AF_INET;
		if (connect(sock,(struct sockaddr *)&remote,sizeof(remote))<0) {
			switch (errno) {
			    case EINTR:
			    case EADDRINUSE:
			    case ETIMEDOUT:
			    case ECONNREFUSED:
				(void)close(sock);
				sleep(1);
				continue;
			    default:
				(void)close(sock);
				return -1;
			}
		} else {
			local_conn = sock;
			break;
		}

	}
	DIS_tcp_setup(local_conn);
	return (local_conn);
}

/*
**	startcom() - send request header to local pbs_mom.
**	If required, make connection to her.
*/

static int
startcom(com, event)
     int	com;
     tm_event_t	event;
{
	int     ret;

	if (localmom() == -1)
		return -1;

	ret = diswsi(local_conn, TM_PROTOCOL);
	if (ret != DIS_SUCCESS)
		goto done;
	ret = diswsi(local_conn, TM_PROTOCOL_VER);
	if (ret != DIS_SUCCESS)
		goto done;
	ret = diswcs(local_conn, tm_jobid, tm_jobid_len);
	if (ret != DIS_SUCCESS)
		goto done;
	ret = diswcs(local_conn, tm_jobcookie, tm_jobcookie_len);
	if (ret != DIS_SUCCESS)
		goto done;
	ret = diswsi(local_conn, com);
	if (ret != DIS_SUCCESS)
		goto done;
	ret = diswsi(local_conn, event);
	if (ret != DIS_SUCCESS)
		goto done;
	ret = diswui(local_conn, tm_jobtid);
	if (ret != DIS_SUCCESS)
		goto done;
	return DIS_SUCCESS;

 done:
	DBPRT(("startcom: send error %s\n", dis_emsg[ret]))
	close(local_conn);
	local_conn = -1;
	return ret;
}



/*
**	Initialize the Task Manager interface.
*/
int
tm_init(info, roots)
    void		*info;		/* in, currently unused */
    struct tm_roots	*roots;		/* out */
{
	static	int		flag = 0;
	tm_event_t		nevent, revent;
	char			*env, *hold;
	int			err, nerr;

	if (flag)
		return TM_ESYSTEM;
	flag = 1;

	if ((tm_jobid = getenv("PBS_JOBID")) == NULL)
		return TM_EBADENVIRONMENT;
	tm_jobid_len = strlen(tm_jobid);
	if ((tm_jobcookie = getenv("PBS_JOBCOOKIE")) == NULL)
		return TM_EBADENVIRONMENT;
	tm_jobcookie_len = strlen(tm_jobcookie);
	if ((env = getenv("PBS_NODENUM")) == NULL)
		return TM_EBADENVIRONMENT;
	tm_jobndid = (tm_node_id)strtol(env, &hold, 10);
	if (env == hold)
		return TM_EBADENVIRONMENT;
	if ((env = getenv("PBS_TASKNUM")) == NULL)
		return TM_EBADENVIRONMENT;
	if ((tm_jobtid = atoi(env)) == 0)
		return TM_EBADENVIRONMENT;
	if ((env = getenv("PBS_MOMPORT")) == NULL)
		return TM_EBADENVIRONMENT;
	if ((tm_momport = atoi(env)) == 0)
		return TM_EBADENVIRONMENT;

	nevent = new_event();

	/*
	 * send the following request:
	 *	header		(tm_init)
	 *	int		node number
	 *	int		task number
	 */

	if (startcom(TM_INIT, nevent) != DIS_SUCCESS)
		return TM_ESYSTEM;
	DIS_tcp_wflush(local_conn);
	add_event(nevent, TM_ERROR_NODE, TM_INIT, (void *)roots);

	if ((err = tm_poll(TM_NULL_EVENT, &revent, 1, &nerr)) != TM_SUCCESS)
		return err;
	return nerr;
}

/*
**
*/
int
tm_nodeinfo(list, nnodes)
    tm_node_id		**list;
    int			*nnodes;
{
	tm_node_id	*np;
	int		i;
	int		n = 0;

	if (node_table == NULL)
		return TM_ESYSTEM;

	for (np=node_table; *np != TM_ERROR_NODE; np++)
		n++;		/* how many nodes */

	np = (tm_node_id *)calloc(n, sizeof(tm_node_id));
	for (i=0; i<n; i++)
		np[i] = node_table[i];
	*list = np;
	*nnodes = i;
	return TM_SUCCESS;
}


/*
**	Starts <argv>[0] with environment <envp> at <where>.
*/
int
tm_spawn(argc, argv, envp, where, tid, event)
    int		 argc;		/* in  */
    char       **argv;		/* in  */
    char       **envp;		/* in  */
    tm_node_id	 where;		/* in  */
    tm_task_id	*tid;		/* out */
    tm_event_t	*event;		/* out */
{
	char		*cp;
	int		i;

	*event = new_event();
	if (startcom(TM_SPAWN, *event) != DIS_SUCCESS)
		return TM_ENOTCONNECTED;

	if (diswsi(local_conn, where) != DIS_SUCCESS)	/* send where */
		return TM_ENOTCONNECTED;

	if (diswsi(local_conn, argc) != DIS_SUCCESS)	/* send argc */
		return TM_ENOTCONNECTED;

	/* send argv strings across */

	for (i=0; i < argc; i++) {
		cp = argv[i];
		if (diswcs(local_conn, cp, strlen(cp)) != DIS_SUCCESS)
			return TM_ENOTCONNECTED;
	}

	/* send envp strings across */
	if (envp != NULL) {
		for (i=0; cp = envp[i]; i++) {
			if (diswcs(local_conn, cp, strlen(cp)) != DIS_SUCCESS)
				return TM_ENOTCONNECTED;
		}
	}
	if (diswcs(local_conn, "", 0) != DIS_SUCCESS)
		return TM_ENOTCONNECTED;
	DIS_tcp_wflush(local_conn);
	add_event(*event, where, TM_SPAWN, (void *)tid);
	return TM_SUCCESS;
}

/*
**	Sends a <sig> signal to all the process groups in the task
**	signified by the handle, <tid>.
*/
int
tm_kill(tid, sig, event)
    tm_task_id	tid;		/* in  */
    int		sig;		/* in  */
    tm_event_t	*event;		/* out */
{
	task_info	*tp;

	if ((tp = find_task(tid)) == NULL)
		return TM_ENOTFOUND;
	*event = new_event();
	if (startcom(TM_SIGNAL, *event) != DIS_SUCCESS)
		return TM_ENOTCONNECTED;
	if (diswsi(local_conn, tp->t_node) != DIS_SUCCESS)
		return TM_ENOTCONNECTED;
	if (diswsi(local_conn, tid) != DIS_SUCCESS)
		return TM_ENOTCONNECTED;
	if (diswsi(local_conn, sig) != DIS_SUCCESS)
		return TM_ENOTCONNECTED;
	DIS_tcp_wflush(local_conn);
	add_event(*event, tp->t_node, TM_SIGNAL, NULL);
	return TM_SUCCESS;
}

/*
**	Returns an event that can be used to learn when a task
**	dies.
*/
int
tm_obit(tid, obitval, event)
    tm_task_id	 tid;		/* in  */
    int		*obitval;	/* out */
    tm_event_t	*event;		/* out */
{
	task_info	*tp;

	if ((tp = find_task(tid)) == NULL)
		return TM_ENOTFOUND;
	*event = new_event();
	if (startcom(TM_OBIT, *event) != DIS_SUCCESS)
		return TM_ESYSTEM;
	if (diswsi(local_conn, tp->t_node) != DIS_SUCCESS)
		return TM_ESYSTEM;
	if (diswsi(local_conn, tid) != DIS_SUCCESS)
		return TM_ESYSTEM;
	DIS_tcp_wflush(local_conn);
	add_event(*event, tp->t_node, TM_OBIT, (void *)obitval);
	return TM_SUCCESS;
}

struct	taskhold {
	tm_task_id	*list;
	int		size;
	int		*ntasks;
};

/*
**	Makes a request for the list of tasks on <node>.  If <node>
**	is a valid node number, it returns the event that the list of
**	tasks on <node> is available.
*/
int
tm_taskinfo(node, tid_list, list_size, ntasks, event)
    tm_node_id	node;		/* in  */
    tm_task_id	*tid_list;	/* out */
    int		list_size;	/* in  */
    int		*ntasks;	/* out */
    tm_event_t	*event;		/* out */
{
	struct	taskhold	*thold;

	if (tid_list == NULL || list_size == 0 || ntasks == NULL)
		return TM_EBADENVIRONMENT;
	*event = new_event();
	if (startcom(TM_TASKS, *event) != DIS_SUCCESS)
		return TM_ESYSTEM;
	if (diswsi(local_conn, node) != DIS_SUCCESS)
		return TM_ESYSTEM;
	DIS_tcp_wflush(local_conn);

	thold = (struct taskhold *)malloc(sizeof(struct taskhold));
	assert(thold != NULL);
	thold->list = tid_list;
	thold->size = list_size;
	thold->ntasks = ntasks;
	add_event(*event, node, TM_TASKS, (void *)thold);
	return TM_SUCCESS;
}

/*
**	Returns the job-relative node number that holds or held <tid>.  In
**	case of an error, it returns TM_ERROR_NODE.
*/
int
tm_atnode(tid, node)
    tm_task_id	tid;		/* in  */
    tm_node_id	*node;		/* out */
{
	task_info	*tp;

	if ((tp = find_task(tid)) == NULL)
		return TM_ENOTFOUND;
	*node = tp->t_node;
	return TM_SUCCESS;
}

struct	reschold {
	char	*resc;
	int	len;
};

/*
**	Makes a request for a string specifying the resources
**	available on <node>.  If <node> is a valid node number, it
**	returns the event that the string specifying the resources on
**	<node> is available.  It returns ERROR_EVENT otherwise.
*/
int
tm_rescinfo(node, resource, len, event)
    tm_node_id		node;		/* in  */
    char		*resource;	/* out */
    int			len;		/* in  */
    tm_event_t		*event;		/* out */
{
	struct	reschold	*rhold;

	if (resource == NULL || len == 0)
		return TM_EBADENVIRONMENT;
	*event = new_event();
	if (startcom(TM_RESOURCES, *event) != DIS_SUCCESS)
		return TM_ESYSTEM;
	if (diswsi(local_conn, node) != DIS_SUCCESS)
		return TM_ESYSTEM;
	DIS_tcp_wflush(local_conn);

	rhold = (struct reschold *)malloc(sizeof(struct reschold));
	assert(rhold != NULL);
	rhold->resc = resource;
	rhold->len = len;

	add_event(*event, node, TM_RESOURCES, (void *)rhold);
	return TM_SUCCESS;
}

/*
**	Posts the first <nbytes> of a copy of *<info> within MOM on
**	this node, and associated with this task.  If <info> is
**	non-NULL, it returns the event that the effort to post *<info>
**	is complete.  It returns ERROR_EVENT otherwise.
*/
int
tm_publish(name, info, len, event)
    char	*name;		/* in  */
    void	*info;		/* in  */
    int		len;		/* in  */
    tm_event_t	*event;		/* out */
{

	*event = new_event();
	if (startcom(TM_POSTINFO, *event) != DIS_SUCCESS)
		return TM_ESYSTEM;
	if (diswst(local_conn, name) != DIS_SUCCESS)
		return TM_ESYSTEM;
	if (diswcs(local_conn, info, len) != DIS_SUCCESS)
		return TM_ESYSTEM;

	DIS_tcp_wflush(local_conn);
	add_event(*event, TM_ERROR_NODE, TM_POSTINFO, NULL);
	return TM_SUCCESS;
}

struct	infohold {
	void	*info;
	int	len;
	int	*info_len;
};

/*
**	Makes a request for a copy of the info posted by <tid>.  If
**	<tid> is a valid task, it returns the event that the
**	string specifying the info posted by <tid> is available.
*/
int
tm_subscribe(tid, name, info, len, info_len, event)
    tm_task_id		tid;		/* in  */
    char		*name;		/* in  */
    void		*info;		/* out */
    int			len;		/* in  */
    int			*info_len;	/* out */
    tm_event_t		*event;		/* out */
{
	task_info		*tp;
	struct	infohold	*ihold;

	if ((tp = find_task(tid)) == NULL)
		return TM_ENOTFOUND;
	*event = new_event();
	if (startcom(TM_GETINFO, *event) != DIS_SUCCESS)
		return TM_ESYSTEM;
	if (diswsi(local_conn, tp->t_node) != DIS_SUCCESS)
		return TM_ESYSTEM;
	if (diswsi(local_conn, tid) != DIS_SUCCESS)
		return TM_ESYSTEM;
	if (diswst(local_conn, name) != DIS_SUCCESS)
		return TM_ESYSTEM;
	DIS_tcp_wflush(local_conn);

	ihold = (struct infohold *)malloc(sizeof(struct infohold));
	assert(ihold != NULL);
	ihold->info = info;
	ihold->len = len;
	ihold->info_len = info_len;

	add_event(*event, tp->t_node, TM_GETINFO, (void *)ihold);
	return TM_SUCCESS;
}

/*
**	tm_finalize() - close out task manager interface
**
**	This function should be the last one called.  It is illegal to call
**	any other task manager function following this one.   All events are
**	freed and any connection to the task manager (pbs_mom) is closed.
**	This call is synchronous.
*/
int
tm_finalize()
{
	event_info	*e;
	int		 i = 0;

	while (event_count && (i < EVENT_HASH)) {
		while (e = event_hash[i]) {
			del_event(e);
		}
		++i;	/* check next slot in hash table */
	}
	return TM_SUCCESS;	/* what else */
}

/*
**	tm_notify() - set the signal to be sent on event arrival.
*/
int
tm_notify(tm_signal)
    int		tm_signal;
{
	return TM_ENOTIMPLEMENTED;
}

/*
**	tm_alloc() - make a request for additional resources.
*/
int
tm_alloc(resources, event)
    char	*resources;
    tm_event_t	*event;
{
	return TM_ENOTIMPLEMENTED;
}

/*
**	tm_dealloc() - drop a node from the job.
*/
int
tm_dealloc(node, event)
    tm_node_id	node;
    tm_event_t	*event;
{
	return TM_ENOTIMPLEMENTED;
}

/*
**	tm_create_event() - create a persistent event.
*/
int
tm_create_event(event)
    tm_event_t	*event;
{
	return TM_ENOTIMPLEMENTED;
}

/*
**	tm_destroy_event() - destroy a persistent event.
*/
int
tm_destroy_event(event)
    tm_event_t	*event;
{
	return TM_ENOTIMPLEMENTED;
}

/*
**	tm_register() - link a persistent event with action requests
**		from the task manager.
*/
int
tm_register(what, event)
    tm_whattodo_t	*what;
    tm_event_t		*event;
{
	return TM_ENOTIMPLEMENTED;
}

/*
**	tm_poll - poll to see if an event has be completed.
**
**	If "poll_event" is a valid event handle, see if it is completed;
**	else if "poll_event" is the null event, check for the first event that
**	is completed.
**
**	result_event is set to the completed event or the null event.
**
**	If wait is non_zero, wait for "poll_event" to be completed.
**
**	If an error ocurs, set tm_errno non-zero.
*/

int
tm_poll(poll_event, result_event, wait, tm_errno)
    tm_event_t	poll_event;
    tm_event_t	*result_event;
    int		wait;
    int		*tm_errno;
{
	static	char	id[] = "tm_poll";
	int		num, i;
	int		ret, mtype, nnodes;
	int		prot, protver;
	int		*obitvalp;
	event_info	*ep = NULL;
	tm_task_id	tid, *tidp;
	tm_event_t	nevent;
	tm_node_id	node;
	char		*jobid;
	char		*info;
	struct tm_roots	*roots;
	struct taskhold	*thold;
	struct infohold	*ihold;
	struct reschold	*rhold;
	extern	int	pbs_tcp_timeout;

	if (result_event == NULL)
		return TM_EBADENVIRONMENT;
	*result_event = TM_ERROR_EVENT;
	if (poll_event != TM_NULL_EVENT)
		return TM_ENOTIMPLEMENTED;
	if (tm_errno == NULL)
		return TM_EBADENVIRONMENT;

	/*
	** Setup tcp dis routines with a wait value appropriate for
	** the value of wait the user set.
	*/
	pbs_tcp_timeout = wait ? 2592000 : 0;
	DIS_tcp_funcs();

	prot = disrsi(local_conn, &ret);
	if (ret == DIS_EOD) {
		*result_event = TM_NULL_EVENT;
		return TM_SUCCESS;
	}
	else if (ret != DIS_SUCCESS) {
		DBPRT(("%s: protocol number dis error %d\n", id, ret))
		goto err;
	}
	if (prot != TM_PROTOCOL) {
		DBPRT(("%s: bad protocol number %d\n", id, prot))
		goto err;
	}

	/*
	** We have seen the start of a message.  Set the timeout value
	** so we wait for the remaining data of a message.
	*/
	pbs_tcp_timeout = 2592000;
	protver = disrsi(local_conn, &ret);
	if (ret != DIS_SUCCESS) {
		DBPRT(("%s: protocol version dis error %d\n", id, ret))
		goto err;
	}
	if (protver != TM_PROTOCOL_VER) {
		DBPRT(("%s: bad protocol version %d\n", id, protver))
		goto err;
	}

	mtype = disrsi(local_conn, &ret);
	if (ret != DIS_SUCCESS) {
		DBPRT(("%s: mtype dis error %d\n", id, ret))
		goto err;
	}
	nevent = disrsi(local_conn, &ret);
	if (ret != DIS_SUCCESS) {
		DBPRT(("%s: event dis error %d\n", id, ret))
		goto err;
	}

	*result_event = nevent;
	DBPRT(("%s: got event %d return %d\n", id, nevent, mtype))
	if ((ep = find_event(nevent)) == NULL) {
		DBPRT(("%s: No event found for number %d\n", id, nevent));
		(void)close(local_conn);
		local_conn = -1;
		return TM_ENOEVENT;
	}

	if (mtype == TM_ERROR) {	/* problem, read error num */
		*tm_errno = disrsi(local_conn, &ret);
		DBPRT(("%s: event %d error %d\n", id, nevent, *tm_errno));
		goto done;
	}

	*tm_errno = TM_SUCCESS;
	switch (ep->e_mtype) {

	case TM_INIT:
		nnodes = disrsi(local_conn, &ret);
		if (ret != DIS_SUCCESS)
			goto err;

		node_table = (tm_node_id *)calloc(nnodes+1,
				sizeof(tm_node_id));
		DBPRT(("%s: nodes %d\n", id, nnodes))
		for (i=0; i<nnodes; i++) {
			node_table[i] = disrsi(local_conn, &ret);
			if (ret != DIS_SUCCESS)
				goto err;
		}
		node_table[nnodes] = TM_ERROR_NODE;

		jobid = disrst(local_conn, &ret);
		if (ret != DIS_SUCCESS)
			goto err;
		DBPRT(("%s: daddy jobid %s\n", id, jobid))
		node = disrsi(local_conn, &ret);
		if (ret != DIS_SUCCESS)
			goto err;
		DBPRT(("%s: daddy node %d\n", id, node))
		tid = disrsi(local_conn, &ret);
		if (ret != DIS_SUCCESS)
			goto err;
		DBPRT(("%s: daddy tid %lu\n", id, tid))

		roots = (struct tm_roots *)ep->e_info;
		roots->tm_parent = new_task(jobid, node, tid);
		roots->tm_me = new_task(tm_jobid,
				tm_jobndid,
				tm_jobtid);
		roots->tm_nnodes = nnodes;
		roots->tm_ntasks = 0;		/* TODO */
		roots->tm_taskpoolid = -1;	/* what? */
		roots->tm_tasklist = NULL;	/* TODO */

		break;

	case TM_TASKS:
		thold = (struct taskhold *)ep->e_info;
		tidp = thold->list;
		num = thold->size;
		for (i=0;; i++) {
			tid = disrsi(local_conn, &ret);
			if (tid == TM_NULL_TASK)
				break;
			if (ret != DIS_SUCCESS)
				goto err;
			if (i < num) {
				tidp[i] = new_task(tm_jobid,
						ep->e_node, tid);
			}
		}
		if (i < num)
			tidp[i] = TM_NULL_TASK;
		*(thold->ntasks) = i;
		break;

	case TM_SPAWN:
		tid = disrsi(local_conn, &ret);
		if (ret != DIS_SUCCESS)
			goto err;
		tidp = (tm_task_id *)ep->e_info;
		*tidp = new_task(tm_jobid, ep->e_node, tid);
		break;

	case TM_SIGNAL:
		break;

	case TM_OBIT:
		obitvalp = (int *)ep->e_info;
		*obitvalp = disrsi(local_conn, &ret);
		if (ret != DIS_SUCCESS)
			goto err;
		break;

	case TM_POSTINFO:
		break;

	case TM_GETINFO:
		ihold = (struct infohold *)ep->e_info;
		info = disrcs(local_conn, (size_t *)ihold->info_len, &ret);
		if (ret != DIS_SUCCESS)
			break;
		memcpy(ihold->info, info, MIN(*ihold->info_len, ihold->len));
		free(info);
		break;

	case TM_RESOURCES:
		rhold = (struct reschold *)ep->e_info;
		info = disrst(local_conn, &ret);
		if (ret != DIS_SUCCESS)
			break;
		strncpy(rhold->resc, info, rhold->len);
		free(info);
		break;

	default:
		DBPRT(("%s: unknown event command %d\n", id, ep->e_mtype))
		goto err;
	}
done:
	del_event(ep);
	return TM_SUCCESS;

err:
	if (ep)
		del_event(ep);
	(void)close(local_conn);
	local_conn = -1;
	return TM_ENOTCONNECTED;
}
