/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/

#include <pbs_config.h>   /* the master config generated by configure */

#include <assert.h>
#include <limits.h>
#include <stdio.h>
#include <unistd.h>
#include <dirent.h>
#include <fcntl.h>
#include <errno.h>
#include <strings.h>
#include <pwd.h>
#include <mntent.h>
#include <asm/types.h>
#include <linux/quota.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/procfs.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/vfs.h>
#include <sys/sysmacros.h>
#include <sys/resource.h>
#include <syscall.h>

#include "pbs_error.h"
#include "portability.h"
#include "list_link.h"
#include "server_limits.h"
#include "attribute.h"
#include "resource.h"
#include "job.h"
#include "log.h"
#include "mom_mach.h"
#include "resmon.h"
#include "../rm_dep.h"


/*
**	System dependent code to gather information for the resource
**	monitor for a Linux i386 machine.
**
**	Resources known by this code:
**		cput		cpu time for a pid or session
**		mem		memory size for a pid or session in KB
**		resi		resident memory size for a pid or session in KB
**		sessions	list of sessions in the system
**		pids		list of pids in a session
**		nsessions	number of sessions in the system
**		nusers		number of users in the system
**		totmem		total memory size in KB
**		availmem	available memory size in KB
**		ncpus		number of cpus
**		physmem		physical memory size in KB
**		size		size of a file or filesystem
**		idletime	seconds of idle time
**		walltime	wall clock time for a pid
**		loadave		current load average
**		quota		quota information (sizes in kb)
*/

static char ident[] = "@(#) $RCSfile: mom_mach.c,v $ $Revision: 2.12 $";

#ifndef TRUE
#define FALSE	0
#define TRUE	1
#endif	/* TRUE */

static	char    procfs[] = "/proc";
static	DIR	*pdir = NULL;
static	int	pagesize;

extern	char	*ret_string;
extern	char	extra_parm[];
extern	char	no_parm[];

extern	struct	pbs_err_to_txt	pbs_err_to_txt[];
extern	time_t			time_now;

/*
** external functions and data
*/
extern	struct	config		*search A_((struct config *, char *));
extern	struct	rm_attribute	*momgetattr A_((char *));
extern	int			rm_errno;
extern	double	cputfactor;
extern	double	wallfactor;

/*
** local functions and data
*/
static char	*resi		A_((struct rm_attribute *attrib));
static char	*totmem		A_((struct rm_attribute *attrib));
static char	*availmem	A_((struct rm_attribute *attrib));
static char	*physmem	A_((struct rm_attribute *attrib));
static char	*ncpus		A_((struct rm_attribute *attrib));
static char	*walltime	A_((struct rm_attribute *attrib));
static char	*quota		A_((struct rm_attribute *attrib));

extern char	*loadave	A_((struct rm_attribute *attrib));
extern char	*nullproc	A_((struct rm_attribute *attrib));

int	wait_time = 10;

typedef	struct	proc_mem  {
        unsigned	total;
        unsigned	used;
        unsigned	free;
} proc_mem_t;

/*
** local resource array
*/
struct	config	dependent_config[] = {
	{ "resi",	resi },
	{ "totmem",	totmem },
	{ "availmem",	availmem },
	{ "physmem",	physmem },
	{ "ncpus",	ncpus },
	{ "loadave",	loadave },
	{ "walltime",	walltime },
	{ "quota",	quota },
	{ NULL,		nullproc },
};

unsigned linux_time = 0;
/*
 * support routine for getting system time -- sets linux_time
 */
void	proc_get_btime()
{
	FILE	*fp;
	char	label[256];
	

	if ((fp = fopen("/proc/stat", "r")) == NULL)  
		return;

	while (!feof(fp))  {
		fscanf(fp, "%s", label);
		if (strcmp(label, "btime"))  {
			fscanf(fp, "%*[^\n]%*c");
		}  else  {
			fscanf(fp, "%u", &linux_time);
			fclose(fp);
			return;
		}
	}
	
	fclose(fp);
	return;
}

static	char	stat_str[] = "%d (%[^)]) %c %*d %*d %d %*d %*d %u %*u \
%*u %*u %*u %d %d %d %d %*d %*d %*u %*u %u %u %u %*u %*u \
%*u %*u %*u %*u %*u %*u %*u %*u %*u %*u %*u";

/*
**	Convert jiffies to seconds.
*/
#define	JTOS(x)	 x = (x + CLK_TCK/2) / CLK_TCK;

/*
 * Linux /proc status routine.
 *
 *	Returns a pointer to a malloc'd proc_stat_t structure given
 *	a process number, or NULL if there is an error.  Takes the
 *	place of the ioctl call PIOCSTATUS in the irix imp of mom_mach.c
 *
 */
proc_stat_t	*get_proc_stat(pid)
	int	pid;
{
	static	proc_stat_t	ps;
	static	char		path[1024];
	FILE			*fd;
	unsigned		jiffies;
	struct 	stat		sb;

	sprintf(path, "/proc/%d/stat", pid);

	if ((fd = fopen(path, "r")) == NULL)
		return(NULL);

	if (fscanf(fd, stat_str, &ps.pid, path, &ps.state,
			&ps.session, &ps.flags, &ps.utime, &ps.stime, 
			&ps.cutime, &ps.cstime, &jiffies, &ps.vsize,
			&ps.rss) != 12)  {
		fclose(fd);
		return(NULL);
	}
	if (fstat(fileno(fd), &sb) == -1)  {
		fclose(fd);
		return(NULL);
	}  else  {
		ps.uid = sb.st_uid;
	}


	ps.start_time = linux_time + (jiffies / 100);
	ps.name = path;

	JTOS(ps.utime)
	JTOS(ps.stime)
	JTOS(ps.cutime)
	JTOS(ps.cstime)
	
	fclose(fd);
	return(&ps);
}


proc_mem_t      *get_proc_mem()
{
        static	proc_mem_t	mm;
        FILE    	*fp;
        unsigned        m_tot, m_use, m_free;
        unsigned        s_tot, s_use, s_free;

        
        if ((fp = fopen("/proc/meminfo", "r")) == NULL)
                return(NULL);

        fscanf(fp, "%*[^\n]%*c");       /* lose the text header */;
        fscanf(fp, "%*s %u %u %u %*[^\n]%*c", &m_tot, &m_use, &m_free);
        fscanf(fp, "%*s %u %u %u %*[^\n]%*c", &s_tot, &s_use, &s_free);
 
        mm.total = m_tot + s_tot;
        mm.used = m_use + s_use;
        mm.free = m_free + s_free;

	fclose(fp);
        return(&mm);
}

void
dep_initialize()
{
	char	*id = "dep_initialize";

	pagesize = getpagesize();

	if ((pdir = opendir(procfs)) == NULL) {
		log_err(errno, id, "opendir");
		return;
	}

	proc_get_btime();

	return;
}

void
dep_cleanup()
{
	char	*id = "dep_cleanup";

	log_record(PBSEVENT_SYSTEM, 0, id, "dependent cleanup");
	if (pdir) {
		closedir(pdir);
		pdir = NULL;
	}
}

/*
**	Don't need any periodic procsessing.
*/
void
end_proc()
{
	return;
}


/*
 * Internal size decoding routine.
 *
 *	Accepts a resource pointer and a pointer to the unsigned long integer
 *	to receive the decoded value.  It returns a PBS error code, and the
 *	decoded value in the unsigned long integer.
 *
 *		sizeof(word) = sizeof(int)
 */

static int getsize(pres, ret)
    resource		*pres;
    unsigned long	*ret;
{
	unsigned long	value;

	if (pres->rs_value.at_type != ATR_TYPE_SIZE)
		return (PBSE_ATTRTYPE);
	value = pres->rs_value.at_val.at_size.atsv_num;
	if (pres->rs_value.at_val.at_size.atsv_units ==
	    ATR_SV_WORDSZ) {
		if (value > ULONG_MAX / sizeof(int))
			return (PBSE_BADATVAL);
		value *= sizeof(int);
	}
	if (value > ULONG_MAX >>
	    pres->rs_value.at_val.at_size.atsv_shift)
	        return (PBSE_BADATVAL);
	*ret = value << pres->rs_value.at_val.at_size.atsv_shift;

	return (PBSE_NONE);
}

/*
 * Internal time decoding routine.
 *
 *	Accepts a resource pointer and a pointer to the unsigned long integer
 *	to receive the decoded value.  It returns a PBS error code, and the
 *	decoded value of time in seconds in the unsigned long integer.
 */

static int gettime(pres, ret)
    resource		*pres;
    unsigned long	*ret;
{

	if (pres->rs_value.at_type != ATR_TYPE_LONG)
		return (PBSE_ATTRTYPE);
	if (pres->rs_value.at_val.at_long < 0)
	        return (PBSE_BADATVAL);
	*ret = pres->rs_value.at_val.at_long;

	return (PBSE_NONE);
}

static
int
injob(pjob, sid)
    job		*pjob;
    pid_t	sid;
{
	task    *ptask;

	for (ptask = (task *)GET_NEXT(pjob->ji_tasks);
			ptask;
			ptask = (task *)GET_NEXT(ptask->ti_jobtask)) {
		if (ptask->ti_qs.ti_sid <= 1)
			continue;
		if (ptask->ti_qs.ti_sid == sid)
			return TRUE;
	}
	return FALSE;
}

/*
 * Internal session cpu time decoding routine.
 *
 *	Accepts a job pointer.  Returns the sum of all cpu time
 *	consumed for all tasks executed by the job, in seconds,
 *	adjusted by cputfactor.
 */
static unsigned long cput_sum(pjob)
    job		*pjob;
{
	char			*id = "cput_ses";
	int			fd;
	struct dirent		*dent;
	ulong			cputime;
	int			nps = 0;
	proc_stat_t		*ps;

	cputime = 0.0;
	rewinddir(pdir);
	while ((dent = readdir(pdir)) != NULL) {
		if (!isdigit(dent->d_name[0]))
			continue;

		if ((ps = get_proc_stat(atoi(dent->d_name))) == NULL)  {
			if (errno != ENOENT) {
				sprintf(log_buffer,
					"%s: get_proc_stat", dent->d_name);
				log_err(errno, id, log_buffer);
			}
			continue;
		}

		if (!injob(pjob, ps->session))
			continue;

		nps++;
		cputime += (ps->utime + ps->stime + ps->cutime + ps->cstime);
		DBPRT(("%s: ses %d pid %d cputime %d\n",
				id, ps->session, ps->pid, cputime));
	}

	if (nps == 0)
		pjob->ji_flags |= MOM_NO_PROC;

	return ((unsigned long)((double)cputime * cputfactor));
}

/*
 * Return TRUE if any process in the job is over limit for cputime usage.
 */
static int overcpu_proc(pjob, limit)
    job			*pjob;
    unsigned long	limit;
{
	char		*id = "overcpu_proc";
	ulong		memsize;
	int		fd;
	struct dirent	*dent;
	ulong		cputime;
	proc_stat_t	*ps;

	memsize = 0;
	rewinddir(pdir);
	while ((dent = readdir(pdir)) != NULL) {
		if (!isdigit(dent->d_name[0]))
			continue;

		if ((ps = get_proc_stat(atoi(dent->d_name))) == NULL)  {
			if (errno != ENOENT) {
				sprintf(log_buffer,
					"%s: get_proc_stat", dent->d_name);
				log_err(errno, id, log_buffer);
			}
			continue;
		}

		if (!injob(pjob, ps->session))
			continue;

		cputime = (ulong)((double)(ps->cutime + ps->cstime)*cputfactor);
		if (cputime > limit)
			return (TRUE);
        }

	return (FALSE);
}

/*
 * Internal session memory usage function.
 *
 *	Returns the total number of bytes of address
 *	space consumed by all current processes within the job.
 */
static unsigned long mem_sum(pjob)
    job			*pjob;
{
	char			*id="mem_sum";
	struct dirent		*dent;
	char			procname[100];
	int			num, i;
	unsigned long		segadd;
	proc_stat_t		*ps;

	segadd = 0;
	rewinddir(pdir);

	while ((dent = readdir(pdir)) != NULL) {
		if (!isdigit(dent->d_name[0]))
			continue;

		if ((ps = get_proc_stat(atoi(dent->d_name))) == NULL) {
			if (errno != ENOENT) {
				sprintf(log_buffer,
					"%s: get_proc_stat", dent->d_name);
				log_err(errno, id, log_buffer);
			}
			continue;
		}

		if (!injob(pjob, ps->session))
			continue;
		segadd += ps->vsize;
	}

	return (segadd);
}

/*
 * Internal session workingset size function.
 */
static unsigned long resi_sum(pjob)
    job			*pjob;
{
	char		*id="resi_sum";
	ulong		resisize;
	struct dirent	*dent;
	proc_stat_t	*ps;

	resisize = 0;
	rewinddir(pdir);
	while ((dent = readdir(pdir)) != NULL) {
		if (!isdigit(dent->d_name[0]))
			continue;

		if ((ps = get_proc_stat(atoi(dent->d_name))) == NULL) {
			if (errno != ENOENT) {
				sprintf(log_buffer,
					"%s: get_proc_stat", dent->d_name);
				log_err(errno, id, log_buffer);
			}
			continue;
		}

		if (!injob(pjob, ps->session))
                        continue;

                resisize += ps->rss * pagesize;
        }

	return (resisize);
}

/*
 * Return TRUE if any process in the job is over limit for memory usage.
 */
static int overmem_proc(pjob, limit)
    job			*pjob;
    unsigned long	limit;
{
	char		*id = "overmem_proc";
	ulong		memsize;
	struct dirent	*dent;
	proc_stat_t	*ps;

	memsize = 0;
	rewinddir(pdir);
	while ((dent = readdir(pdir)) != NULL) {
		if (!isdigit(dent->d_name[0]))
			continue;

		if ((ps = get_proc_stat(atoi(dent->d_name))) == NULL) {
			if (errno != ENOENT) {
				sprintf(log_buffer,
					"%s: get_proc_stat", dent->d_name);
				log_err(errno, id, log_buffer);
			}
			continue;
		}

		if (!injob(pjob, ps->session))
			continue;
                if (ps->vsize > limit)
			return (TRUE);
        }

	return (FALSE);
}

extern char *msg_momsetlim;

/*
 * Internal error routine
 */
int error(string, value)
    char	*string;
    int		value;
{
	int		i = 0;
	char		*message;

	assert(string != NULL);
	assert(*string != '\0');
	assert(value > PBSE_);			/* minimum PBS error number */
	assert(value <= PBSE_NOSYNCMSTR);	/* maximum PBS error number */
	assert(pbs_err_to_txt[i].err_no != 0);

	do {
		if (pbs_err_to_txt[i].err_no == value)
			break;
	} while (pbs_err_to_txt[++i].err_no != 0);

	assert(pbs_err_to_txt[i].err_txt != NULL);
	message = *pbs_err_to_txt[i].err_txt;
	assert(message != NULL);
	assert(*message != '\0');
	(void)fprintf(stderr, msg_momsetlim, string, message);
	(void)fflush(stderr);

	return (value);
}

/*
 * Establish system-enforced limits for the job.
 *
 *	Run through the resource list, checking the values for all items
 *	we recognize.
 *
 *	If set_mode is SET_LIMIT_SET, then also set hard limits for the
 *	system enforced limits (not-polled).
 *	If anything goes wrong with the process, return a PBS error code
 *	and print a message on standard error.  A zero-length resource list
 *	is not an error.
 *
 *	If set_mode is SET_LIMIT_SET the entry conditions are:
 *	    1.	MOM has already forked, and we are called from the child.
 *	    2.	The child is still running as root.
 *	    3.  Standard error is open to the user's file.
 *
 *	If set_mode is SET_LIMIT_ALTER, we are beening called to modify
 *	existing limits.  Cannot alter those set by setrlimit (kernel)
 *	because we are the wrong process.  
 */
int mom_set_limits(pjob, set_mode)
    job			*pjob;
    int			 set_mode;	/* SET_LIMIT_SET or SET_LIMIT_ALTER */
{
	char		*id = "mom_set_limits";
	char		*pname;
	int		retval;
	unsigned long	value;	/* place in which to build resource value */
	resource	*pres;
       	struct rlimit	reslim;
	unsigned long	mem_limit  = 0;

	DBPRT(("%s: entered\n", id))
	assert(pjob != NULL);
	assert(pjob->ji_wattr[(int)JOB_ATR_resource].at_type == ATR_TYPE_RESC);
	pres = (resource *)
	    GET_NEXT(pjob->ji_wattr[(int)JOB_ATR_resource].at_val.at_list);

/*
 * Cycle through all the resource specifications,
 * setting limits appropriately.
 */

	while (pres != NULL) {
		assert(pres->rs_defin != NULL);
		pname = pres->rs_defin->rs_name;
		assert(pname != NULL);
		assert(*pname != '\0');

		if (strcmp(pname, "cput") == 0) {
			/* cpu time - check, if less than pcput use it */
			retval = gettime(pres, &value);
			if (retval != PBSE_NONE)
			        return (error(pname, retval));
		} else if (strcmp(pname, "pcput") == 0) {
			/* process cpu time - set */
			retval = gettime(pres, &value);
			if (retval != PBSE_NONE)
			        return (error(pname, retval));
			reslim.rlim_cur = reslim.rlim_max = 
				(unsigned long)((double)value / cputfactor);
			if (setrlimit(RLIMIT_CPU, &reslim) < 0)
	        		return (error("RLIMIT_CPU", PBSE_SYSTEM));
		} else if (strcmp(pname, "file") == 0) {	/* set */
			if (set_mode == SET_LIMIT_SET)  {
			    retval = getsize(pres, &value);
			    if (retval != PBSE_NONE)
			        return (error(pname, retval));
			    if (value > INT_MAX)
			        return (error(pname, PBSE_BADATVAL));
			    reslim.rlim_cur = reslim.rlim_max = value;
			    if (setrlimit(RLIMIT_FSIZE, &reslim) < 0)
			        return (error(pname, PBSE_SYSTEM));
			}
		} else if (strcmp(pname, "vmem") == 0) {	/* check */
			retval = getsize(pres, &value);
			if (retval != PBSE_NONE)
			        return (error(pname, retval));
			if ((mem_limit == 0) || (value < mem_limit))
				mem_limit = value;
		} else if (strcmp(pname, "pvmem") == 0) {	/* set */
			if (set_mode == SET_LIMIT_SET)  {
			    retval = getsize(pres, &value);
			    if (retval != PBSE_NONE)
			        return (error(pname, retval));
			    if (value > INT_MAX)
			        return (error(pname, PBSE_BADATVAL));
			if ((mem_limit == 0) || (value < mem_limit))
				mem_limit = value;
			}
		} else if (strcmp(pname, "mem") == 0) {		/* ignore */
		} else if (strcmp(pname, "pmem") == 0) {	/* set */
			if (set_mode == SET_LIMIT_SET)  {
			    retval = getsize(pres, &value);
			    if (retval != PBSE_NONE)
			        return (error(pname, retval));
			    reslim.rlim_cur = reslim.rlim_max = value;
			    if (setrlimit(RLIMIT_RSS, &reslim) < 0)
	        		return (error("RLIMIT_RSS", PBSE_SYSTEM));
			}
		} else if (strcmp(pname, "walltime") == 0) {	/* Check */
			retval = gettime(pres, &value);
			if (retval != PBSE_NONE)
			        return (error(pname, retval));
		} else if (strcmp(pname, "nice") == 0) {	/* set nice */
			if (set_mode == SET_LIMIT_SET)  {
			    errno = 0;
			    if ((nice((int)pres->rs_value.at_val.at_long) == -1)
				&& (errno != 0))
				return (error(pname, PBSE_BADATVAL));
			}
		} else if ((pres->rs_defin->rs_flags & ATR_DFLAG_RMOMIG) == 0)
			/* don't recognize and not marked as ignore by mom */
			return (error(pname, PBSE_UNKRESC));
		pres = (resource *)GET_NEXT(pres->rs_link);
	}
	if (set_mode == SET_LIMIT_SET)  {
	    /* if either of vmem or pvmem was given, set sys limit to lesser */
	    if (mem_limit != 0) {
		reslim.rlim_cur = reslim.rlim_max = mem_limit;
		if (setrlimit(RLIMIT_DATA, &reslim) < 0)
	        	return (error("RLIMIT_DATA", PBSE_SYSTEM));
		if (setrlimit(RLIMIT_STACK, &reslim) < 0)
	        	return (error("RLIMIT_STACK", PBSE_SYSTEM));
	    }
	}
	return (PBSE_NONE);
}
/*
 * State whether MOM main loop has to poll this job to determine if some
 * limits are being exceeded.
 *
 *	Sets flag TRUE if polling is necessary, FALSE otherwise.  Actual
 *	polling is done using the mom_over_limit machine-dependent function.
 */

int mom_do_poll(pjob)
    job			*pjob;
{
	char		*id = "mom_do_poll";
	char		*pname;
	resource	*pres;

	DBPRT(("%s: entered\n", id))
	assert(pjob != NULL);
	assert(pjob->ji_wattr[(int)JOB_ATR_resource].at_type == ATR_TYPE_RESC);
	pres = (resource *)
	    GET_NEXT(pjob->ji_wattr[(int)JOB_ATR_resource].at_val.at_list);

	while (pres != NULL) {
		assert(pres->rs_defin != NULL);
		pname = pres->rs_defin->rs_name;
		assert(pname != NULL);
		assert(*pname != '\0');

		if (strcmp(pname, "walltime") == 0 ||
		    strcmp(pname, "cput") == 0 ||
		    strcmp(pname, "pcput") == 0 ||
		    strcmp(pname, "pvmem") == 0 ||
		    strcmp(pname, "vmem") == 0)
			return (TRUE);
		pres = (resource *)GET_NEXT(pres->rs_link);
	}

	return (FALSE);
}

/*
 * Setup for polling.
 *
 *	Open kernel device and get namelist info.
 */
int mom_open_poll()
{
	char		*id = "mom_open_poll";

	DBPRT(("%s: entered\n", id))
	pagesize = getpagesize();

	return (PBSE_NONE);
}

/*
 * Declare start of polling loop.
 *
 */

int mom_get_sample()
{
	return (PBSE_NONE);
}

/*
 * Measure job resource usage and compare with its limits.
 *
 *	If it has exceeded any well-formed polled limit return TRUE.
 *	Otherwise, return FALSE.
 */

int mom_over_limit(pjob)
    job			*pjob;
{
	char		*id = "mom_over_limit";
	char		*pname;
	int		retval;
	unsigned long	value, num;
	resource	*pres;

	assert(pjob != NULL);
	assert(pjob->ji_wattr[(int)JOB_ATR_resource].at_type == ATR_TYPE_RESC);
	pres = (resource *)
	    GET_NEXT(pjob->ji_wattr[(int)JOB_ATR_resource].at_val.at_list);

	for ( ; pres != NULL; pres = (resource *)GET_NEXT(pres->rs_link)) {
		assert(pres->rs_defin != NULL);
		pname = pres->rs_defin->rs_name;
		assert(pname != NULL);
		assert(*pname != '\0');
		if (strcmp(pname, "cput") == 0) {
			retval = gettime(pres, &value);
			if (retval != PBSE_NONE)
				continue;
			if ((num = cput_sum(pjob)) > value) {
				sprintf(log_buffer,
					"cput %lu exceeded limit %lu",
					num, value);
				return (TRUE);
			}
		} else if (strcmp(pname, "pcput") == 0) {
			retval = gettime(pres, &value);
			if (retval != PBSE_NONE)
				continue;
			if (overcpu_proc(pjob, value)) {
				sprintf(log_buffer,
					"pcput exceeded limit %lu",
					value);
				return (TRUE);
			}
		} else if (strcmp(pname, "vmem") == 0) {
			retval = getsize(pres, &value);
			if (retval != PBSE_NONE)
				continue;
			if ((num = mem_sum(pjob)) > value) {
				sprintf(log_buffer,
					"vmem %lu exceeded limit %lu",
					num, value);
				return (TRUE);
			}
		} else if (strcmp(pname, "pvmem") == 0) {
			retval = getsize(pres, &value);
			if (retval != PBSE_NONE)
				continue;
			if (overmem_proc(pjob, value)) {
				sprintf(log_buffer,
					"pvmem exceeded limit %lu",
					value);
				return (TRUE);
			}
		} else if (strcmp(pname, "walltime") == 0) {
			retval = gettime(pres, &value);
			if (retval != PBSE_NONE)
				continue;
			num = (unsigned long)((double)(time_now - pjob->ji_qs.ji_stime) * wallfactor);
			if (num > value) {
				sprintf(log_buffer,
					"walltime %d exceeded limit %d",
					num, value);
				return (TRUE);
			}
		}
	}

	return (FALSE);
}

/*
 * Update the job attribute for resources used.
 *
 *	The first time this is called for a job, set up resource entries for
 *	each resource that can be reported for this machine.  Fill in the
 *	correct values.  Return an error code.
 */
int mom_set_use(pjob)
    job			*pjob;
{
	char			*id = "mom_set_use";
	resource		*pres;
	attribute		*at;
	resource_def		*rd;
	unsigned long		*lp, lnum;

	assert(pjob != NULL);
	at = &pjob->ji_wattr[(int)JOB_ATR_resc_used];
	assert(at->at_type == ATR_TYPE_RESC);

	at->at_flags |= ATR_VFLAG_MODIFY;
	if ((at->at_flags & ATR_VFLAG_SET) == 0) {
		at->at_flags |= ATR_VFLAG_SET;

		rd = find_resc_def(svr_resc_def, "cput", svr_resc_size);
		assert(rd != NULL);
		pres = add_resource_entry(at, rd);
		pres->rs_value.at_flags |= ATR_VFLAG_SET;
		pres->rs_value.at_type = ATR_TYPE_LONG;

		rd = find_resc_def(svr_resc_def, "vmem", svr_resc_size);
		assert(rd != NULL);
		pres = add_resource_entry(at, rd);
		pres->rs_value.at_flags |= ATR_VFLAG_SET;
		pres->rs_value.at_type = ATR_TYPE_SIZE;
		pres->rs_value.at_val.at_size.atsv_shift = 10;	/* KB */
		pres->rs_value.at_val.at_size.atsv_units = ATR_SV_BYTESZ;

		rd = find_resc_def(svr_resc_def, "walltime", svr_resc_size);
		assert(rd != NULL);
		pres = add_resource_entry(at, rd);
		pres->rs_value.at_flags |= ATR_VFLAG_SET;
		pres->rs_value.at_type = ATR_TYPE_LONG;

		rd = find_resc_def(svr_resc_def, "mem", svr_resc_size);
		assert(rd != NULL);
		pres = add_resource_entry(at, rd);
		pres->rs_value.at_flags |= ATR_VFLAG_SET;
		pres->rs_value.at_type = ATR_TYPE_SIZE;
		pres->rs_value.at_val.at_size.atsv_shift = 10;	/* KB */
		pres->rs_value.at_val.at_size.atsv_units = ATR_SV_BYTESZ;
	}

	rd = find_resc_def(svr_resc_def, "cput", svr_resc_size);
	assert(rd != NULL);
	pres = find_resc_entry(at, rd);
	assert(pres != NULL);
	lp = (unsigned long *)&pres->rs_value.at_val.at_long;
	lnum = cput_sum(pjob);
	*lp = MAX(*lp, lnum);

	rd = find_resc_def(svr_resc_def, "vmem", svr_resc_size);
	assert(rd != NULL);
	pres = find_resc_entry(at, rd);
	assert(pres != NULL);
	lp = &pres->rs_value.at_val.at_size.atsv_num;
	lnum = (mem_sum(pjob) + 1023) >> 10;	/* as KB */
	*lp = MAX(*lp, lnum);

	rd = find_resc_def(svr_resc_def, "walltime", svr_resc_size);
	assert(rd != NULL);
	pres = find_resc_entry(at, rd);
	assert(pres != NULL);
	pres->rs_value.at_val.at_long = (long)((double)(time_now - pjob->ji_qs.ji_stime) * wallfactor);

	rd = find_resc_def(svr_resc_def, "mem", svr_resc_size);
	assert(rd != NULL);
	pres = find_resc_entry(at, rd);
	assert(pres != NULL);
	lp = &pres->rs_value.at_val.at_size.atsv_num;
	lnum = (resi_sum(pjob) + 1023) >> 10;	/* as KB */
	*lp = MAX(*lp, lnum);

	return (PBSE_NONE);
}

/*
 *	Kill a task session.
 *	Call with the task pointer and a signal number.
 */
int kill_task(ptask, sig)
    task	*ptask;
    int		sig;
{
	char		*id = "kill_task";
	int		ct = 0;
	struct dirent	*dent;
	proc_stat_t	*ps;
	int		sesid;


	sesid = ptask->ti_qs.ti_sid;
	if (sesid <= 1)
		return 0;

	rewinddir(pdir);
	while ((dent = readdir(pdir)) != NULL) {
		if (!isdigit(dent->d_name[0]))
			continue;

		if ((ps = get_proc_stat(atoi(dent->d_name))) == NULL) {
			if (errno != ENOENT) {
				sprintf(log_buffer,
					"%s: get_proc_stat", dent->d_name);
				log_err(errno, id, log_buffer);
			}
			continue;
		}

		if (sesid == ps->session) {
			(void)kill(ps->pid, sig);
			++ct;
		}
	}
	return ct;
}

/*
 * Clean up everything related to polling.
 */
int mom_close_poll()
{
	char	*id = "mom_close_poll";

	DBPRT(("%s: entered\n", id))
	if (pdir) {
		if (closedir(pdir) != 0) {
			log_err(errno, id, "closedir");
			return (PBSE_SYSTEM);
		}
		pdir = NULL;
	}

	return (PBSE_NONE);
}

/*
 * mom_does_chkpnt - return 1 if mom supports checkpoint
 *			    0 if not
 */

int mom_does_chkpnt()
{
	return (0);
}

/*
 * Checkpoint the job.
 *
 *	If abort is true, kill it too.
 */

int mach_checkpoint(ptask, file, abort)
    task	*ptask;
    char	*file;
    int		abort;
{
       	return (-1);
}

/*
 * Restart the job from the checkpoint file.
 *
 *	Return -1 on error or sid if okay.
 */

long mach_restart(ptask, file)
    task	*ptask;
    char	*file;
{
	return (-1);
}

#define	dsecs(val) ( (double)(val) )

char	*
cput_job(jobid)
pid_t	jobid;
{
	char			*id = "cput_job";
	int			found = 0;
	struct dirent		*dent;
	double			cputime, addtime;
	proc_stat_t		*ps;

	cputime = 0.0;
	rewinddir(pdir);
	while ((dent = readdir(pdir)) != NULL) {
		if (!isdigit(dent->d_name[0]))
			continue;

		if ((ps = get_proc_stat(atoi(dent->d_name))) == NULL) {
			if (errno != ENOENT) {
				sprintf(log_buffer,
					"%s: get_proc_stat", dent->d_name);
				log_err(errno, id, log_buffer);
			}
			continue;
		}

		if (jobid != ps->session)
			continue;

		found = 1;
		addtime = dsecs(ps->cutime) + dsecs(ps->cstime);

		cputime += addtime;
		DBPRT(("%s: total %.2f pid %d %.2f\n", id, cputime,
				ps->pid, addtime))

	}
	if (found) {
		sprintf(ret_string, "%.2f", cputime * cputfactor);
		return ret_string;
	}

	rm_errno = RM_ERR_EXIST;
	return NULL;
}

char	*
cput_proc(pid)
pid_t	pid;
{
	char			*id = "cput_pid";
	double			cputime;
	proc_stat_t		*ps;

	cputime = 0.0;

	if ((ps = get_proc_stat(pid)) == NULL) {
		if (errno != ENOENT) {
			sprintf(log_buffer, "%d: get_proc_stat", pid);
			log_err(errno, id, log_buffer);
		}
		rm_errno = RM_ERR_SYSTEM;
		return NULL;
	}
	cputime = dsecs(ps->utime) + dsecs(ps->stime);

	sprintf(ret_string, "%.2f", cputime * cputfactor);
	return ret_string;
}


char	*
cput(attrib)
struct	rm_attribute	*attrib;
{
	char			*id = "cput";
	int			value;

	if (attrib == NULL) {
		log_err(-1, id, no_parm);
		rm_errno = RM_ERR_NOPARAM;
		return NULL;
	}
	if ((value = atoi(attrib->a_value)) == 0) {
		sprintf(log_buffer, "bad param: %s", attrib->a_value);
		log_err(-1, id, log_buffer);
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}
	if (momgetattr(NULL)) {
		log_err(-1, id, extra_parm);
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}

	if (strcmp(attrib->a_qualifier, "session") == 0)
		return (cput_job((pid_t)value));
	else if (strcmp(attrib->a_qualifier, "proc") == 0)
		return (cput_proc((pid_t)value));
	else {
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}
}

char	*
mem_job(sid)
pid_t	sid;
{
	static	char	id[] = "mem_job";
	ulong			memsize;
	struct dirent		*dent;
	char			procname[100];
	int			num, i;
	proc_stat_t		*ps;

	memsize = 0;
	rewinddir(pdir);

	while ((dent = readdir(pdir)) != NULL) {
		if (!isdigit(dent->d_name[0]))
			continue;

		if ((ps = get_proc_stat(atoi(dent->d_name))) == NULL) {
			if (errno != ENOENT) {
				sprintf(log_buffer,
					"%s: get_proc_stat", dent->d_name);
				log_err(errno, id, log_buffer);
			}
			continue;
		}

		if (sid != ps->session)
			continue;
		memsize += ps->vsize;
	}

	if (memsize == 0) {
		rm_errno = RM_ERR_EXIST;
		return NULL;
	}
	else {
		sprintf(ret_string, "%lukb", memsize >> 10); /* KB */
		return ret_string;
	}
}

char	*
mem_proc(pid)
pid_t	pid;
{
	char			*id = "mem_proc";
	proc_stat_t		*ps;

	if ((ps = get_proc_stat(pid)) == NULL) {
		if (errno != ENOENT) {
			sprintf(log_buffer, "%d: get_proc_stat", pid);
			log_err(errno, id, log_buffer);
		}
		rm_errno = RM_ERR_SYSTEM;
		return NULL;
	}

	sprintf(ret_string, "%lukb", (ulong)ps->vsize >> 10); /* KB */
	return ret_string;
}

char	*
mem(attrib)
struct	rm_attribute	*attrib;
{
	char			*id = "mem";
	int			value;

	if (attrib == NULL) {
		log_err(-1, id, no_parm);
		rm_errno = RM_ERR_NOPARAM;
		return NULL;
	}
	if ((value = atoi(attrib->a_value)) == 0) {
		sprintf(log_buffer, "bad param: %s", attrib->a_value);
		log_err(-1, id, log_buffer);
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}
	if (momgetattr(NULL)) {
		log_err(-1, id, extra_parm);
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}

	if (strcmp(attrib->a_qualifier, "session") == 0)
		return (mem_job((pid_t)value));
	else if (strcmp(attrib->a_qualifier, "proc") == 0)
		return (mem_proc((pid_t)value));
	else {
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}
}

static char	*
resi_job(jobid)
pid_t	jobid;
{
	char			*id = "resi_job";
	ulong			resisize;
	int			fd;
	int			found = 0;
	char			procname[100];
	struct dirent		*dent;
	proc_stat_t		*ps;

	resisize = 0;
	rewinddir(pdir);
	while ((dent = readdir(pdir)) != NULL) {
		if (!isdigit(dent->d_name[0]))
			continue;

		if ((ps = get_proc_stat(atoi(dent->d_name))) == NULL) {
			if (errno != ENOENT) {
				sprintf(log_buffer,
					"%s: get_proc_stat", dent->d_name);
				log_err(errno, id, log_buffer);
			}
			continue;
		}

		if (jobid != ps->session)
			continue;

		found = 1;
		resisize += ps->rss;
	}
	if (found) {
		/* in KB */
		sprintf(ret_string, "%lukb",(resisize * (ulong)pagesize) >> 10);
		return ret_string;
	}

	rm_errno = RM_ERR_EXIST;
	return NULL;
}

static char	*
resi_proc(pid)
pid_t	pid;
{
	char			*id = "resi_proc";
	proc_stat_t		*ps;


	if ((ps = get_proc_stat(pid)) == NULL) {
		if (errno != ENOENT) {
			sprintf(log_buffer,
				"%d: get_proc_stat(PIOCPSINFO)", pid);
			log_err(errno, id, log_buffer);
		}
		rm_errno = RM_ERR_SYSTEM;
		return NULL;
	}

	/* in KB */
	sprintf(ret_string, "%lukb", ((ulong)ps->rss * (ulong)pagesize) >> 10);
	return ret_string;
}

static char	*
resi(attrib)
struct	rm_attribute	*attrib;
{
	char			*id = "resi";
	int			value;

	if (attrib == NULL) {
		log_err(-1, id, no_parm);
		rm_errno = RM_ERR_NOPARAM;
		return NULL;
	}
	if ((value = atoi(attrib->a_value)) == 0) {
		sprintf(log_buffer, "bad param: %s", attrib->a_value);
		log_err(-1, id, log_buffer);
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}
	if (momgetattr(NULL)) {
		log_err(-1, id, extra_parm);
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}

	if (strcmp(attrib->a_qualifier, "session") == 0)
		return (resi_job((pid_t)value));
	else if (strcmp(attrib->a_qualifier, "proc") == 0)
		return (resi_proc((pid_t)value));
	else {
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}
}

char	*
sessions(attrib)
struct	rm_attribute	*attrib;
{
	char			*id = "sessions";
	int			j;
	struct dirent		*dent;
	proc_stat_t		*ps;
	char			*fmt;
	int			njids = 0;
	pid_t			*jids, *hold;
	static		int	maxjid = 200;
	register	pid_t	jobid;


	if (attrib) {
		log_err(-1, id, extra_parm);
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}
	if ((jids = (pid_t *)calloc(maxjid, sizeof(pid_t))) == NULL) {
		log_err(errno, id, "no memory");
		rm_errno = RM_ERR_SYSTEM;
		return NULL;
	}

	/*
	** Search for members of session
	*/
	rewinddir(pdir);
	while ((dent = readdir(pdir)) != NULL) {
		if (!isdigit(dent->d_name[0]))
			continue;

		if ((ps = get_proc_stat(atoi(dent->d_name))) == NULL) {
			if (errno != ENOENT) {
				sprintf(log_buffer,
					"%s: get_proc_stat", dent->d_name);
				log_err(errno, id, log_buffer);
			}
			continue;
		}

		if (ps->uid == 0)
			continue;
		if ((jobid = ps->session) == 0)
			continue;
		DBPRT(("%s[%d]: pid %d sid %d\n",
		       id, njids, ps->pid, jobid))

		for (j=0; j<njids; j++) {
			if (jids[j] == jobid)
				break;
		}
		if (j == njids) {		/* not found */
			if (njids == maxjid) {	/* need more space */
				maxjid += 100;
				hold = (pid_t *)realloc(jids, maxjid);
				if (hold == NULL) {
					log_err(errno, id, "realloc");
					rm_errno = RM_ERR_SYSTEM;
					free(jids);
					return NULL;
				}
				jids = hold;
			}
			jids[njids++] = jobid;	/* add jobid to list */
		}
	}

	fmt = ret_string;
	for (j=0; j<njids; j++) {
		checkret(&fmt, 100);
		sprintf(fmt, " %d", (int)jids[j]);
		fmt += strlen(fmt);
	}
	free(jids);
	return ret_string;
}

char	*
nsessions(attrib)
struct	rm_attribute	*attrib;
{
	char	*result, *ch;
	int	num = 0;

	if ((result = sessions(attrib)) == NULL)
		return result;

	for (ch=result; *ch; ch++) {
		if (*ch == ' ')		/* count blanks */
			num++;
	}
	sprintf(ret_string, "%d", num);
	return ret_string;
}

char	*
pids(attrib)
struct	rm_attribute	*attrib;
{
	char		*id = "pids";
	pid_t		jobid;
	struct	dirent	*dent;
	proc_stat_t	*ps;
	int		i;
	char		*fmt;
	int		num_pids;

	if (attrib == NULL) {
		log_err(-1, id, no_parm);
		rm_errno = RM_ERR_NOPARAM;
		return NULL;
	}
	if ((jobid = (pid_t)atoi(attrib->a_value)) == 0) {
		sprintf(log_buffer, "bad param: %s", attrib->a_value);
		log_err(-1, id, log_buffer);
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}
	if (momgetattr(NULL)) {
		log_err(-1, id, extra_parm);
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}

	if (strcmp(attrib->a_qualifier, "session") != 0) {
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}

	/*
	** Search for members of session
	*/
	rewinddir(pdir);
	fmt = ret_string;
	while ((dent = readdir(pdir)) != NULL) {
		if (!isdigit(dent->d_name[0]))
			continue;

		if ((ps = get_proc_stat(atoi(dent->d_name))) == NULL) {
			if (errno != ENOENT) {
				sprintf(log_buffer,
					"%s: get_proc_stat", dent->d_name);
				log_err(errno, id, log_buffer);
			}
			continue;
		}

		DBPRT(("%s[%d]: pid: %s sid %d\n",
		       id, num_pids, dent->d_name, ps->session))
		if (jobid != ps->session)
			continue;

		sprintf(fmt, "%s ", dent->d_name);
		fmt += strlen(fmt);
		num_pids++;
	}
	if (num_pids == 0) {
		rm_errno = RM_ERR_EXIST;
		return NULL;
	}
	return ret_string;
}

char	*
nusers(attrib)
struct	rm_attribute	*attrib;
{
	char			*id = "nusers";
	int			j;
	struct dirent		*dent;
	proc_stat_t		*ps;
	int			nuids = 0;
	uid_t			*uids, *hold;
	static		int	maxuid = 200;
	register	uid_t	uid;

	if (attrib) {
		log_err(-1, id, extra_parm);
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}
	if ((uids = (uid_t *)calloc(maxuid, sizeof(uid_t))) == NULL) {
		log_err(errno, id, "no memory");
		rm_errno = RM_ERR_SYSTEM;
		return NULL;
	}

	rewinddir(pdir);
	while ((dent = readdir(pdir)) != NULL) {
		if (!isdigit(dent->d_name[0]))
			continue;

		if ((ps = get_proc_stat(atoi(dent->d_name))) == NULL) {
			if (errno != ENOENT) {
				sprintf(log_buffer,
					"%s: get_proc_stat", dent->d_name);
				log_err(errno, id, log_buffer);
			}
			continue;
		}
		if ((uid = ps->uid) == 0)
			continue;

		DBPRT(("%s[%d]: pid %d uid %d\n",
		       id, nuids, ps->pid, uid))

		for (j=0; j<nuids; j++) {
			if (uids[j] == uid)
				break;
		}
		if (j == nuids) {		/* not found */
			if (nuids == maxuid) {	/* need more space */
				maxuid += 100;
				hold = (uid_t *)realloc(uids, maxuid);
				if (hold == NULL) {
					log_err(errno, id, "realloc");
					rm_errno = RM_ERR_SYSTEM;
					free(uids);
					return NULL;
				}
				uids = hold;
			}
			uids[nuids++] = uid;	/* add uid to list */
		}
	}

	sprintf(ret_string, "%d", nuids);
	free(uids);
	return ret_string;
}

static char	*
totmem(attrib)
struct	rm_attribute	*attrib;
{
	char	*id = "totmem";
	proc_mem_t	*mm;

	if (attrib) {
		log_err(-1, id, extra_parm);
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}

	if ((mm = get_proc_mem()) == NULL) {
		log_err(errno, id, "get_proc_mem");
		rm_errno = RM_ERR_SYSTEM;
		return NULL;
	}
	DBPRT(("%s: total mem=%u\n", id, mm->total))
	sprintf(ret_string, "%lukb", (ulong)mm->total >> 10); /* KB */
	return ret_string;
}

static char	*
availmem(attrib)
struct	rm_attribute	*attrib;
{
	char	*id = "availmem";
	proc_mem_t	*mm;

	if (attrib) {
		log_err(-1, id, extra_parm);
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}

	if ((mm = get_proc_mem()) == NULL) {
		log_err(errno, id, "get_proc_mem");
		rm_errno = RM_ERR_SYSTEM;
		return NULL;
	}
	DBPRT(("%s: free mem=%u\n", id, mm->free))
	sprintf(ret_string, "%lukb", (ulong)mm->free >> 10); /* KB */
	return ret_string;
}


static char	*
ncpus(attrib)
struct	rm_attribute	*attrib;
{
	char		*id = "ncpus", label[128];
	FILE		*fp;
	int		procs;
	unsigned int	pmem;

	if (attrib) {
		log_err(-1, id, extra_parm);
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}

	if ((fp = fopen("/proc/cpuinfo", "r")) == NULL)
		return(NULL);

	procs = 0;
	while (!feof(fp))  {
		fscanf(fp, "%s %*[^\n]%*c", label);
		if (strcmp("processor", label) == 0)
			procs++;
	}
			
	sprintf(ret_string, "%d", procs);
	fclose(fp);
	return ret_string;
}

static char	*
physmem(attrib)
struct	rm_attribute	*attrib;
{
	char		*id = "physmem";
	struct	stat	buf;


	if (attrib) {
		log_err(-1, id, extra_parm);
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}
	if (stat("/proc/kcore", &buf))  {
		log_err(-1, id, extra_parm);
		rm_errno = RM_ERR_SYSTEM;
		return NULL;
	}  else  {
		/* length of /proc/kcore == physical memory + 4K */
		sprintf(ret_string, "%lukb",
			((ulong)buf.st_size - (ulong)4096) >> 10); /* KB */
		return ret_string;
	}
}

char	*
size_fs(param)
char	*param;
{
	char		*id = "size_fs";
	FILE		*mf;
	struct	mntent	*mp;
	struct	statfs	fsbuf;

	if (param[0] != '/') {
		sprintf(log_buffer, "%s: not full path filesystem name: %s\n",
			id, param);
		log_err(-1, id, log_buffer);
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}
	if (statfs(param, &fsbuf) == -1) {
		log_err(errno, id, "statfs");
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}
	sprintf(ret_string, "%lukb",
		(ulong)(((double)fsbuf.f_bsize * (double)fsbuf.f_bfree) / 1024.0)); /* KB */
	return ret_string;
}

char	*
size_file(param)
char	*param;
{
	char		*id = "size_file";
	struct	stat	sbuf;

	if (param[0] != '/') {
		sprintf(log_buffer, "%s: not full path filesystem name: %s\n",
			id, param);
		log_err(-1, id, log_buffer);
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}

	if (stat(param, &sbuf) == -1) {
		log_err(errno, id, "stat");
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}

	sprintf(ret_string, "%ukb", sbuf.st_size >> 10); /* KB */
	return ret_string;
}

char	*
size(attrib)
struct	rm_attribute	*attrib;
{
	char	*id = "size";
	char	*param;

	if (attrib == NULL) {
		log_err(-1, id, no_parm);
		rm_errno = RM_ERR_NOPARAM;
		return NULL;
	}
	if (momgetattr(NULL)) {
		log_err(-1, id, extra_parm);
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}

	param = attrib->a_value;
	if (strcmp(attrib->a_qualifier, "file") == 0)
		return (size_file(param));
	else if (strcmp(attrib->a_qualifier, "fs") == 0)
		return (size_fs(param));
	else {
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}
}

time_t	maxtm;

void
setmax(dev)
char	*dev;
{
	struct	stat	sb;

	if (stat(dev, &sb) == -1)
		return;
	if (maxtm < sb.st_atime)
		maxtm = sb.st_atime;

	return;
}

char	*
idletime(attrib)
struct	rm_attribute	*attrib;
{
	char	*id = "idletime";
	DIR	*dp;
	struct	dirent	*de;
	char	ttyname[50];
	time_t	curtm;

	if (attrib) {
		log_err(-1, id, extra_parm);
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}

	if ((dp=opendir("/dev")) == NULL) {
		log_err(errno, id, "opendir /dev");
		rm_errno = RM_ERR_SYSTEM;
		return NULL;
	}

	maxtm = 0;
	curtm = time(NULL);

	setmax("/dev/mouse");
	while ((de=readdir(dp)) != NULL) {
		if (maxtm >= curtm)
			break;
		if (strncmp(de->d_name, "tty", 3))
			continue;
		sprintf(ttyname, "/dev/%s", de->d_name);
		setmax(ttyname);
	}
	closedir(dp);

	sprintf(ret_string, "%d", MAX(0, curtm - maxtm));
	return ret_string;
}



static char	*
walltime(attrib)
struct	rm_attribute	*attrib;
{
	char			*id = "walltime";
	int			value, job, found = 0;
	time_t			now, start;
	proc_stat_t		*ps;
	struct dirent		*dent;

	if (attrib == NULL) {
		log_err(-1, id, no_parm);
		rm_errno = RM_ERR_NOPARAM;
		return NULL;
	}
	if ((value = atoi(attrib->a_value)) == 0) {
		sprintf(log_buffer, "bad param: %s", attrib->a_value);
		log_err(-1, id, log_buffer);
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}
	if (momgetattr(NULL)) {
		log_err(-1, id, extra_parm);
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}

	if (strcmp(attrib->a_qualifier, "proc") == 0)
		job = 0;
	else if (strcmp(attrib->a_qualifier, "session") == 0)
		job = 1;
	else {
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}

	if ((now = time(NULL)) <= 0) {
		log_err(errno, id, "time");
		rm_errno = RM_ERR_SYSTEM;
		return NULL;
	}

	start = now;
	rewinddir(pdir);
	while ((dent = readdir(pdir)) != NULL) {
		if (!isdigit(dent->d_name[0]))
			continue;

		if ((ps = get_proc_stat(atoi(dent->d_name))) == NULL) {
			if (errno != ENOENT) {
				sprintf(log_buffer,
					"%s: get_proc_stat", dent->d_name);
				log_err(errno, id, log_buffer);
			}
			continue;
		}
		if (job) {
			if (value != ps->session)
				continue;
		}
		else {
			if (value != ps->pid)
				continue;
		}

		found = 1;
		start = MIN(start, ps->start_time);
	}

	if (found) {
		sprintf(ret_string, "%ld", (long)((double)(now - start) * wallfactor));
		return ret_string;
	}

	rm_errno = RM_ERR_EXIST;
	return NULL;
}

int
get_la(rv)
	double	*rv;
{
	FILE	*fp;
	char	*id = "get_la";
	float	load;

        if ((fp = fopen("/proc/loadavg", "r")) == NULL)
                return(rm_errno = RM_ERR_SYSTEM);

	if (fscanf(fp, "%f", &load) != 1) {
		log_err(errno, id, "fscanf of load in /proc/loadavg");
		(void) fclose(fp);
		return (rm_errno = RM_ERR_SYSTEM);
	}

	*rv = (double)load;
	(void) fclose(fp);
	return 0;
}

u_long
gracetime(secs)
u_long	secs;
{
	time_t	now = time((time_t *)NULL);

	if (secs > now)		/* time is in the future */
		return (secs - now);
	else
		return 0;
}

static char	*
quota(attrib)
struct	rm_attribute	*attrib;
{
	char	*id = "quota";
	int			type;
	dev_t			dirdev;
	uid_t			uid;
	struct	stat		sb;
	struct	mntent		*me;
	struct	dqblk		qi;
	FILE			*m;
	struct	passwd		*pw;
	static	char		*type_array[] = {
		"harddata",
		"softdata",
		"currdata",
		"hardfile",
		"softfile",
		"currfile",
		"timedata",
		"timefile",
	};
	enum	type_name {
		harddata,
		softdata,
		currdata,
		hardfile,
		softfile,
		currfile,
		timedata,
		timefile,
		type_end
	};

	if (attrib == NULL) {
		log_err(-1, id, no_parm);
		rm_errno = RM_ERR_NOPARAM;
		return NULL;
	}
	if (strcmp(attrib->a_qualifier, "type")) {
		sprintf(log_buffer, "unknown qualifier %s",
			attrib->a_qualifier);
		log_err(-1, id, log_buffer);
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}		

	for (type=0; type<type_end; type++) {
		if (strcmp(attrib->a_value, type_array[type]) == 0)
			break;
	}
	if (type == type_end) {		/* check to see if command is legal */
		sprintf(log_buffer, "bad param: %s=%s",
			attrib->a_qualifier, attrib->a_value);
		log_err(-1, id, log_buffer);
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}

	if ((attrib = momgetattr(NULL)) == NULL) {
		log_err(-1, id, no_parm);
		rm_errno = RM_ERR_NOPARAM;
		return NULL;
	}
	if (strcmp(attrib->a_qualifier, "dir") != 0) {
		sprintf(log_buffer, "bad param: %s=%s",
			attrib->a_qualifier, attrib->a_value);
		log_err(-1, id, log_buffer);
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}
	if (attrib->a_value[0] != '/') {	/* must be absolute path */
		sprintf(log_buffer,
			"not an absolute path: %s", attrib->a_value);
		log_err(-1, id, log_buffer);
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}
	if (stat(attrib->a_value, &sb) == -1) {
		sprintf(log_buffer, "stat: %s", attrib->a_value);
		log_err(errno, id, log_buffer);
		rm_errno = RM_ERR_EXIST;
		return NULL;
	}
	dirdev = sb.st_dev;
	DBPRT(("dir has devnum %d\n", dirdev))

	if ((m = setmntent(MOUNTED, "r")) == NULL) {
		log_err(errno, id, "setmntent");
		rm_errno = RM_ERR_SYSTEM;
                return NULL;
        }
	while ((me = getmntent(m)) != NULL) {
		if (strcmp(me->mnt_type, MNTTYPE_IGNORE) == 0)
			continue;
		if (stat(me->mnt_dir, &sb) == -1) {
			sprintf(log_buffer, "stat: %s", me->mnt_dir);
			log_err(errno, id, log_buffer);
			continue;
		}
		DBPRT(("%s\t%s\t%d\n", me->mnt_fsname, me->mnt_dir, sb.st_dev))
		if (sb.st_dev == dirdev)
			break;
	}
	endmntent(m);
	if (me == NULL)	{
		sprintf(log_buffer,
			"filesystem %s not found", attrib->a_value);
		log_err(-1, id, log_buffer);
		rm_errno = RM_ERR_EXIST;
		return NULL;
	}

#if defined(MNTOPT_NOQUOTA)
	if (hasmntopt(me, MNTOPT_NOQUOTA) != NULL) {
		sprintf(log_buffer,
			"no quotas on filesystem %s", me->mnt_dir);
		log_err(-1, id, log_buffer);
		rm_errno = RM_ERR_EXIST;
		return NULL;
	}
#endif /* MNTOPT_NOQUOTA */

	if ((attrib = momgetattr(NULL)) == NULL) {
		log_err(-1, id, no_parm);
		rm_errno = RM_ERR_NOPARAM;
		return NULL;
	}
	if (strcmp(attrib->a_qualifier, "user") != 0) {
		sprintf(log_buffer, "bad param: %s=%s",
			attrib->a_qualifier, attrib->a_value);
		log_err(-1, id, log_buffer);
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}
	if ((uid = (uid_t)atoi(attrib->a_value)) == 0) {
		if ((pw = getpwnam(attrib->a_value)) == NULL) {
			sprintf(log_buffer,
				"user not found: %s", attrib->a_value);
			log_err(-1, id, log_buffer);
			rm_errno = RM_ERR_EXIST;
			return NULL;
		}
		uid = pw->pw_uid;
	}

	if (syscall(SYS_quotactl, QCMD(Q_GETQUOTA,USRQUOTA),
			me->mnt_fsname, uid, (caddr_t)&qi) == -1) {
		log_err(errno, id, "quotactl");
		rm_errno = RM_ERR_SYSTEM;
		return NULL;
	}

	/* sizes in KB */
	switch (type) {
	case harddata:
		sprintf(ret_string, "%ukb", qi.dqb_bhardlimit >> 10);
		break;
	case softdata:
		sprintf(ret_string, "%ukb", qi.dqb_bsoftlimit >> 10);
		break;
	case currdata:
		sprintf(ret_string, "%ukb", qi.dqb_curblocks >> 10);
		break;
	case hardfile:
		sprintf(ret_string, "%u", qi.dqb_ihardlimit);
		break;
	case softfile:
		sprintf(ret_string, "%u", qi.dqb_isoftlimit);
		break;
	case currfile:
		sprintf(ret_string, "%u", qi.dqb_curinodes);
		break;
	case timedata:
		sprintf(ret_string, "%u", gracetime((u_long)qi.dqb_btime));
		break;
	case timefile:
		sprintf(ret_string, "%u", gracetime((u_long)qi.dqb_itime));
		break;
	}

	return ret_string;
}
