/*
This program is confidential and proprietary to MRJ Technology Solutions
and may not be reproduced, published or disclosed to others without
written authorization from MRJ.

Copyright (c) 1998 MRJ Technology Solutions, Inc. All Rights Reserved.

This program is derived from prior work some of which was originally
written as a joint project between the Numerical Aerospace Simulation
(NAS) Systems Division of NASA Ames Research Center and the National
Energy Research Supercomputer Center (NERSC) of Lawrence Livermore
National Laboratory.
*/
#include <pbs_config.h>   /* the master config generated by configure */

#include <sys/types.h>
#include <sys/wait.h>
#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <pwd.h>
#include <assert.h>
#include "portability.h"
#include "libpbs.h"
#include "list_link.h"
#include "log.h"
#include "server_limits.h"
#include "attribute.h"
#include "resource.h"
#include "job.h"
#include "mom_mach.h"
#include "mom_func.h"

static char ident[] = "@(#) $RCSfile: mom_start.c,v $ $Revision: 2.0 $";

/* Global Variables */

extern int	 exiting_tasks;
extern char	 mom_host[];
extern list_head svr_alljobs;
extern int	 termin_child;

/* note, all variables are defined in /etc/environment */

char *obtain_vnames[NUM_LCL_ENV_VAR] = {
	(char *)0
};

/* Private variables */

/*
 * set_job - set up a new job session
 * 	Set session id and whatever else is required on this machine
 *	to create a new job.
 *
 *      Return: session/job id or if error:
 *		-1 - if setsid() fails
 *		-2 - if other, message in log_buffer
 */

int set_job(pjob, sjr)
	job *pjob;
	struct startjob_rtn *sjr;
{
	return (sjr->sj_session = setsid());
}

/*
**	set_globid - set the global id for a machine type.
*/
void
set_globid(pjob, sjr)
	job *pjob;
	struct startjob_rtn *sjr;
{
	extern	char	noglobid[];

	if (pjob->ji_globid == NULL)
		pjob->ji_globid = strdup(noglobid);
}

/*
 * set_mach_vars - setup machine dependent environment variables
 */

int
set_mach_vars(pjob, vtab)
	job   		 *pjob;    /* pointer to the job	*/
	struct var_table *vtab;    /* pointer to variable table */
{
	FILE			*efile;
	char			buf[256];	
        resource		*pres;
	extern	char		*path_home;

	efile = fopen("/etc/environment", "r");
	if (efile != (FILE *)0) {

	    while (fgets(buf, sizeof(buf)-1, efile) ) {
		if ((buf[0] != '#') && (buf[0] != ' ') && (buf[0] != '\n')) {
		    buf[strlen(buf)-1] = '\0';
		    bld_env_variables(vtab, buf, (char *)0);
		}
	    }
	    fclose(efile);
	}

        assert(pjob != NULL);
        assert(pjob->ji_wattr[(int)JOB_ATR_resource].at_type == ATR_TYPE_RESC);
        pres = find_resc_entry(&pjob->ji_wattr[(int)JOB_ATR_resource],
                               find_resc_def(svr_resc_def, "nodect",
                                             svr_resc_size) );
	if (pres != NULL) {
		sprintf(buf, "%s/aux/%s", path_home, pjob->ji_qs.ji_jobid);
		bld_env_variables(vtab, "MP_HOSTFILE", buf);
		sprintf(buf, "%d", pres->rs_value.at_val.at_long);
		bld_env_variables(vtab, "MP_PROCS", buf);
	}
	return 0;
}

char *set_shell(pjob, pwdp)
	job	      *pjob;
	struct passwd *pwdp;
{
	char *cp;
	int   i;
	char *shell;
	struct array_strings *vstrs;
	/*
	 * find which shell to use, one specified or the login shell
	 */

	shell = pwdp->pw_shell;
	if ((pjob->ji_wattr[(int)JOB_ATR_shell].at_flags & ATR_VFLAG_SET) &&
	    (vstrs = pjob->ji_wattr[(int)JOB_ATR_shell].at_val.at_arst)) {
		for (i = 0; i < vstrs->as_usedptr; ++i) {
			cp = strchr(vstrs->as_string[i], '@');
			if (cp) {
				if (!strncmp(mom_host, cp+1, strlen(cp+1))) {
					*cp = '\0';	/* host name matches */
					shell = vstrs->as_string[i];
					break;
				}
			} else {
				shell = vstrs->as_string[i];	/* wildcard */
			}
		}
	}
	return (shell);
}

/* 
 * scan_for_terminated - scan the list of runnings jobs for one whose
 *	session id matched that of a terminated child pid.  Mark that
 *	job as Exiting.
 */

void scan_for_terminated()
{
	static	char	id[] = "scan_for_terminated";
	int		exiteval;
	pid_t		pid;
	job		*pjob;
	task		*ptask;
	int		statloc;

	/* update the latest intelligence about the running jobs;         */
	/* must be done before we reap the zombies, else we lose the info */

	termin_child = 0;

	if (mom_get_sample() == PBSE_NONE) {
		pjob = (job *)GET_NEXT(svr_alljobs);
		while (pjob) {
			mom_set_use(pjob);
			pjob = (job *)GET_NEXT(pjob->ji_alljobs);
		}
	}

	/* Now figure out which job(s) have terminated (are zombies) */

	while ((pid = waitpid(-1, &statloc, WNOHANG)) != 0) {
		if ((pid == -1) && (errno != EINTR))
			break;
		
		pjob = (job *)GET_NEXT(svr_alljobs);
		while (pjob) {
			ptask = (task *)GET_NEXT(pjob->ji_tasks);
			while (ptask) {
				if (ptask->ti_qs.ti_sid == pid)
					break;
				ptask = (task *)GET_NEXT(ptask->ti_jobtask);
			}
			if (ptask != NULL)
				break;
			pjob = (job *)GET_NEXT(pjob->ji_alljobs);
		}
		if (pjob == NULL) {
			DBPRT(("%s: pid %d not a task\n", id, pid))
			continue;
		}
		/*
		** We found task within the job which has exited.
		*/
		if (WIFEXITED(statloc))
			exiteval = WEXITSTATUS(statloc);
		else if (WIFSIGNALED(statloc))
			exiteval = WTERMSIG(statloc) + 10000;
		else 
			exiteval = 1;
		DBPRT(("%s: task %d pid %d exit value %d\n", id,
				ptask->ti_qs.ti_task, pid, exiteval))
		kill_task(ptask, SIGKILL);
		ptask->ti_qs.ti_exitstat = exiteval;
		ptask->ti_qs.ti_status = TI_STATE_EXITED;
		task_save(ptask);
		sprintf(log_buffer, "task %d terminated", ptask->ti_qs.ti_task);
		LOG_EVENT(PBSEVENT_DEBUG, PBS_EVENTCLASS_JOB,
				pjob->ji_qs.ji_jobid, log_buffer);
		exiting_tasks = 1;
	}
}



/*
 * creat the master pty, this particular
 * piece of code depends on multiplexor /dev/ptc
 */

int open_master(rtn_name)
	char **rtn_name;	/* RETURN name of slave pts */
{
	int   ptc;

	if ((ptc = open("/dev/ptc", O_RDWR | O_NOCTTY, 0)) < 0) {
		return (-1);
	}

	*rtn_name  = ttyname(ptc);	/* name of slave side */
	return (ptc);
}



/*
 * struct sig_tbl = map of signal names to numbers,
 * see req_signal() in ../requests.c
 */

struct sig_tbl sig_tbl[] = {
	{ "NULL", 0 },
	{ "HUP", SIGHUP },
	{ "INT", SIGINT },
	{ "QUIT", SIGQUIT },
	{ "ILL", SIGILL },
	{ "TRAP", SIGTRAP },
	{ "ABRT", SIGABRT },
	{ "EMT", SIGEMT },
	{ "FPE", SIGFPE },
	{ "KILL", SIGKILL },
	{ "BUS", SIGBUS },
	{ "SEGV", SIGSEGV },
	{ "SYS", SIGSYS },
	{ "PIPE", SIGPIPE },
	{ "ALRM", SIGALRM },
	{ "TERM", SIGTERM },
	{ "URG", SIGURG },
	{ "STOP", SIGSTOP },
	{ "TSTP", SIGTSTP },
	{ "CONT", SIGCONT },
	{ "CHLD", SIGCHLD },
	{ "TTIN", SIGTTIN },
	{ "TTOU", SIGTTOU },
	{ "IO", SIGIO },
	{ "XCPU", SIGXCPU },
	{ "XFSZ", SIGXFSZ },
	{ "MSG", SIGMSG },
	{ "WINCH", SIGWINCH },
	{ "PWR", SIGPWR },
	{ "USR1", SIGUSR1 },
	{ "USR2", SIGUSR2 },
	{ "PROF", SIGPROF },
	{ "DANGER", SIGDANGER },
	{ "VTALRM", SIGVTALRM },
	{ "MIGRATE", SIGMIGRATE },
	{ "PRE", SIGPRE },
	{ "VIRT", SIGVIRT },
	{ "GRANT", SIGGRANT },
	{ "RETRACT", SIGRETRACT },
	{ "SOUND", SIGSOUND },
	{ "SAK", SIGSAK },
	{ (char *)0, -1 }
};
