/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/

#include <pbs_config.h>   /* the master config generated by configure */

#ifndef _POSIX_SOURCE
#define _POSIX_SOURCE 1
#endif  /* _POSIX_SOURCE */

/* System headers */
#include <stdio.h>
#include <ctype.h>
#include <malloc.h>
#include <string.h>
#include <setjmp.h>
#include <stdlib.h>

#include "Node.h"
#include "SymTabGlob.h"

static char ident[] = "@(#) $RCSfile: Node.c,v $ $Revision: 2.1 $";
/* Global variables */
FILE *NodeFpOut;

/* File Scope Variables */
static char *NodeName = "Node";

static char * NodeErrors[] =
{
        "0 no such error",
        "1 lexem ptr = NULL",
        "2 Np = NULL",
        "3 malloc problem",
        "4 str ptr = NULL",
        "5 can not access funDescr for non-fun",
        "6 error msg delimeter",
        "7 NodeFpOut is NULL"
};
static int NodeMaxErrors = 8;

static int NodeDF = 0;

/*
 * These functions support the Node class for the linked list
 */
Np NodeNew(lexem, typ, lin, leve, funFla)
char 	*lexem;
int 	typ;
int 	lin;
int 	leve;
int 	funFla;
{
	Np nx;

	NodeCondPrint("NodeNew");
	
	if (lexem == NULL) 
		NodeErr(1);

	nx = (Np) malloc(sizeof(struct Node));
	if (nx == NULL) 
		NodeErr(3);

	strcpy(nx->lexeme, lexem);
	nx->type = typ;
	nx->lineDef = lin;
	nx->level = leve;
	nx->funFlag = funFla;

	nx->funDescr.paramCnt = 0;

	nx->funDescr.paramPtr = NULL;

	nx->rptr = NULL;
	return(nx);
}

 
void NodeInit(nx, lexem, typ, lin, leve, funFla)
Np 	nx;
char 	*lexem;
int 	typ;
int 	lin;
int 	leve;
int 	funFla;
{
	NodeCondPrint("NodeInit");

	if (nx == NULL) 
		NodeErr(2);

	if (lexem == NULL) 
		NodeErr(1);

	strcpy(nx->lexeme, lexem);
	nx->type = typ;
	nx->lineDef = lin;
	nx->level = leve;
	nx->funFlag = funFla;

	nx->funDescr.paramCnt = 0;

	nx->funDescr.paramPtr = NULL;

	nx->rptr = NULL;

}



void NodePrint(nx)
Np 	nx;
{
	if(NodeFpOut == NULL)
		NodeErr(7);

	fprintf(NodeFpOut, "NodePrint\n");

	if(nx == NULL) 
		NodeErr(2);

	fprintf(NodeFpOut, "lexeme=%s type=%s, lineDef=%d level=%d funFlag=%d\n", nx->lexeme, SymTypes[nx->type], nx->lineDef, nx->level, nx->funFlag);

	

	/*optional*/
	/*
	fprintf(NodeFpOut, "nx=%x, rptr=%x\n", nx, nx->rptr);
	fprintf(NodeFpOut, "nx->funDescr.paramPtr=%x\n", nx->funDescr.paramPtr);
	*/

	if(nx->funFlag != NO)
		NodeFunDescrPrint(nx);
}


void NodePrint2(nx)
Np	nx;
{
	if(NodeFpOut == NULL)
		NodeErr(7);

	fprintf(NodeFpOut, "NodePrint2\n");

	if(nx == NULL) 
		NodeErr(2);

	fprintf(NodeFpOut, "lexeme=%s type=%s, lineDef=%d level=%d funFlag=%d\n", nx->lexeme, SymTypes[nx->type], nx->lineDef, nx->level, nx->funFlag);

	
	/*optional*/
	/*
	fprintf(NodeFpOut, "nx=%x, rptr=%x\n", nx, nx->rptr);
	fprintf(NodeFpOut, "nx->funDescr.paramPtr=%x\n", nx->funDescr.paramPtr);
	*/



}


void NodeFunDescrPrint(nx)
Np	nx;
{
	Np paramPt;


	if(NodeFpOut == NULL)
		NodeErr(7);

	fprintf(NodeFpOut, "NodeFunDescrPrint\n");

	if(nx == NULL) 
		NodeErr(2);


	fprintf(NodeFpOut, "paramCnt=%d\n", nx->funDescr.paramCnt);

	paramPt = nx->funDescr.paramPtr;

	while(paramPt != NULL)
	{
		NodePrint2(paramPt);
		paramPt = paramPt->funDescr.paramPtr;
	}

	fprintf(NodeFpOut, "\nend of funDescr\n");

}


Np NodeFunDescrFindByLexeme(nx, lexem)
Np	nx;
char	*lexem;
{
	Np paramPt;

	if(NodeFpOut == NULL)
		NodeErr(7);

	if( NodeDF == 1 )
		fprintf(NodeFpOut, "NodeFunDescrFindByLexeme\n");

	if(nx == NULL) 
		NodeErr(2);


	paramPt = nx->funDescr.paramPtr;

	while(paramPt != NULL)
	{
        	if (strcmp(paramPt->lexeme, lexem) == 0)
        	{
                	return(paramPt);
        	}
		paramPt = paramPt->funDescr.paramPtr;
	}
	return(NULL);
}

int NodeCmp(nx, lexem)
Np 	nx;
char 	*lexem;
{
	NodeCondPrint("NodeCmp");

	if (nx == NULL) 
		NodeErr(2);

	if (lexem == NULL) 
		NodeErr(1);

	return(strcmp(nx->lexeme, lexem) == 0);
}


void NodeErr(e)
int	e;
{

	if(NodeFpOut == NULL)
		NodeErr(7);

	fprintf(NodeFpOut, "NodeErr\n");

	if (e >= NodeMaxErrors)
		e = 0;
	fprintf(NodeFpOut, "rs: %s\n",  NodeErrors[e]);
	exit(1);
}



void NodeCondPrint(str)
char	*str;
{
	if(NodeFpOut == NULL)
		NodeErr(7);

	if (str == NULL)
		NodeErr(4);

	if (NodeDF == 1)
	{
		fprintf(NodeFpOut, "%s\t", NodeName);
		fprintf(NodeFpOut, "%s\n", str);
	}
}

void NodePutDF(df)
int	df;
{
	NodeDF = df;
}


char * NodeGetLexeme(nxp)
Np	nxp;
{
	NodeCondPrint("NodeGetLexeme");

	if (nxp == NULL) 
		NodeErr(2);

	return(nxp->lexeme);
}
	

int NodeGetType(nxp)
Np	nxp;
{
	NodeCondPrint("NodeGetType");

	if (nxp == NULL) 
		NodeErr(2);

	return(nxp->type);
}
	

int NodeGetLineDef(nxp)
Np	nxp;
{
	NodeCondPrint("NodeGetLineDef");

	if (nxp == NULL) 
		NodeErr(2);

	return(nxp->lineDef);
}
	


int NodeGetLevel(nxp)
Np	nxp;
{
	NodeCondPrint("NodeGetLevel");

	if (nxp == NULL) 
		NodeErr(2);

	return(nxp->level);
}
	

int NodeGetFunFlag(nxp)
Np	nxp;
{
	NodeCondPrint("NodeGetFunFlag");

	if (nxp == NULL) 
		NodeErr(2);

	return(nxp->funFlag);
}
	

int NodeGetParamCnt(nxp)
Np	nxp;
{
	NodeCondPrint("NodeGetParamCnt");

	if (nxp == NULL) 
		NodeErr(2);

	if (nxp->funFlag == NO)
		NodeErr(5);

	return(nxp->funDescr.paramCnt);

}
	

Np NodeGetParamPtr(nxp)
Np	nxp;
{
	NodeCondPrint("NodeGetParamPtr");

	if (nxp == NULL) 
		NodeErr(2);

	if (nxp->funFlag == NO)
		NodeErr(5);

	return(nxp->funDescr.paramPtr);

}
	


Np NodeGetRptr(nxp)
Np	nxp;
{
	NodeCondPrint("NodeGetRptr");

	if (nxp == NULL) 
		NodeErr(2);

	return(nxp->rptr);
}
	

void NodePutLexeme(nxp, lexem)
Np 	nxp;
char 	*lexem;
{
	NodeCondPrint("NodePutLexeme");

	if (nxp == NULL) 
		NodeErr(2);

	strcpy(nxp->lexeme, lexem);
}
	
void NodePutType(nxp, typ)
Np 	nxp;
int 	typ;
{
	NodeCondPrint("NodePutType");

	if (nxp == NULL) 
		NodeErr(2);

	nxp->type = typ;
}
	

void NodePutLineDef(nxp, lin)
Np 	nxp;
int 	lin;
{
	NodeCondPrint("NodePutlineDef");

	if (nxp == NULL) 
		NodeErr(2);

	nxp->lineDef = lin;
}
	


void NodePutLevel(nxp, leve)
Np 	nxp;
int 	leve;
{
	NodeCondPrint("NodePutLevel");

	if (nxp == NULL) 
		NodeErr(2);

	nxp->level = leve;
}
	

void NodePutFunFlag(nxp, funFla)
Np 	nxp;
int 	funFla;
{
	NodeCondPrint("NodePutFunFlag");

	if (nxp == NULL) 
		NodeErr(2);

	nxp->funFlag = funFla;
}
	

void NodePutParamCnt(nxp, paramCn)
Np 	nxp;
int 	paramCn;
{
	NodeCondPrint("NodePutParamCnt");

	if (nxp == NULL) 
		NodeErr(2);

	if (nxp->funFlag == NO)
		NodeErr(5);

	nxp->funDescr.paramCnt = paramCn;

}
	

void NodePutParamPtr(nxp, paramPt)
Np 	nxp;
Np 	paramPt;
{
	NodeCondPrint("NodePutParamPtr");

	if (nxp == NULL) 
		NodeErr(2);

	if (nxp->funFlag == NO)
		NodeErr(5);

	nxp->funDescr.paramPtr = paramPt;

}
	

void NodePutRptr(nxp, rp)
Np 	nxp;
Np 	rp;
{
	NodeCondPrint("NodePutRptr");

	if (nxp == NULL) 
		NodeErr(2);

	nxp->rptr = rp;
}
	

void NodeParamCntIncr(nxp)
Np nxp;
{
	NodeCondPrint("NodeParamCntIncr");

	if (nxp == NULL) 
		NodeErr(2);

	if (nxp->funFlag == NO)
		NodeErr(5);

	nxp->funDescr.paramCnt++;

}
	

void NodeParamCntDecr(nxp)
Np nxp;
{
	NodeCondPrint("NodeParamCntDecr");

	if (nxp == NULL) 
		NodeErr(2);

	if (nxp->funFlag == NO)
		NodeErr(5);

	nxp->funDescr.paramCnt--;

}
