/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/

#include <pbs_config.h>   /* the master config generated by configure */

#ifndef _POSIX_SOURCE
#define _POSIX_SOURCE 1
#endif  /* _POSIX_SOURCE */

/* System headers */
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <stdarg.h>
/* local headers */
#include "portability.h"
#include "af_resmom.h"
/*
#include "af_config.h"
*/

#include "log.h"
/* Macros */
/* File Scope Variables */
static char ident[] = "@(#) $RCSfile: af_resmom.c,v $ $Revision: 2.1 $";

/* External Variables */
extern int errno;

/* External Functions */
extern int
openrm  A_((char * host, unsigned int port));

extern int
closerm  A_((int));

extern int
addreq          A_((int, char *));

extern char *
getreq          A_((int));

extern int
activereq       A_((void));

extern void 
fullresp       A_((int flag));

/* Structures and Unions */
/* NONE */
/* Signal catching functions */
/* NONE */
/* Functions */

char *ResMomInetAddrGet ( mom )
ResMom *mom;
{
      assert( mom != NULL );

      return mom->inetAddr;
}

int ResMomPortNumberGet ( mom )
ResMom *mom;
{
      assert( mom != NULL );

      return mom->portNumber;
}

int ResMomConnectFdGet ( mom )
ResMom *mom;
{
      assert( mom != NULL );

      return mom->connectFd;
}

void ResMomInetAddrPut (mom, mom_name)
ResMom *mom;
char   *mom_name;
{
       assert( mom != NULL );

       dynamic_strcpy(&mom->inetAddr, mom_name);
}

void ResMomPortNumberPut (mom, port)
ResMom *mom;
int    port;
{
       assert( mom != NULL );

       mom->portNumber = port;
}

void ResMomConnectFdPut (mom, fd)
ResMom *mom;
int    fd;
{
       assert( mom != NULL );

       mom->connectFd = fd;
}


/* ResMomOpen: attempts to connect to mom. 
   RETURNS a value >= 0 if connection was a success; < 0 otherwise */
int ResMomOpen( mom )
ResMom *mom;
{
       int    connect;

       static char id[] = "ResMomOpen";

       assert( ResMomInetAddrGet(mom) != NULLSTR && \
	       ResMomPortNumberGet(mom) >= 0 );

       connect = openrm( ResMomInetAddrGet(mom), ResMomPortNumberGet(mom) );

#ifdef DEBUG
	printf("%s: openrm( %s, %d )\n", id, ResMomInetAddrGet(mom),
						ResMomPortNumberGet(mom));
#endif

/*     We only want a full response */
       fullresp(TRUE);

       if ( connect < 0 ) {
	  (void)sprintf(log_buffer, "openrm (%s, %d)", ResMomInetAddrGet(mom), 
						   ResMomPortNumberGet(mom));
          log_err(errno, id, log_buffer);
          perror(log_buffer);
       }

       ResMomConnectFdPut(mom, connect);
       return (connect);
}

/* ResMomClose: attempts to disconnect from mom. 
   RETURNS: 0 if closerm() was succesful; non-zero otherwise */
int ResMomClose( mom )
ResMom *mom;
{
        static char id[] = "ResMomClose";
	int  ret = -1;

	assert( mom != NULL );

        if( ResMomConnectFdGet(mom) < 0 ) {
	  (void)sprintf(log_buffer, "ResMom %s not connected!",
		   ResMomInetAddrGet(mom) ? ResMomInetAddrGet(mom):"null");
          log_err(-1, id, log_buffer);
        } else {
          ret = closerm(ResMomConnectFdGet(mom));

#ifdef DEBUG
          printf("%s: closerm( %d ): %d\n", id, ResMomConnectFdGet(mom), ret);
#endif

          if ( ret != 0 ) {
	  (void)sprintf(log_buffer, "closerm of %s",
		   ResMomInetAddrGet(mom) ? ResMomInetAddrGet(mom):"null");
            log_err(errno, id, log_buffer);
            perror(log_buffer);
          }
        }
        
	return (ret);
}

/* ResMomWrite: send a 'buffer' query to Resmom. 
   RETURNS: 1 if succesful; 0 otherwise. */
int ResMomWrite( mom, buffer )
ResMom *mom;
char   *buffer;
{
        static char id[] = "ResMomWrite";
	int ret;

	assert( mom != NULL );

	if (buffer == NULLSTR ) {
          log_err(-1, id, "Nothing to query\n");
	  return 0;
	}

        if( ResMomConnectFdGet(mom) < 0 ) {
	  (void)sprintf(log_buffer, "ResMom %s not connected!",
		   ResMomInetAddrGet(mom) ? ResMomInetAddrGet(mom):"null");
          log_err(-1, id, log_buffer);
	  return 0;
        }

#ifdef DEBUG
	printf("%s: addreq(%d, %s)\n", id, ResMomConnectFdGet(mom), buffer); 
#endif

        ret = addreq(ResMomConnectFdGet(mom), buffer );
	if ( ret != 0 ) {
	  (void)sprintf(log_buffer, "failed to add request to ResMom %s",
		   ResMomInetAddrGet(mom) ? ResMomInetAddrGet(mom):"null");
          log_err(-1, id, log_buffer);
	  return 0;
	}
	return 1;
}

/* ResMomRead: queries the ResMom for data. Returns  result as a string. */
/* This is a malloc-ed return so a call to varstrfree of the return pointer */
/* should be made */
char *ResMomRead( mom )
ResMom  *mom;
{
        static char id[] = "ResMomRead";
	void *ptr;	


	ptr = getreq(ResMomConnectFdGet(mom));
	varstrAdd(ptr, 0, mom);

#ifdef DEBUG
	printf("%s: getreq(%d): %s\n", id, ResMomConnectFdGet(mom), ptr);
#endif
	return((char *)ptr);
}

void ResMomPrint(mom)
ResMom	*mom;
{
	assert( mom != NULL );

	(void)printf("\t\t\t\tResMom Host = %s\n",
		ResMomInetAddrGet( mom ) ? ResMomInetAddrGet( mom ):"null");
	(void)printf("\t\t\t\t  ResMom Port Number = %d\n",
		ResMomPortNumberGet( mom ));
	(void)printf("\t\t\t\t  ResMom Connect Fd = %d\n",
		ResMomConnectFdGet( mom ));
}

void ResMomInit(mom)
ResMom *mom;
{
	assert( mom != NULL );

	mom->inetAddr = NULLSTR;
	ResMomPortNumberPut( mom, 0 );  
	ResMomConnectFdPut ( mom, -1 );  
}

void ResMomFree(mom)
ResMom *mom;
{
	if( mom == NULL )
		return;

	varstrFree(mom->inetAddr);
	varstrFreeByPptr(mom);
	mom->inetAddr = NULLSTR;
}
