#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* PBS header files */
#include "pbs_ifl.h"
#include "log.h"

/* Toolkit header files */
#include "toolkit.h"

/*
 * Parse a string like "foo@*.bar.com,bar@qux.baz.org,fred@pox.com" into a
 * linked list of UserAcl's.  Each element's user [and possibly host] field
 * points to an individually schd_strdup()'d string.
 */
UserAcl *
schd_create_useracl(char *useracl)
{
    char  *id = "schd_create_useracl";
    char  *useracl_copy, *user, *atsign;
    UserAcl *acl, *new, *acltail;

    /*
     * Copy the string.  This copy will be chopped up with '\0's to create
     * the strings pointed to by the array of UserAcl's pointed to by acl.
     */
    if ((useracl_copy = schd_strdup(useracl)) == NULL) {
	log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, id, 
	    "schd_strdup(useracl) failed");
	DBPRT(("schd_strdup(useracl) failed\n"));
	return (NULL);
    }

    acl     = NULL;
    acltail = NULL;

    user = strtok(useracl_copy, ",");
    while (user != NULL) {

	new = (UserAcl *)malloc(sizeof (UserAcl));
	if (new == NULL) {
	    log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, id, 
		"malloc(UserAcl) failed");
	    DBPRT(("malloc(UserAcl) failed\n"));
	    if (acl)
		schd_free_useracl(acl);
	    return (NULL);
	}

	/* 
	 * If a host string is given, change the '@' into a '\0' to terminate
	 * the user string (for the strncpy() below) and place a reference to
	 * a copy of the host string into the host pointer.
	 */

	new->host = NULL;
	if ((atsign = strchr(user, '@')) != NULL) {
	    *atsign = '\0';

	    /* Skip forward to the start of the remaining host string. */
	    atsign ++;

	    new->host = schd_strdup(atsign);
	    if (new->host == NULL) {
		log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, id, 
		    "schd_strdup(host) failed");
		DBPRT(("schd_strdup(host) failed\n"));
		if (acl)
		    schd_free_useracl(acl);
		free (new);
		return (NULL);
	    }
	}

	/*
	 * Copy the username into the static array in the UserAcl struct.
	 */
	strncpy(new->user, user, PBS_MAXUSER);

	/*
	 * Place the new ACL element on the tail of the list, or create it
	 * if this is the first element.
	 */
	if (acltail)
	    acltail->next = new;
	else
	    acl = new;
	acltail = new;
	acltail->next = NULL;

	/* Move on to the next user entry in the list. */
	user = strtok(NULL, ",");
    }

    /*
     * Free the storage used by the copy of the string that was strtok()'d.
     */
    free(useracl_copy);

    return (acl);
}

int
schd_free_useracl(UserAcl *acl)
{
    int      num = 0;
    UserAcl *aclptr, *aclnext;

    for (aclptr = acl; aclptr != NULL; aclptr = aclnext) {
	aclnext = aclptr->next;

	if (aclptr->host)
	    free(aclptr->host);
	free(aclptr);

	num ++;
    }

    return (num);
}

int
schd_useracl_okay(Job *job, Queue *queue, char *reason)
{
    /* char    *id = "schd_useracl_okay"; */
    UserAcl *ptr;
    char    *jobowner;

    /* No ACL?  Don't disallow this user. */
    if (!(queue->flags & QFLAGS_USER_ACL))
	return 1;

    jobowner = job->owner;

    for (ptr = queue->useracl; ptr != NULL; ptr = ptr->next) {
	if (!strcmp(jobowner, ptr->user)) {
	    return (1);
	}

	/* Note: we are ignoring the host for now. */
    }

    if (reason)
	sprintf(reason, "Denied by queue %s access control list", 
	    queue->qname);
    
    return (0);
}
