/* $Id: time_limits.c,v 1.1 1999/11/15 23:34:55 hender Exp $ */

#include <sys/types.h>
#include <stdio.h>
#include <string.h>
#include <time.h>

/* PBS header files */

#include "pbs_ifl.h"
#include "log.h"

/* Scheduler header files */

#include "toolkit.h"
#include "gblxvars.h"

/*
 * Determine whether a job will fit the primetime walltime limits.  This
 * means that the sum of time the job would run while it is primetime must
 * be no more than the appropriate prime-time walltime limit.
 */
/* ARGSUSED */
int 
schd_primetime_limits(Job *job, Queue *queue, time_t when, char *reason)
{
#ifdef DEBUGLOTS
    char   *id = "schd_primetime_limits";
#endif /* DEBUGLOTS */
    int     is_pt;
    int     secs_in_pt, secs_left;
    int     length;
#ifdef DEBUGLOTS
    char    buff[64];
#endif /* DEBUGLOTS */

    /* If not enforcing primetime, than no limits apply. */
    if (!(schd_ENFORCE_PRIME_TIME && schd_TimeNow >= schd_ENFORCE_PRIME_TIME))
	return (0);

    /* If this particular queue is ignoring primetime, no limits apply. */
    if (!queue->observe_pt)
	return (0);

    /* High-priority jobs do not have to meet primetime limits. */
    if (job->flags & JFLAGS_PRIORITY)
	return (0);

    /* If 'when' is 0, initialize it to the "current time". */
    if (when == 0)
	when = schd_TimeNow;

#ifdef DEBUGLOTS
    DBPRT(("%s: check primetime limits for job %s (%s)\n", id, job->jobid,
	schd_sec2val(job->walltime)));
#endif /* DEBUGLOTS */

    secs_in_pt = 0;
    secs_left  = job->walltime;

    while (secs_left > 0) {
	/* Is it prime-time at the time 'when' indicates? */
	is_pt = schd_prime_time(when);

	/* Find out how long the current period will last. */
	if (is_pt)
	    length = schd_secs_til_nonprime(when);
	else
	    length = schd_secs_til_prime(when);

	/* 
	 * If this is a prime-time period, account for the time the job
	 * will spend within it.  This is either the length of the period
	 * itself, or the time remaining (if it doesn't completely span 
	 * the prime-time period).
	 */
	if (is_pt)
	    secs_in_pt += (secs_left < length) ? secs_left : length;

	/* Account for the time spent in this period. */
	secs_left -= (secs_left < length) ? secs_left : length;

#ifdef DEBUGLOTS
	strcpy(buff, ctime(&when));
	buff[19] = '\0';
	DBPRT(("%s:    %sprime period at %s (for %s), %d in_pt %d remain\n", id,
	    is_pt ? "" : "non-", buff, schd_sec2val(length), secs_in_pt, 
	    secs_left));
#endif /* DEBUGLOTS */

	/* Push "time" forward to the beginning next period. */
	when += length;
    }

#ifdef DEBUGLOTS
    DBPRT(("%s: Job requests %d secs, %d of which are in prime time.\n",
	id, job->walltime, secs_in_pt));
#endif /* DEBUGLOTS */

    /* If there is no overlap into primetime, then the job is okay. */
    if (secs_in_pt == 0)
	return (0);

    /*
     * Check the appropriate time limit for this job (based on cpu count). 
     * Note that, if PT_SMALL_NODE_LIMIT is not set, the reason is different
     * from the small/large case.
     */
    if (schd_PT_SMALL_NODE_LIMIT > 0) {
	if (job->nodes <= schd_PT_SMALL_NODE_LIMIT) {
	    if (secs_in_pt > schd_PT_SMALL_WALLT_LIMIT) {
		if (reason)
		    (void)sprintf(reason, 
			"Exceeds primetime walltime limit (%s)",
			schd_sec2val(schd_PT_SMALL_WALLT_LIMIT));

		return (1);
	    }
	} else {
	    if (secs_in_pt > schd_PT_WALLT_LIMIT) {
		if (reason)
		    (void)sprintf(reason, 
			"Exceeds primetime walltime limit (%s)",
			schd_sec2val(schd_PT_WALLT_LIMIT));
		return (1);
	    }
	}
    } else {
	if (secs_in_pt > schd_PT_WALLT_LIMIT) {
	    if (reason)
		(void)sprintf(reason, 
		    "Exceeds primetime walltime limit (%s)",
		    schd_sec2val(schd_PT_WALLT_LIMIT));

	    return (1);
	}
    }

    /* Job is okay. */
    return (0);
}

/*
 * Determine whether or not this job will be able to complete
 * before the next non-primetime, if started "when" (usually
 * when==now). Returns 1: yes, job can finish before nonprimetime
 *                 or  0: no, could not complete.
 */

int schd_finish_before_np(Job *job, Queue *queue, time_t when, char *reason)
{
    int nextnpt = 0;

    /* If not primetime, than no limit. */
    if (!(schd_ENFORCE_PRIME_TIME && schd_TimeNow >= schd_ENFORCE_PRIME_TIME))
	return (1);

    /* If the job is a special-priority job, this limit does not apply. */
    if (job->flags & JFLAGS_PRIORITY)
	return (1);

    /* If this particular queue is ignoring primetime, no limits apply. */
    if (!queue->observe_pt)
	return (1);

    if (schd_prime_time(when)) {
        nextnpt = schd_secs_til_nonprime(when);

	/* 
	 * If the non-primetime drain has been violated (for a special job,
	 * most likely), don't bother trying to enforce it.
	 */
	if (queue->empty_by && ((queue->empty_by - schd_TimeNow) >= nextnpt))
	    return(1);

        if (job->walltime >= nextnpt) {
	    if (reason)
		(void)sprintf(reason, "System draining for non-primetime at %s",
		    schd_sec2val(schd_PRIME_TIME_END));
	    return(0);
	}
    }
    return(1);
}
