/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/

#include <stdio.h>
#include <pbs_ifl.h>
#include <log.h>
#include "check.h"
#include "config.h"
#include "server_info.h"
#include "queue_info.h"
#include "job_info.h"
#include "misc.h"
#include "constant.h"
#include "globals.h"
#include "dedtime.h"

static char *ident = "$Id: check.c,v 2.4.2.1 2000/01/22 00:17:11 hender Exp $";

/*
 *
 *	is_ok_to_run_in_queue - check to see if jobs can be run in a queue
 *
 *	  qinfo - queue info 
 *
 *	returns SUCCESS on success or failure code
 *
 *	NOTE: This function will be run once per queue every scheduling cycle
 *
 */

int is_ok_to_run_queue( queue_info *qinfo )
{
  int rc = UNSPECIFIED;		/* Return Code */

  if( qinfo -> is_exec )
  {
    if( qinfo -> is_started )
    {
      if( !( rc = check_ded_time_queue( qinfo ) ) )
      {
        rc = SUCCESS;
      }
    }
    else
      rc = QUEUE_NOT_STARTED;
  }
  else
    rc = QUEUE_NOT_EXEC;

  return rc;
}

/*
 *
 *	is_ok_to_run_job - check to see if the job can currently fit into the 
 *				system limits
 *
 *	  pbs_sd - the connection descriptor to the pbs_server
 *	  sinfo - server info
 *	  qinfo - queue info
 *	  jinfo - job info
 *
 *	returns  0 on success or failure code
 *
 *
 */
int is_ok_to_run_job( int pbs_sd, server_info *sinfo, queue_info *qinfo,
                                                        job_info *jinfo)
{
  int rc;                       /* Return Code */

  if( !jinfo -> is_queued )
    return NOT_QUEUED;

  if( (rc = check_server_max_run( sinfo )) )
    return rc;

  if( (rc = check_server_max_user_run(sinfo, jinfo -> account) ) )
    return rc;

  if( ( rc = check_server_max_group_run( sinfo, jinfo -> group) ) )
    return rc;

  if( ( rc = check_queue_max_run( qinfo ) ) )
    return rc;

  if( ( rc = check_queue_max_user_run( qinfo, jinfo -> account) ) )
    return rc;

  if( ( rc = check_queue_max_group_run( qinfo, jinfo -> group) ) )
    return rc;

  if( (rc = check_ded_time_boundry( jinfo ) ) )
    return rc;

  if( ( rc = check_starvation( jinfo ) ) )
    return rc;

  if( ( rc = check_nodes( pbs_sd, jinfo, sinfo -> timesharing_nodes )))
    return rc;
  
  if( ( rc = check_avail_resources(qinfo -> qres, jinfo) ) != SUCCESS )
    return rc;

  if( ( rc = check_avail_resources(sinfo -> res, jinfo) ) != SUCCESS )
    return rc;

  return SUCCESS;
}


/* 
 *
 *	check_server_max_user_run - check if the user is within server 
 *					user run limits
 *
 *	  sinfo - the server
 *	  account - the account name
 *
 *	returns 
 *	  0: if the user has not reached their limit
 *	  SERVER_USER_LIMIT_REACHED: if the user has reached their limit
 *
 */
int check_server_max_user_run( server_info *sinfo, char *account )
{
  if( sinfo -> max_user_run == INFINITY || 
      count_by_user( sinfo -> running_jobs, account ) < sinfo -> max_user_run)
    return 0;
  
  return SERVER_USER_LIMIT_REACHED;
}

/*
 *
 *	check_queue_max_user_run - check if the user is within queue 
 *					user run limits
 *
 *	  qinfo - the queue
 *	  account - the account name
 *
 *	returns 
 *	  0: if the user is under their limit
 *	  QUEUE_USER_LIMIT_REACHED: if the user has reached their limit
 *
 */
int check_queue_max_user_run( queue_info *qinfo, char *account )
{
  if( qinfo -> max_user_run == INFINITY ||
      count_by_user( qinfo -> running_jobs, account ) < qinfo -> max_user_run )
    return 0;
  
  return QUEUE_USER_LIMIT_REACHED;
}

/*
 *
 *	check_queue_max_group_run - check to see if the group is within their
 *					queue running limits
 *
 *	  qinfo - the queue 
 *	  group - the groupname
 *
 *	returns 
 *	  0 if the group is within limits
 *	  QUEUE_GROUP_LIMIT_REACHED: if group is not wihtin limits
 *
 */
int check_queue_max_group_run( queue_info *qinfo, char *group )
{
  if( qinfo -> max_group_run == INFINITY ||
      count_by_group( qinfo -> running_jobs, group ) < qinfo -> max_group_run)
    return 0;

  return QUEUE_GROUP_LIMIT_REACHED;
}

/*
 *
 *	check_server_max_group_run - check to see if group is within their 
 *				     server running limits
 *
 *	  sinfo - the server
 *	  group - the groupname
 *
 *	returns 
 *	  0 if the group is not at their server max_group_run limit
 *	  SERVER_GROUP_LIMIT_REACHED : if the limit is reached
 *
 */
int check_server_max_group_run( server_info *sinfo, char *group )
{
  if( sinfo -> max_group_run == INFINITY ||
      count_by_group( sinfo -> running_jobs, group ) < sinfo -> max_group_run)
    return 0;
  
  return SERVER_GROUP_LIMIT_REACHED;
}

/*
 *
 *      check_avail_resources - check if there is available resources to run
 *                              a job on the server
 *
 *        reslist   - resources list
 *        jinfo - job
 *
 *      returns NULL on success or failure code
 *
 *      NOTE: if a resource is not available it is skipped
 *            All resources which are checked are in the global variable
 *              res_to_check
 *
 */
int check_avail_resources( resource *reslist, job_info *jinfo)
{
  /* The resource needs to be found on the server and the requested resource
   * needs to be found from the job, these pointers are used to store the
   * results
   */
  resource_req *resreq;
  resource *res;
  int ret_code = UNSPECIFIED;
  char done = 0;                        /* Are we done? */
  int avail;                            /* amount of available resource */
  int i;

  for( i = 0; (i < num_res) && !done; i++ )
  {
    res = find_resource(reslist, res_to_check[i].name);
    resreq = find_resource_req(jinfo -> resreq, res_to_check[i].name);

    /* if either of these are NULL then the system admin did not set a maximum
     * default or avail for the resource.  Skip it and move on
     */
    if( (res == NULL || resreq == NULL) )
      continue;
    else
    {
      avail = dynamic_avail( res );
      if ( avail != INFINITY && avail < resreq -> amount )
      {
        /* An insuffient amount of a resource has been found. */
        done = 1;
        ret_code = i;
      }
    }
  }
  if( !done )
    ret_code = SUCCESS;

  return ret_code;
}



/*
 *
 *	dynamic_avail - find out how much of a resource is available on a 
 *			server.  If the resources_available attribute is 
 *			set, use that, else use resources_max.
 *
 *	  res - the resource to check
 *
 *	returns the available amount of the resource
 *
 */

sch_resource_t dynamic_avail( resource *res )
{
  if( res -> max == INFINITY && res -> avail == UNSPECIFIED )
    return INFINITY;
  else
    return ( res -> avail == UNSPECIFIED ? 
	(res -> max - res -> assigned) : (res -> avail - res -> assigned) );
}

/*
 *
 *	count_by_user - count the amount of jobs a user has in a job array
 *
 *	  jobs - job array
 *	  user - the username
 *
 *	returns the count
 *
 */
int count_by_user( job_info **jobs, char *user )
{
  int count = 0;		/* the accumulator to count the user's jobs */
  int i;

  if( jobs != NULL )
  {
    for( i = 0; jobs[i] != NULL; i++ )
      if( !strcmp( user, jobs[i] -> account) )
	count++;
  }

  return count;
}

/*
 *
 *
 *	count_by_group - count number of jobs a group has in job array
 *
 *	  jobs - array of jobs
 *	  group - group name
 *
 *	returns the count
 *
 */
int count_by_group( job_info **jobs, char *group )
{
  int i;
  int count = 0;		/* an accumulator to count the group's jobs */

  if( jobs != NULL )
  {
    for( i = 0; jobs[i] != NULL; i++ )
    {
      if( !strcmp( jobs[i] -> group, group ) )
	count++;
    }
  }

  return count;
}

/*
 *
 *	check_ded_time_boundry  - check to see if a job would cross into 
 *				   	dedicated time
 *
 *	  jinfo - the job
 *
 *	returns 0: will not cross a ded time boundry
 *		CROSS_DED_TIME_BOUNDRY: will cross a ded time boundry
 *
 */
int check_ded_time_boundry( job_info *jinfo )
{
  resource_req *res;		/* used to get jobs walltime request */
  sch_resource_t finish_time;	/* the finish time of the job */

  if( conf.ded_time[0].from )
  {
    res = find_resource_req(jinfo -> resreq, "walltime");

    if( res != NULL )
    {
      finish_time = cstat.current_time + res -> amount;
      if(!cstat.is_ded_time && finish_time > conf.ded_time[0].from)
	return CROSS_DED_TIME_BOUNDRY;
      else if( cstat.is_ded_time && finish_time > conf.ded_time[0].to )
	return CROSS_DED_TIME_BOUNDRY;
      else
	return 0;
    }
    else
      /* no walltime found for the job.  We have to assume it will cross a 
       * dedtime boundry
       */
      return CROSS_DED_TIME_BOUNDRY;
  }
  return 0;
}

/*
 *
 *	check_nodes - check to see if there is suficient nodes available to 
 *		      run a job.
 *
 *	  pbs_sd - communication descriptor to the pbs server
 *	  jinfo  - job to check
 *	  ninfo_arr - the node array
 *
 *	returns 0 if the job can run
 *		NOT_ENOUGH_NODES_AVAIL if the job can not run
 *		SCHD_ERROR on error
 *
 */
int check_nodes( int pbs_sd, job_info *jinfo, node_info **ninfo_arr )
{
  resource_req *nodes;		/* pointer for the nodes resource requested */
  int av;			/* are the nodes available */
  int al, res, down;		/* unused variables passed into pbs_rescquery */
  char buf[256];		/* used for request to pbs_rescquery() */
  char *tmp;			/* needed to make buf a char ** the call */
  char errbuf[256];
  int rc;			/* return code */

  nodes = find_resource_req(jinfo -> resreq, "nodes");

  if( nodes != NULL )
  {
    /* pbs_rescquery needs a char ** for its second argument.  Since buf is 
     * on the stack &buf == buf.  The tmp variable is used to provide the 
     * char **
     */
    sprintf(buf, "nodes=%s", nodes -> res_str );
    tmp = buf;
    if( (rc = pbs_rescquery(pbs_sd, &tmp, 1, &av, &al, &res, &down)) != 0 )
    {
      sprintf(errbuf, "pbs_resquery error: %d", rc);
      log(PBSEVENT_SYSTEM, PBS_EVENTCLASS_NODE, jinfo -> name, errbuf );
      return SCHD_ERROR;
    }
    else
    {
      /* the requested nodes will never be available */
      if( av < 0 )
        jinfo -> can_never_run = 1;

      if( av > 0 )
	return 0;
      else
	return NOT_ENOUGH_NODES_AVAIL;
    }
  }

  return check_node_availability( jinfo, ninfo_arr );
}

/*
 *
 *      check_node_availability - determine that there is a timesharing node 
 *                          available to run the job
 *
 *        jinfo - the job to run
 *        ninfo_arr - the array of nodes to check in
 *
 *      returns 
 *	  0: if we are load balancing and there is a node available or 
 *	     we are not load balancing
 *	  NO_AVAILABLE_NODE: if there is no node available to run the job
 *
 */
int check_node_availability( job_info *jinfo, node_info **ninfo_arr )
{
  int rc = NO_AVAILABLE_NODE;	/* return code */
  resource_req *req;		/* used to get resource values */
  sch_resource_t ncpus;		/* number of CPUS job requested */
  sch_resource_t mem;		/* amount of memory job requested */
  char *arch;			/* arch the job requested */
  char *host;			/* host name the job requested */
  int i;

  if( cstat.load_balancing || cstat.load_balancing_rr )
  {
    if( jinfo != NULL && ninfo_arr != NULL )
    {
      if( ( req = find_resource_req( jinfo -> resreq, "ncpus" ) ) == NULL )
	ncpus = 1;
      else
	ncpus = req -> amount;
   
      if( ( req = find_resource_req( jinfo -> resreq, "mem" ) ) == NULL )
	mem = 0;
      else
	mem = req -> amount;
   
      if( ( req = find_resource_req( jinfo -> resreq, "arch" ) ) == NULL )
	arch = NULL;
      else
	arch = req -> res_str;
   
      if( ( req = find_resource_req( jinfo -> resreq, "host" ) ) == NULL )
	host = NULL;
      else
	host = req -> res_str;
   
      for(i = 0; ninfo_arr[i] != NULL && rc != 0; i++)
      {
	if( ninfo_arr[i] -> is_free &&
	    (host == NULL || !strcmp(host, ninfo_arr[i] -> name) ) &&
	    (arch == NULL || !strcmp(arch, ninfo_arr[i] -> arch) ) &&
	    mem <= ninfo_arr[i] -> physmem )
	{
	  if(ninfo_arr[i] -> loadave + ncpus <= ninfo_arr[i] -> max_load)
	    rc = 0;
	}
      }
    }
    else
      rc = 0;		/* no nodes */
  }
  else
    rc = 0;		/* we are not load balancing */
  return rc;
}

/*
 *
 *	check_ded_time_queue - check if it is the approprate time to run jobs
 *			       in a dedtime queue
 *
 *	  qinfo - the queue
 *
 *	returns:
 *	  0: if it is dedtime and qinfo is a dedtime queue or
 *	     if it is not dedtime and qinfo is not a dedtime queue
 *
 *	  DED_TIME: if jobs can not run in queue because of dedtime restrictions
 *
 */
int check_ded_time_queue( queue_info *qinfo )
{
  int rc = 0;		/* return code */

  if( cstat.is_ded_time )
  {
    if( qinfo -> dedtime_queue )
      rc = 0;
    else 
      rc = DED_TIME;
  }
  else
  {
    if( qinfo -> dedtime_queue )
      rc = DED_TIME;
    else
      rc = 0;
  }
  return rc;
}

/*
 *
 *	check_queue_max_run - check to see if queues max run has been reached
 *
 *	  qinfo - the queue to check
 *
 *	returns 
 *	  0				: limit has not been reached
 *	  QUEUE_JOB_LIMIT_REACHED	: limit has been reached
 *
 */
check_queue_max_run( queue_info *qinfo )
{
  if( qinfo->max_run == INFINITY )
    return 0;
  else if( qinfo -> max_run > qinfo -> sc.running )
    return 0;
  else
    return QUEUE_JOB_LIMIT_REACHED;
}

/*
 *
 *	check_server_max_run - check to see if the server max_running limit has
 *			       been reached
 *
 *	  sinfo - the server to check
 *
 *	returns
 *	  0: if the limit has not been reached
 *	  SERVER_JOB_LIMIT_REACHED: if the limit has been reached
 *
 */
check_server_max_run( server_info *sinfo )
{
  if( sinfo -> max_run == INFINITY )
    return 0;
  else if( sinfo -> max_run > sinfo -> sc.running )
    return 0;
  else
    return SERVER_JOB_LIMIT_REACHED;
}

/*
 *
 *	check_starvation - if there is a starving job, don't allow jobs to run
 *			   until the starving job runs.
 *
 *	  jinfo - the current job to check
 *
 *	returns 
 *	  0: if the job is the starving job or no jobs are starving
 *	  JOB_STARVING: if it is not the starving job
 *
 */
int check_starvation( job_info *jinfo )
{
  if( cstat.starving_job == NULL || cstat.starving_job == jinfo )
    return 0;
  else
    return JOB_STARVING;
}
