/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <memory.h>
#include "misc.h"
#include "dedtime.h"
#include "globals.h"

static char *ident = "$Id: dedtime.c,v 2.1 1999/04/16 19:26:11 hender Exp $";

/*
 *
 *	parse_ded_file - read in dedicated times from file
 *
 *	  filename - filename of dedicatd time file
 *
 *	returns 0 on success non-zero on failure
 *
 *	NOTE: modifies conf data structure
 *
 *	FORMAT:      start         finish
 *		MM/DD/YY HH:MM MM/DD/YYYY HH:MM
 */
int parse_ded_file( char *filename )
{
  FILE *fp;			/* file pointer for the dedtime file */
  char line[256];		/* a buffer for a line from the file */
  int error = 0;		/* boolean: is there an error? */
  struct tm tm_from, tm_to;	/* tm structs used to convert to time_t */
  time_t from, to;		/* time_t values for dedtime start - end */
  int i;

  if( ( fp = fopen(filename, "r") ) == NULL )
  {
    sprintf(line, "Error opening file %s", filename);
    perror(line);
    return 1;
  }

  i = 0;
  memset(conf.ded_time, 0, MAX_DEDTIME_SIZE);
  while( fgets(line, 256, fp) != NULL )
  {
    if( !skip_line(line) )
    {
      /* mktime() will figure out if it is dst or not if tm_isdst == -1 */
      memset(&tm_from, 0, sizeof(struct tm));
      tm_from.tm_isdst = -1;

      memset(&tm_to, 0, sizeof(struct tm));
      tm_to.tm_isdst = -1;

      if( sscanf(line, "%d/%d/%d %d:%d %d/%d/%d %d:%d", &tm_from.tm_mon, &tm_from.tm_mday, &tm_from.tm_year, &tm_from.tm_hour, &tm_from.tm_min, &tm_to.tm_mon, &tm_to.tm_mday, &tm_to.tm_year, &tm_to.tm_hour, &tm_to.tm_min ) != 10 )
	error = 1;
      else
      {
	/* tm_mon starts at 0, where the file will start at 1 */
	tm_from.tm_mon--;

    /* the MM/DD/YY is the wrong date format, but we will accept it anyways */
	/* if year is less then 90, assume year is > 2000 */
	if( tm_from.tm_year < 90 )
	  tm_from.tm_year += 100;

	/* MM/DD/YYYY is the correct date format */
	if( tm_from.tm_year > 1900 )
	  tm_from.tm_year -= 1900;
	from = mktime(&tm_from);

	tm_to.tm_mon--;
	if( tm_from.tm_year < 90 )
	  tm_from.tm_year += 100;
	if( tm_to.tm_year > 1900 )
	  tm_to.tm_year -= 1900;
	to = mktime(&tm_to);

	/* ignore all dedtime which has passed */
	if( !(from < cstat.current_time && to < cstat.current_time ) )
	{
	  conf.ded_time[i].from = from;
	  conf.ded_time[i].to = to;
	  i++;
	}
      }
      if( error )
      {
	printf("Error: %s\n", line);
	error = 0;
      }
    }
  }
  /* sort dedtime in ascending order with all 0 elements at the end */
  qsort(conf.ded_time, MAX_DEDTIME_SIZE, sizeof(struct timegap), cmp_ded_time);
  fclose(fp);
  return 0;
}

/*
 *
 *	cmp_ded_time - compare function for qsort for the ded time array
 *
 *	  Sort Keys:
 *	    - zero elements to the end of the array
 *	    - descending by the start time
 *
 */
int cmp_ded_time(const void *v1, const void *v2)
{
  if(((struct timegap *)v1) -> from == 0 && ((struct timegap *)v2) -> from != 0)
    return 1;
  else if(((struct timegap *)v2) -> from == 0 && ((struct timegap *)v1) -> from != 0 )
    return -1;
  else if( ((struct timegap *)v1) -> from > ((struct timegap *)v2) -> from )
    return 1;
  else if ( ((struct timegap *)v1) -> from < ((struct timegap *)v2) -> from )
    return -1;
  else 
    return 0;
}

/*
 *
 *	is_ded_time - checks if it is currently dedicated time
 *
 *	returns: 1 if it is currently ded time
 *		 0 if it is not ded time
 *
 */
int is_ded_time()
{

  if( cstat.current_time > conf.ded_time[0].from && 
      cstat.current_time < conf.ded_time[0].to )
    return 1;
  else
    return 0;
}

