/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <ctype.h>
#include <strings.h>
#include <pbs_ifl.h>
#include <log.h>
#include "config.h"
#include "constant.h"
#include "misc.h"
#include "globals.h"
#include "fairshare.h"

static char *ident = "$Id: misc.c,v 2.1 1999/04/16 19:27:21 hender Exp $";

/*
 *
 *	string_dup - duplicate a string
 *
 *	  str - string to duplicate
 *
 *	returns newly allocated string
 *
 */

char *string_dup( char *str )
{
  char *newstr;
  if( ( newstr = (char *) malloc( strlen(str) + 1 ) ) == NULL )
    return NULL;
  
  strcpy(newstr, str);

  return newstr;
}

/*
 *
 *      res_to_num - convert a resource string to a  sch_resource_t to
 *                      kilobytes
 *                      example: 1mb -> 1024 
 *				 1mw -> 1024 * SIZE_OF_WORD
 *
 *      res_str - the resource string
 *
 *      return a number in kilobytes or UNSPECIFIED on error
 *
 */

sch_resource_t res_to_num( char * res_str )
{
  sch_resource_t count = UNSPECIFIED;	/* convert string resource to numeric */
  char *endp;				/* used for strtol() */
  char *endp2;				/* used for strtol() */
  long multiplier;			/* multiplier to count */

  count = strtol( res_str, &endp, 10 );

  if( *endp == ':' )		/* time resource -> convert to seconds */
  {
    count *= 3600;					/* hours */
    if( *endp == ':' )
    {
      count += (strtol( endp+1, &endp2, 10) * 60);	/* minutes */
      if( *endp2 == ':' )
        count += strtol( endp2 + 1, &endp, 10);		/* seconds */
    }
    multiplier = 1;
  }
  else if( *endp == 'k' )
    multiplier = 1;
  else if( *endp == 'm' )
    multiplier = MEGATOKILO;
  else if( *endp == 'g' )
    multiplier = GIGATOKILO;
  else if( *endp == 't' )
    multiplier = TERATOKILO;
  else if( *endp == 'b' )
  {
    count /= KILO;
    multiplier = 1;
  }
  else if( *endp == 'w' )
  {
    count /= KILO;
    multiplier = SIZE_OF_WORD;
  }
  else	/* error */
    multiplier = 1;

  if( *endp != '\0' && *(endp + 1) == 'w' )
    multiplier *= SIZE_OF_WORD;

  return count * multiplier;
}

/*
 *
 *      skip_line - find if the line of the config file needs to be skipped
 *                  due to it being a comment or other means
 *
 *        line - the line from the config file
 *
 *      returns true if the line should be skipped or false if it should be
 *              parsed
 *
 */
int skip_line( char *line )
{
  int skip = 0;				/* whether or not to skil the line */
 
  if( line != NULL )
  {
    while( isspace( (int) *line ) )
      line++;
 
    /* '#' is comment in config files and '*' is comment in holidays file */
    if( line[0] == '\0' || line[0] == '#' || line[0] == '*' )
      skip = 1;
  }
 
  return skip;
}

/*
 *
 *	log - write a log entry to the scheduler log file using log_record
 *
 *	  event - the event type
 *	  class - the event class
 *	  name  - the name of the object
 *	  text  - the text of the message 
 *
 *	returns nothing
 *
 */

void log( int event, int class, char *name, char *text)
{
  if( !(conf.log_filter & event) && text[0] != '\0' )
    log_record(event, class, name, text);
}

/*
 *
 *	break_comma_list - break apart a comma delemetd string into an arary
 *			   of strings
 *
 *	  list - the comma list
 *
 *	returns an array of strings
 *
 */
char **break_comma_list( char *list )
{
  int num_words = 1;			/* number of words delimited by commas*/
  char **arr = NULL;			/* the array of words */
  char *tok;				/* used with strtok() */
  int i;

  if( list != NULL )
  {
    for( i = 0; list[i] != '\0'; i++ )
      if( list[i] == ',' )
	num_words++;
    
    if( ( arr = (char **) malloc( sizeof(char*)  * (num_words + 1) ) ) == NULL )
    {
      perror("Memory Allocation Error");
      return NULL;
    }

    tok = strtok(list, ",");

    for( i = 0; tok != NULL; i++ )
    {
      while( isspace( (int) *tok) )
	tok++;
      
      arr[i] = string_dup(tok);
      tok = strtok(NULL, ",");
    }
    arr[i] = NULL;
  }
  return arr;
}

/* 
 *
 *	free_string_array - free an array of strings with a NULL as a sentinal
 *
 *	  arr - the array to free
 *
 *	returns nothing
 *
 */
void free_string_array( char **arr )
{
  int i;

  if( arr != NULL )
  {
    for( i = 0; arr[i] != NULL; i++)
      free(arr[i]);
    
    free(arr);
  }
}
