/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <ctype.h>
#include "data_types.h"
#include "parse.h"
#include "constant.h"
#include "config.h"
#include "misc.h"
#include "globals.h"
#include "fairshare.h"
#include "prime.h"
#include "node_info.h"

static char *ident = "$Id: parse.c,v 2.1.8.1 2000/01/22 00:17:13 hender Exp $";

/*
 *
 *	parse_config - parse the config file and set a struct config
 *
 *	  fname - file name of the config file
 *	  GLOBAL: conf  - config structure
 *
 *	returns success/failure
 *
 *	FILE FORMAT:
 *	config_name [white space] : [white space] config_value [prime_value]
 *
 *	EX: sort_by: shortest_job_first prime
 */
int parse_config( char *fname )
{
  FILE *fp;			/* file pointer to config file */
  char buf[256];		/* used to read in lines from the file */
  char *config_name;		/* parse first word of line */
  char *config_value;		/* parsed second word - right after colen (:) */
  char *prime_value;		/* optional third word */
  int num = -1;			/* used to convert string -> integer */
  char *endp;			/* used for strtol() */
  char error = 0;		/* boolean: is there an error? */
  enum prime_time prime;	/* used to convert string -> prime value */
  int linenum = 0;		/* the current line number in the file */
  int pkey_num = 1;		/* number of prime time keys for multisort */
  int npkey_num = 1;		/* number of nonprime time keys for multisort */
  int i;

  if( ( fp = fopen( fname, "r") ) == NULL )
  {
    sprintf(buf, "Can not open file: %s", fname);
    perror(buf);
    return 0;
  }

  while( fgets(buf, 256, fp) != NULL ) 
  {
    linenum++;
    error = 0;
    prime = ALL;
    num = -1;
    /* skip blank lines and comments */
    if( !skip_line( buf ) )
    {
      if( buf[strlen(buf)-1] == '\n' )
	buf[strlen(buf)-1] = '\0';

      config_name = strtok(buf, ": 	");
      config_value = strtok(NULL, " 	");
      prime_value = strtok(NULL, " 	");
      if( config_name != NULL && config_value != NULL )
      {
	/* value is true */
	if( !strcmp(config_value, "true") || !strcmp(config_value, "True") 
					  || !strncmp(config_value, "TRUE", 4))
	{
	  num = 1;
	}
	/* value is false */
	else if(!strcmp(config_value, "false") || 
	       !strcmp(config_value, "False") || !strcmp(config_value, "FALSE"))
	{
	  num = 0;
	}
	else if( isdigit( (int) config_value[0] ) )	/* value is number */
	{
	  num = strtol(config_value, &endp, 10);
	}
	else ; /* value is a string */

	if( prime_value != NULL )
	{
	  if( !strcmp(prime_value, "prime") || !strcmp(prime_value, "PRIME") )
	    prime = PRIME;
	  else if( !strcmp( prime_value, "non_prime") || 
		   !strcmp(prime_value, "NON_PRIME") )
	    prime = NON_PRIME;
	  else if(!strcmp( prime_value, "all" ) || !strcmp(prime_value, "ALL") )
	    prime = ALL;
	  else if(!strcmp(prime_value, "none") || !strcmp(prime_value, "NONE") )
	    prime = NONE;
	  else
	    error = 1;
	}

	if( !strcmp(config_name, PARSE_ROUND_ROBIN) )
	{
	  if( prime == PRIME || prime == ALL )
	    conf.prime_rr = num ? 1 : 0;
	  if( prime == NON_PRIME || prime == ALL )
	    conf.non_prime_rr = num ? 1 : 0;
	}
	else if( !strcmp(config_name, PARSE_BY_QUEUE) )
	{
	  if( prime == PRIME || prime == ALL )
	    conf.prime_bq = num ? 1 : 0;
	  if( prime == NON_PRIME || prime == ALL )
	    conf.non_prime_bq = num ? 1 : 0;
	}
	else if( !strcmp(config_name, PARSE_STRICT_FIFO) )
	{
	  if( prime == PRIME || prime == ALL )
	    conf.prime_sf = num ? 1 : 0;
	  if( prime == NON_PRIME || prime == ALL )
	    conf.non_prime_sf = num ? 1 : 0;
	}
	else if( !strcmp(config_name, PARSE_FAIR_SHARE) )
	{
	  if( prime == PRIME || prime == ALL )
	    conf.prime_fs = num ? 1 : 0;
	  if( prime == NON_PRIME || prime == ALL )
	    conf.non_prime_fs = num ? 1 : 0;
	}
	else if( !strcmp(config_name, PARSE_LOAD_BALENCING) )
	{
	  if( prime == PRIME || prime == ALL )
	    conf.prime_lb = num ? 1 : 0;
	  if( prime == NON_PRIME || prime == ALL )
	    conf.non_prime_lb = num ? 1 : 0;
	}
	else if( !strcmp(config_name, PARSE_HELP_STARVING_JOBS) )
	{
	  if( prime == PRIME || prime == ALL )
	    conf.prime_hsv = num ? 1 : 0;
	  if( prime == NON_PRIME || prime == ALL )
	    conf.non_prime_hsv = num ? 1 : 0;
	}
	else if( !strcmp( config_name, PARSE_SORT_QUEUES ) )
	{
	  if( prime == PRIME || prime == ALL )
	    conf.prime_sq = num ? 1 : 0;
	  if( prime == NON_PRIME || prime == ALL )
	    conf.non_prime_sq = num ? 1 : 0;
	}
	else if( !strcmp( config_name, PARSE_LOAD_BALENCING_RR ) )
	{
	  if( prime == PRIME || prime == ALL )
	    conf.prime_lbrr = num ? 1 : 0;
	  if( prime == NON_PRIME || prime == ALL )
	    conf.non_prime_lbrr = num ? 1 : 0;
	}
	else if( !strcmp( config_name, PARSE_MAX_STARVE ) )
	  conf.max_starve = res_to_num(config_value);
	else if( !strcmp( config_name, PARSE_HALF_LIFE ) )
	  conf.half_life = res_to_num( config_value );
	else if( !strcmp( config_name, PARSE_SYNC_TIME ) )
	  conf.sync_time = res_to_num( config_value );
	else if( !strcmp( config_name, PARSE_UNKNOWN_SHARES) )
	  conf.unknown_shares = num;
	else if( !strcmp( config_name, PARSE_LOG_FILTER) )
	  conf.log_filter = num;
	else if( !strcmp( config_name, PARSE_DEDICATED_PREFIX ) )
	{
	  if( strlen(config_value) > PBS_MAXQUEUENAME )
	    error = 1;
	  else
	    strcpy(conf.ded_prefix, config_value);
	}
	else if( !strcmp( config_name, PARSE_SORT_BY) )
	{
	  error = 1;
	  for( i = 0; i < num_sorts; i++ )
	  {
	    if( !strcmp(config_value, sorting_info[i].config_name) )
	    {
	      error = 0;
	      if( prime == PRIME || prime == ALL )
	        conf.prime_sort[0] = sorting_info[i];
	      if( prime == NON_PRIME || prime == ALL )
		conf.non_prime_sort[0] = sorting_info[i];
	    }
	  }
	}
	else if( !strcmp( config_name, PARSE_KEY ) )
	{
	  error = 1;
	  for( i = 0; i < num_sorts; i++ )
	  {
	    if( !strcmp(config_value, sorting_info[i].config_name) )
	    {
	      error = 0;
	      if( ( (prime == PRIME||prime == ALL) && pkey_num <= num_sorts ) ||
		  ((prime == NON_PRIME||prime == ALL) && npkey_num <= num_sorts) 
	        )
	      {
		if( prime == PRIME || prime == ALL )
		{
		  conf.prime_sort[pkey_num] = sorting_info[i];
		  pkey_num++;
		}
		if( prime == PRIME || prime == ALL )
		{
		  conf.non_prime_sort[npkey_num] = sorting_info[i];
		  npkey_num++;
		}
	      }
	      else
		fprintf(stderr, "Too many keys for %s , ignoring %s.\n", 
			prime == PRIME ? "prime" : "non-prime", config_value);
	    }
	  }
	}
      }
      else
	error = 1;
    }
    
    if( error )
      fprintf(stderr, "Error reading line %d of file %s\n%s\n", linenum, fname, buf);
  }
  fclose(fp);
  return 1;
}

/*
 *
 *      init_config - initalize the config struture
 *
 *      returns success / failure
 *
 */
int init_config()
{
  memset(&conf, 0, sizeof(struct config) );
  memset(&cstat, 0, sizeof(struct status) );

  if( (conf.prime_sort = malloc( (num_sorts + 1) * sizeof(struct sort_info) ) )
== NULL )
  {
    perror("Error Allocating Memory");
    return 0;
  }
  memset( conf.prime_sort, 0, (num_sorts + 1 ) * sizeof( struct sort_info ) );

  if( (conf.non_prime_sort = malloc( (num_sorts + 1) * sizeof(struct sort_info)
) ) == NULL )
  {
    perror("Error Allocating Memory");
    return 0;
  }
  memset( conf.non_prime_sort, (int) NULL, (num_sorts + 1 ) * 
  		sizeof( struct sort_info) );
  
  if( is_prime_time() == PRIME )
    init_prime_time();
  else
    init_non_prime_time();

  return 1;
}

/*
 *
 *      reinit_config - reinitialize the conf structure so the config file can
 *                    reparsed.
 *
 *      returns success / failure
 *
 */
int reinit_config()
{
  free( conf.prime_sort );
  free( conf.non_prime_sort );
  free_group_tree( conf.group_root );
  return init_config();
}
