static char USMID[] = "%Z%%M%	%I%	%G% %U%";

/*

    Program: libtoolkit.a
       File: complex.c

        Author : Nicholas P. Cardo
                 Sterling Software
                 NAS Facility
                 NASA Ames Research Center

  Description:
    Miscellaneous routines for the creation and management of 
    of queue complexes.

*/

#include <pbs_config.h>   /* the master config generated by configure */

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <string.h>
/*
 *  Scheduler header files
 */
#include "gblxvars.h"
#include "toolkit.h"

static	struct qcomplex *qc;

/*
 *  Create a complex of queues.  Create a linked list
 *  of complexes.
 */
int create_complex(cfg_option,qs)
char	*cfg_option;
char	*qs;
{
char	*id = "create_complex";
struct	qcomplex *qp;
char	*cname;

	/*
	 *  complex name
	 */
	cname = strstr(cfg_option,"_");
	cname++;

	/*
	 *  max name length is 15 leaving 1 for \0 terminator
	 */
	if(strlen(cname) > 15)
		return(-1);

	/*
	 *  Go to the end of the list
	 */
	qp = qc = Chead;
	while(qc != (struct qcomplex *)0L) {
		if(!strcmp(cname,qc->name)) {
			/*
			 *  Error, duplicate complex
			 */
			return(-1);
		}

		qp = qc;
		qc = qc->next;
	}

	/*
	 *  Now is the time to actually create the complex
	 */
	if(Chead == (struct qcomplex *)0L) {
		Chead = (struct qcomplex *)malloc(sizeof(struct qcomplex));
		qc = Chead;
	} else {
		qp->next = (struct qcomplex *)malloc(sizeof(struct qcomplex));
		qc = qp->next;
	}

	memset(qc,0,sizeof(struct qcomplex));	/* zero out      */
	sprintf(qc->name,"%s\0",cname);		/* complex name  */
	qc->members = strdup(qs);		/* member queues */

	return(0);
}

/*
 *  Store all queues belonging any complex in a linked list
 */
int create_qlist(cfg_arg)
char	*cfg_arg;
{
char	*id = "create_qlist";
char	*qname;
char	*s;
struct	qlist *q;
struct	qlist *qp;
char	*ptr;

	s = strdup(cfg_arg);

	qname = strtok(s,",");

	while(qname != NULL) {
		/*
		 *  max name length is 15 leaving 1 for \0 terminator
		 */
		if(strlen(qname) > 15) {
			free(s);
			return(-1);
		}

		/*
		 *  Go to the end of the list
		 */
		qp = q = Qhead;
		while(q != (struct qlist *)0L) {
			if(!strcmp(qname,q->name)) {
				/*
				 *  Error, duplicate queue
				 */
				free(s);
				return(-1);
			}
	
			qp = q;
			q = q->next;
		}

		/*
		 *  Now is the time to create the complex
		 */
		if(Qhead == (struct qlist *)0L) {
			Qhead = (struct qlist *)malloc(sizeof(struct qlist));
			q = Qhead;
		} else {
			qp->next = (struct qlist *)malloc(sizeof(struct qlist));
			q = qp->next;
		}

		memset(q,0,sizeof(struct qlist));
		sprintf(q->name,"%s\0",qname);

		q->qcom = qc;

		qname = strtok(NULL,",");
	}

	return(0);
}

/*
 *  set a complex limit into a complex
 */
int qcset(cname,tlim,lim)
char	*cname;
char	*tlim;
char	*lim;
{
char	*id = "qcset";
struct	qcomplex *qcom;

	qcom = Chead;
	while(qcom != (struct qcomplex *)0L) {
		if(!strcmp(qcom->name,cname)) 
			break;

		qcom = qcom->next;
	}

	if(qcom == (struct qcomplex *)0L) 
		return(-1);

	if(!strcmp(tlim,"JOBS"))
		qcom->jobs = atoi(lim);
	if(!strcmp(tlim,"MEMORY"))
		qcom->mem  = val2byte(lim);
	if(!strcmp(tlim,"CPU"))
		qcom->cpu  = val2sec(lim);
	if(!strcmp(tlim,"ROUTER")) {
		if(strlen(lim) > 15)
			return(-1);
		sprintf(qcom->router,"%s\0",lim);
	}

	return(0);
}

/*
 *  Create a linked list of router queues with their destinations
 */
int create_router(cfg_arg)
char	*cfg_arg;
{
char	*id = "create_router";
char	*rtr,*qs;
struct	routeq *rp,*rc;

	/*
	 *  point to router
	 */
	rtr = strdup(cfg_arg);

	/*
	 *  Point to queue list
	 */
	qs = strstr(rtr,":");
	*qs = '\0';
	qs++;

	/*
	 *  Go to the end of the list
	 */
	rp = rc = Rhead;
	while(rc != (struct routeq *)0L) {
		if(!strcmp(rtr,rc->name)) {
			/*
			 *  Error, duplicate router
			 */
			free(rtr);
			return(-1);
		}

		rp = rc;
		rc = rc->next;
	}

	/*
	 *  Now is the time to actually create the router
	 */
	if(Rhead == (struct routeq *)0L) {
		Rhead = (struct routeq *)malloc(sizeof(struct routeq));
		rc = Rhead;
	} else {
		rp->next = (struct routeq *)malloc(sizeof(struct routeq));
		rc = rp->next;
	}

	memset(rc,0,sizeof(struct routeq));	/* zero out      */
	sprintf(rc->name,"%s\0",rtr);		/* complex name  */
	rc->qlist = strdup(qs);		/* member queues */

	free(rtr);
	return(0);
}

/*
 *  This routine is used to release all memory held
 *  by all the structures and linked lists used to implement
 *  complexes.
 */
void clrcomp()
{
char	*id = "clrcomp";
struct	qcomplex *qcn, *qcp;
struct	qlist *qln, *qlp;
struct	routeq *rqn, *rqp;
struct	denied_user *dun, *dup;

	qcn = qcp = Chead;

	while(qcp != (struct qcomplex *)NULL) {
		free(qcp->members);
		dun = dup = qcp->du;
		while(dup != (struct denied_user *)NULL) {
			dun = dup->next;
			free(dup);
			dup = dun;
		}
		qcn = qcp->next;
		free(qcp);
		qcp = qcn;
	}

	qln = qlp = Qhead;

	while(qlp != (struct qlist *)NULL) {
		qln = qlp->next;
		free(qlp);
		qlp = qln;
	}

	rqn = rqp = Rhead;

	while(rqp != (struct routeq *)NULL) {
		free(rqp->qlist);
		rqn = rqp->next;
		free(rqp);
		rqp = rqn;
	}
}
