static char USMID[] = "%Z%%M%	%I%	%G% %U%";

/*

    Program: pbs_sched.a
       File: evaluate_system.c

        Author : Nicholas P. Cardo
                 Sterling Software
                 NAS Facility
                 NASA Ames Research Center

  Description:
   Perform some basic analysis of system resources to determine if 
   jobs should be run.

*/

#include <pbs_config.h>   /* the master config generated by configure */

#include <stdio.h>
#include <stdlib.h>
/*
 *  PBS header files
 */
#include "pbs_error.h"
#include "pbs_ifl.h"
#include "log.h"
/*
 *  Scheduler head files
 */
#include "gblxvars.h"
#include "toolkit.h"

#define DONTSTARTJOB 0
#define	STARTJOB 1

int
evaluate_system(rsrcs)
struct	resutl *rsrcs;
{
char	*id = "evaluate_system";

	/*
	 *  Are there enough jobs running
	 */
	if((rsrcs->njobs < MIN_JOBS) && MIN_JOBS) {
		(void) sprintf(log_buffer,"Not running enough jobs");
		log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
		return(STARTJOB);
	}

	/*
	 *  Are there too many jobs running
	 */
	if((rsrcs->njobs >= MAX_JOBS) && MAX_JOBS) {
		(void) sprintf(log_buffer,"Maximum number of jobs running");
		log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
		return(DONTSTARTJOB);
	}

	/*
	 *  Does the system have a sufficient load
	 */
	if((rsrcs->loadave < MIN_LOADAVG) && MIN_LOADAVG) {
		(void) sprintf(log_buffer,"Load Average too low");
		log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
		return(STARTJOB);
	}

	/*
	 *  Does the system have too much work to do
	 */
	if((rsrcs->loadave > MAX_LOADAVG) && MAX_LOADAVG) {
		(void) sprintf(log_buffer,"Load Average too high");
		log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
		return(DONTSTARTJOB);
	}

	/*
	 *  If system time is high and we are permitting deferred jobs
	 *  then change our mind and don't permit deferred jobs.  The
	 *  high system time could be a short duration anomaly due to
	 *  several jobs all doing I/O.  However, we should let regular
	 *  jobs start if we are taking idle time.
	 */
	if((rsrcs->systime >= HIGH_SYSTIME) && HIGH_SYSTIME) {
		(void) sprintf(log_buffer,
			"High system time, no jobs permitted");
		log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);

		return(DONTSTARTJOB);
	}

	/*
	 *  Check swapin rate
	 */
	if((rsrcs->swapinrate >= HIGH_SWAPRATE) && HIGH_SWAPRATE) {
		(void) sprintf(log_buffer,"High swapin rate");
		log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
		return(DONTSTARTJOB);
	}

	(void) sprintf(log_buffer,"Done");
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);

	/*
	 *  If we are not running the nominal number of jobs or if
	 *  we are taking idle, then it's okay to start jobs.
	 */
	if(((rsrcs->loadave < NOM_LOADAVG) && NOM_LOADAVG) || rsrcs->idltime) {
		(void) sprintf(log_buffer,"Okay to start jobs");
		log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
		return(STARTJOB);
	}

	/*
	 *  If we reached here, then we really don't need to start 
	 *  another job.  If we do, then the system could become over
	 *  burdened with work.
	 */
	return(DONTSTARTJOB);
}
