static char USMID[] = "%Z%%M%	%I%	%G% %U%";

/*

    Program: libsched.a
       File: pnp.c

        Author : Nicholas P. Cardo
                 Sterling Software
                 NAS Facility
                 NASA Ames Research Center

  Description:
    It is important to be able to accurately determine when it is
    prime and non-prime time.  This is accomplished by reading the
    holidays file and creating a table of prime and non-prime times.
    A quick search of the table can then determine whether we are in
    prime or non-prime time.

*/

#include <pbs_config.h>   /* the master config generated by configure */

#include <stdio.h>
#include <sys/types.h>
#include <string.h>
#include <time.h>

#define	SUNDAY   0
#define SATURDAY 1
#define WEEKDAY  2
#define	HOLIDAYS 3
#define	PRIME    0
#define NONPRIME 1
#define	ALL      2500
#define NONE     2600

static	int  Num_holidays;
static	int  holidays[20][2];
static  int  day_pnp(int,int);
static	void load_day(int,int,char *);
static	void init_holidays();

/*
 *  Is it prime time or nonprime time
 *  return 0 for prime time
 *  return 1 for nonprime time
 */
int pnp()
{
time_t	timval;
struct	tm *tmptr;
int	cur_time;
int	h_idx;
int	rcode;

	/*
	 *  What is today
	 */
	time(&timval);
	tmptr = localtime(&timval);
	cur_time = (tmptr->tm_hour * 100) + tmptr->tm_min;

	/*
	 *  Is today a holiday
	 */
	for(h_idx=HOLIDAYS;h_idx<Num_holidays+3;h_idx++) {
		if(tmptr->tm_yday+1 == holidays[h_idx][0])
			return(NONPRIME);
	}

	/*
	 *  Today isn't a holiday so is it SATURDAY, SUNDAY or WEEKDAY
	 */
	switch(tmptr->tm_wday) {
		case 0: rcode = day_pnp(SUNDAY,cur_time);
			break;
		case 6: rcode = day_pnp(SATURDAY,cur_time);
			break;
		default:
			rcode = day_pnp(WEEKDAY,cur_time);
			break;
	}
	return(rcode);
}

/*
 *  Based on time of day is it prime time now
 */
static int day_pnp(dow,now)
int	dow;
int	now;
{
int	p;
int	np;

	p = holidays[dow][PRIME];
	np = holidays[dow][NONPRIME];

	switch(p) {
		case ALL: return(PRIME);
		case NONE: return(NONPRIME);
		default: if(np > p) {
				if((now >= p) && (now < np))
					return(PRIME);
				else
					return(NONPRIME);
			} else {
				if((now >= np) && (now < p))
					return(NONPRIME);
				else
					return(PRIME);
			}
	}
}

/*
 *  Read the holidays file into the holidays array
 */
void read_holidays()
{
FILE	*fd;
char	buf[255];
char	*ptr;
char	prime_time[8],nonprime_time[8],dow[20];
int	h_idx = HOLIDAYS;
int	version_ok;

	init_holidays();
	Num_holidays=0;
	version_ok = 0;

	if((fd=fopen("/usr/lib/acct/holidays","r")) == NULL) {
		(void) fprintf(stderr,"error opening holidays file\n");
		exit(-1);
	}

	while(fgets(buf,sizeof(buf),fd)) {
		/*
		 *  go to first character
		 */
		ptr=buf;
		while(isspace(*ptr))
			ptr++;

		/*
		 * Skip comments
		 */
		if(*ptr == '*')
			continue;
		
		if(!strncmp(ptr,"YEAR",4)) {
			/*
			 *  Year
			 */
			continue;
		} else if(!strncmp(ptr,"HOLIDAYFILE_VERSION1",20)) {
			/*
			 *  HOLIDAY VERSION, must be 8.0 style
			 */
			version_ok++;
			continue;
		} else if(!strncmp(ptr,"sunday",6)) {
			/*
			 * Sunday
			 */
			sscanf(buf,"%s %s %s", dow, prime_time, nonprime_time);
			load_day(SUNDAY,PRIME,prime_time);
			load_day(SUNDAY,NONPRIME,nonprime_time);
		} else if(!strncmp(ptr,"saturday",8)) {
			/*
			 * Saturday
			 */
			sscanf(buf,"%s %s %s", dow, prime_time, nonprime_time);
			load_day(SATURDAY,PRIME,prime_time);
			load_day(SATURDAY,NONPRIME,nonprime_time);
		} else if(!strncmp(ptr,"weekday",7)) {
			/*
			 * M-F
			 */
			sscanf(buf,"%s %s %s", dow, prime_time, nonprime_time);
			load_day(WEEKDAY,PRIME,prime_time);
			load_day(WEEKDAY,NONPRIME,nonprime_time);
		} else if(!strncmp(ptr,"199",3) || (!strncmp(ptr,"200",3))) {
			/*
			 * NON-Unicos Format (eg. IRIX, AIX, Solaris...)
			 * Line format:
			 *    * Curr  Prime   Non-Prime
			 *    * Year  Start   Start
			 *    1997    0700    1800
			 */
			sscanf(buf,"%s %s %s", dow, prime_time, nonprime_time);
			load_day(WEEKDAY,PRIME,prime_time);
			load_day(WEEKDAY,NONPRIME,nonprime_time);
		} else {
			/*
			 *  Holidays
			 */
			sscanf(buf,"%s", dow);
			holidays[h_idx][0] = atoi(dow);
			holidays[h_idx][NONPRIME] = ALL;
			h_idx++;
			Num_holidays++;
		}
	}
	fclose(fd);

	/*
	 *  If we are running under Unicos, and got an old style
	 *  holidays file than use the defaults. Only support
	 *  current style Holidays file.
	 *
	 *  Otherwise, accept the normal UNIX format.
	 */
#ifdef _CRAY
	if(!version_ok) {
		Num_holidays = 0;
		init_holidays();
	}
#endif
}

/*
 *  Load a day into the array
 */
static void load_day(dow,pnp,val)
int	dow;
int	pnp;
char	*val;
{
	if(!strcmp(val,"all"))
		holidays[dow][pnp] = ALL;
	else if(!strcmp(val,"none"))
		holidays[dow][pnp] = NONE;
	else
		holidays[dow][pnp] = atoi(val);
}

/*
 *  Initialize holidays array so that 
 *  Saturday and Sunday are non prime and
 *  Monday - Friday 0800-1700 are prime time
 *  and there are no holidays
 */
static void init_holidays()
{
	holidays[SUNDAY][PRIME]      = NONE;
	holidays[SUNDAY][NONPRIME]   = ALL;
	holidays[SATURDAY][PRIME]    = NONE;
	holidays[SATURDAY][NONPRIME] = ALL;
	holidays[WEEKDAY][PRIME]     = 800;
	holidays[WEEKDAY][NONPRIME]  = 1700;

	Num_holidays = 0;
}

