static char USMID[] = "%Z%%M%	%I%	%G% %U%";

/*

    Program: libsched.a
       File: router.c

        Author : Nicholas P. Cardo
                 Sterling Software
                 NAS Facility
                 NASA Ames Research Center

  Description:
    Jobs are moved from a routing queue to execution queues based
    on predescribed policy.  The routines here are designed to 
    enforce those policy limits and move jobs to appropriate 
    execution queues.  In effect, this enforces user limits at a
    complex level.

*/

#include <pbs_config.h>   /* the master config generated by configure */

#include	<stdio.h>
#include        <stdlib.h>
#include        <errno.h>
#include        <time.h>
#include        <string.h>
#include	<sys/types.h>
#ifdef UNICOS
#include	<udb.h>
#endif /* UNICOS */
/*
 *  PBS header files
 */
#include        "pbs_error.h"
#include        "pbs_ifl.h"
#include        "log.h"
/*
 *  Scheduler header files
 */
#include	"gblxvars.h"
#include	"toolkit.h"

extern	int	connector;

static	char *already_denied();
static	void deny_free();
static	void deny_user();

char	*acid2nam();

static	struct rqlist rql[10];
int	compar();

static	char	*nohigh = "Insufficient high priority allocation";
static	char	*noroute = 
	"Unable to route job: requested CPU or Memory limit too large";

static int enforce_policy(struct batch_status *, char *);

int route(rqueue)
char	*rqueue;
{
char	            *id = "route";
struct batch_status *bs,*obs;
int	             ncpus,cput,prio;
long long            mem;
static char	    *q;
struct qcomplex     *qc;
static struct routeq       *rq = (struct routeq *)0L;
int		     x;
int		     found;
char	            *qname;
struct  attrl blist[] = {
	{NULL,ATTR_comment,"",""}
};
struct  attrl clist[] = {
        {NULL,ATTR_A,"",""}
};
char	*vbl;
char	*ptr;
char	*usr;
#ifdef UNICOS
struct	udb *ueptr;
#endif /* UNICOS */
char	*delmsg;

	/*
	 *  Get information about queue
	 */
	if(queue_limits(rqueue))
		return(0);

	/**************************************** old code ****************
	 *  Get the router destinations
	 *
	 * rq = Rhead;
	 * while(rq != (struct routeq *)0L) {
	 *     if(!strcmp(rq->name,rqueue)) 
	 * 		break;
	 *	rq = rq->next;
	 * }
	 * JJPJ
	 *
	 *  Make sure we have a valid router
	 *
	 * if(rq == (struct routeq *)0L)
 	 *	return(-1);
	 * JJPJ
	 **************************************** end old code ************
	 */

	/*
	 *  Make sure we have a valid router, if not, create one and set it 
	 *  up for future use.
	 */
	if(rq == (struct routeq *)0L) {
		sprintf(log_buffer,"No router for queue %s, creating...",rqueue);
		log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER, id,log_buffer);
		/*
		 *  Get the router destinations
		 */
		rq = Rhead;
		while(rq != (struct routeq *)0L) {
			if(!strcmp(rq->name,rqueue)) 
				break;
			rq = rq->next;
		}
	     	if(rq == (struct routeq *)0L)
			return(-1);

	    	/*
	     	 *  Initialize queue destination table
	     	 */
	    	for(x=0;x<10;x++) {
			rql[x].mem=rql[x].cput=UNSPECIFIED;
			if(rql[x].name[0] != '\0')
				free(rql[x].name);
			rql[x].name[0] = '\0';
	    	}

	    	/*
	     	 *  Now build a table of queues to
	     	 *  route to and sort it by memory
	     	 */
	    	bldqtable(rq->qlist);
	}
	/*
	 *  Get the list of all jobs queued in the router queue
	 */
	bs = getjobs(rqueue,"Q");
	obs = bs;

	/*
	 *  Loop by each job
	 */
	while(bs != (struct batch_status *)0L) {
		delmsg = noroute;

		/*
		 *  Get the attributes
		 */
		cput = 0;
		mem = 0L;

		cput = val2sec(getat(ATTR_l,bs,"cput"));
		mem  = val2byte(getat(ATTR_l,bs,"mem"));
		usr  = getat(ATTR_euser,bs,NULL);

		vbl  = getat(ATTR_v,bs,NULL);
		ptr  = strtok(vbl,",");
		while(ptr != NULL) {
			if(!strncmp(ptr,"PBS_O_QUEUE=",12)) {
				ptr = strchr(ptr,61);
				ptr++;
				break;
			}
			ptr = strtok(NULL,",");
		}

		/*
		 *  Check for a match to multitasking queues
		 */
		found = 0;
		if(!strcmp(ptr,"mt")) {
			for(x=0;x<10;x++) {
				if(strncmp(rql[x].name,"mt",2))
					continue;
				if((mem  <= rql[x].mem) &&
				   (cput <= rql[x].cput)) {
					found = 1;
					qname = (char *)&rql[x].name;
					break;
				}
			}
		} else if(!strcmp(ptr,"high")) {
			if(usrok(usr,cput)) {
				found = 1;
				qname = "highbatch";
			} else {
				found = 0;
				delmsg = nohigh;
			}
		} else {
			for(x=0;x<10;x++) {
				if(!strncmp(rql[x].name,"mt",2))
					continue;

				if(!strncmp(rql[x].name,"highbatch",9))
					continue;

				if((cput <= rql[x].cput) &&
				   (mem  <= rql[x].mem )) {
					qname = (char *)&rql[x].name;
					found=1;
					break;
				}
			}
		}

		if(!found) {
			/*
			 *  I don't know what to do with this job
			 *  so delete it.
			 */
			sprintf(log_buffer,"can't route %s",bs->name);
			log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER, id,log_buffer);
			pbs_deljob(connector,bs->name,delmsg);

		/*
		 * JJPJ } else if(!enforce_policy(bs,qname)) {
		 */
		} else {
			/*
			 *  Check to see if queue is enabled
			 */
			if(qenabled(qname)) {
				/*
				 *  Route the job to the real queue
				 */
				sprintf(log_buffer,"routing %s to %s",
					bs->name,qname);
				log_record(PBSEVENT_SYSTEM,
					PBS_EVENTCLASS_SERVER,id,log_buffer);
	
				blist[0].value = "";
				pbs_alterjob(connector,bs->name,blist,NULL);
				pbs_movejob(connector,bs->name,qname,NULL);
			}
		}

		bs = bs->next;
	}

	/*
	 *  Clean up to prevent a memory leak
	 */
	pbs_statfree(obs);

	/*
	 * JJPJ deny_free();
	 */

	return(0);
}

/*
 *  Enforce routing policy based on the pseudo
 *  complex level limits
 */
static int enforce_policy(bs,qn)
struct	batch_status *bs;
char	*qn;
{
char	*id = "enforce_policy";
int	jobs,cpu,tcpu;
long long mem,tmem;
static	char *joblimit = "job limit";
static	char *memlimit = "memory limit";
static	char *cpulimit = "cpu limit";
static	char *p,*s;
struct	batch_status *b1,*b2;
static	struct qlist *q;
/*
static  struct attropl alist[] = {
	{&alist[1],ATTR_state,"","Q",EQ},
	{&alist[2],ATTR_q,    "","", EQ},
	{&alist[3],ATTR_euser,"","", EQ},
	{NULL,     ATTR_qtype,"","E",EQ}
};
*/
static  struct attropl alist[] = {
	{&alist[1],ATTR_state,"","Q",EQ},
	{&alist[2],ATTR_q,    "","", EQ},
	{NULL,ATTR_euser,"","", EQ}
};

struct  attrl blist[] = {
	{NULL,ATTR_comment,"",""}
};

	jobs=cpu=tcpu = 0;
	mem=tmem=0L;

	/*
	 *  Locate the destination queue in the
	 *  list of queues that are part of complexes
	 */
	q = Qhead;
	while(q != (struct qlist *)0L) {
		if(!strcmp(q->name,qn))
			break;
		q = q->next;
	}

	/*
	 *  See if the queue is a member of a complex
	 */
	if(q == (struct qlist *)0L) 
		return(0);

	/*
	 *  Get user requested limits
	 */
	tmem = val2byte(getat(ATTR_l,bs,"mem"));
	tcpu = val2sec (getat(ATTR_l,bs,"cput"));
	alist[2].value = getat(ATTR_euser,bs,NULL);

	/*
	 *  See if we are already restricting this user
	 *  based on a previous
	 */
	if((p=already_denied(alist[2].value,q->qcom->du)) != NULL) {
		blist[0].value = p;
		pbs_alterjob(connector,bs->name,blist,NULL);
		return(1);
	}

	/*
	 *  Total up all information for jobs in the 
	 *  complex's queues for this user
	 */
	s = strdup(q->qcom->members);

	p = strtok(s,",");
	while(p != NULL) {
		alist[1].value = p;
	
		b1 = pbs_selstat(connector,alist,NULL);
		b2 = b1;
		if(pbs_errno) {
			(void) sprintf(log_buffer,"pbs_selstat failed, %d",
				pbs_errno);
			log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,
				id,log_buffer);
			return(-1);
		}
	
		while(b1 != (struct batch_status *)0L) {
			jobs++;
			mem += val2byte(getat(ATTR_l,b1,"mem"));
			cpu += val2sec (getat(ATTR_l,b1,"cput"));
			b1 = b1->next;
		}

		pbs_statfree(b2);
		p = strtok(NULL,",");
	}

	free(s);

	/*
	 *  Complex user job limit
	 */
	if((q->qcom->jobs) && ((jobs+1) > q->qcom->jobs)) {
		blist[0].value = joblimit;
		pbs_alterjob(connector,bs->name,blist,NULL);
		deny_user(alist[2].value,q,joblimit);
		return(1);
	}

	/*
	 *  Complex user memory limit
	 */
	if((q->qcom->mem) && ((mem+tmem) > q->qcom->mem)) {
		blist[0].value = memlimit;
		pbs_alterjob(connector,bs->name,blist,NULL);
		deny_user(alist[2].value,q,memlimit);
		return(1);
	}

	/*
	 *  Complex user cpu limit
	 */
	if((q->qcom->cpu) && ((cpu+tcpu) > q->qcom->cpu)) {
		blist[0].value = cpulimit;
		pbs_alterjob(connector,bs->name,blist,NULL);
		deny_user(alist[2].value,q,cpulimit);
		return(1);
	}

	return(0);
}

/*
 *  See if user is already being held up in the
 *  router.  This is used to prevent time consuming
 *  accumulations for each job.  Instead, once a user
 *  is denied, all jobs from that point on are automatically
 *  denied without rechecking all jobs.  This adds some efficiency.
 */
static char * already_denied(user,dl)
char	*user;
struct	denied_user *dl;
{
struct	denied_user *du;
char	*id = "already_denied";

	du = dl;
	
	while(du != (struct denied_user *)0L) {
		if(!strcmp(du->user,user))
			return(du->reason);

		du = du->next;
	}
	return(NULL);
}

/*
 *  Add a user to the denied users list
 */
static void deny_user(user,q,why)
char	*user;
struct	qlist *q;
char	*why;
{
char	*id = "deny_user";
struct	denied_user *du,*dl;

	/*
	 *  Create list if it doesn't exist
	 */
	if(q->qcom->du == (struct denied_user *)0L) {
		du = (struct denied_user *) malloc(sizeof(struct denied_user));
		q->qcom->du = du;
	} else {
		/*
		 *  Append to list
		 */
		du = q->qcom->du;

		while(du->next != (struct denied_user *)0L) {
			du = du->next;
		}

		du->next = (struct denied_user *)
			malloc(sizeof(struct denied_user));

		du = du->next;
	}

	memset(du,0,sizeof(struct denied_user));

	sprintf(du->user,"%s\0",user);
	du->reason = strdup(why);
}

/*
 *  Free each entry in the denied users list
 */
static void deny_free()
{
struct	denied_user *d1,*d2;
struct	qcomplex *qc;

	qc = Chead;

	/*
	 *  Loop through each complex
	 */
	while(qc != (struct qcomplex *)0L) {
		/*
		 *  no list to free
		 */
		if(qc->du == (struct denied_user *)0L) {
			qc = qc->next;
			continue;
		}

		/*
		 *  Free the list
		 */
		d2 = d1 = qc->du;
		while(d1 != (struct denied_user *)0L) {
			d1 = d1->next;
			free(d2->reason);
			free(d2);
			d2 = d1;
		}
	
		/*
		 *  Reset the pointer
		 */
		qc->du = (struct denied_user *)0L;
	
		qc = qc->next;
	}
}

/*
 *  Construct a table of destination queues 
 */
int bldqtable(queues)
char	*queues;
{
char	*id = "bldqtable";
struct	batch_status *bs;
char	*qs;
char	*q;
static  struct attrl alist[] = {
	{NULL,ATTR_rescmax,"",""}
};

int	x;
struct	rqlist *base = &rql[0];

	/*
	 *  Loop by queue in the member list
	 */
	qs = strdup(queues);
	q = strtok(qs,",");

	x = 0;
	while(q != NULL) {
		/*
		*  Ask the server for queue information
		*/
		if((bs=pbs_statque(connector,q,alist,NULL)) == (struct batch_status *)0L) {
			sprintf(log_buffer,"pbs_statque failed, %d",pbs_errno);
			log_record(PBSEVENT_ERROR,PBS_EVENTCLASS_SERVER,id, log_buffer);
			free(qs);
			return(-1);
		}

		sprintf(rql[x].name,"%s\0",q);

		rql[x].mem = val2byte(getat(ATTR_rescmax,bs,"mem"));
		rql[x].cput = val2sec(getat(ATTR_rescmax,bs,"cput"));

		q = strtok(NULL,",");
		x++;
		pbs_statfree(bs);
	}

	/*
	 *  Sort the list
	 */
	qsort(base,10,sizeof(*base),compar);

	free(qs);
	return(0);
}

/*
 *  Primary key is memory, secondary key is cpu time
 */
int compar(const void *a, const void *b)
{

	if(((struct rqlist *)a)->mem == ((struct rqlist *)b)->mem) {
		/*
		 * Check secondary key
		 */
		return(((struct rqlist *)a)->cput - ((struct rqlist *)b)->cput);

	} else {
		/*
		 * sort on primary key
		 */
		return(((struct rqlist *)a)->mem - ((struct rqlist *)b)->mem);
	}
}

