/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
static char USMID[] = "%Z%%M%	%I%	%G% %U%";

/*

    Program: pbs_sched.a
       File: evaluate_system.c

        Author : Nicholas P. Cardo
                 Sterling Software
                 NAS Facility
                 NASA Ames Research Center

  Description:
   Perform some basic analysis of system resources to determine if 
   jobs should be run.

*/

#include <pbs_config.h>   /* the master config generated by configure */

#include <stdio.h>
#include <stdlib.h>
/*
 *  PBS header files
 */
#include "pbs_error.h"
#include "pbs_ifl.h"
#include "log.h"
/*
 *  Scheduler head files
 */
#include "gblxvars.h"
#include "toolkit.h"

#define DONTSTARTJOB 0
#define	STARTJOB 1

int
evaluate_system(rsrcs)
struct	resutl *rsrcs;
{
char	*id = "evaluate_system";

	/*
	 *  Are there enough jobs running
	 */
	if((rsrcs->njobs < MIN_JOBS) && MIN_JOBS) {
		(void) sprintf(log_buffer,"Not running enough jobs");
		log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
		return(STARTJOB);
	}

	/*
	 *  Are there too many jobs running
	 */
	if((rsrcs->njobs >= MAX_JOBS) && MAX_JOBS) {
		(void) sprintf(log_buffer,"Maximum number of jobs running");
		log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
		return(DONTSTARTJOB);
	}

	/*
	 *  Does the system have a sufficient load
	 */
	if((rsrcs->loadave < MIN_LOADAVG) && MIN_LOADAVG) {
		(void) sprintf(log_buffer,"Load Average too low");
		log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
		return(STARTJOB);
	}

	/*
	 *  Does the system have too much work to do
	 */
	if((rsrcs->loadave > MAX_LOADAVG) && MAX_LOADAVG) {
		(void) sprintf(log_buffer,"Load Average too high");
		log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
		return(DONTSTARTJOB);
	}

	/*
	 *  If system time is high and we are permitting deferred jobs
	 *  then change our mind and don't permit deferred jobs.  The
	 *  high system time could be a short duration anomaly due to
	 *  several jobs all doing I/O.  However, we should let regular
	 *  jobs start if we are taking idle time.
	 */
	if((rsrcs->systime >= HIGH_SYSTIME) && HIGH_SYSTIME) {
		(void) sprintf(log_buffer,
			"High system time, no jobs permitted");
		log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);

		return(DONTSTARTJOB);
	}

	/*
	 *  Check swapin rate
	 */
	if((rsrcs->swapinrate >= HIGH_SWAPRATE) && HIGH_SWAPRATE) {
		(void) sprintf(log_buffer,"High swapin rate");
		log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
		return(DONTSTARTJOB);
	}

	(void) sprintf(log_buffer,"Done");
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);

	/*
	 *  If we are not running the nominal number of jobs or if
	 *  we are taking idle, then it's okay to start jobs.
	 */
	if(((rsrcs->loadave < NOM_LOADAVG) && NOM_LOADAVG) || rsrcs->idltime) {
		(void) sprintf(log_buffer,"Okay to start jobs");
		log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
		return(STARTJOB);
	}

	/*
	 *  If we reached here, then we really don't need to start 
	 *  another job.  If we do, then the system could become over
	 *  burdened with work.
	 */
	return(DONTSTARTJOB);
}
