/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
static char USMID[] = "%Z%%M%	%I%	%G% %U%";

/*

    Program: pbs_sched.a
       File: getrsrcs.c

        Author : Nicholas P. Cardo
                 Sterling Software
                 NAS Facility
                 NASA Ames Research Center

  Description:
    Obtain resource information from the resource monitor and
    job counts from the server.

*/

#include <pbs_config.h>   /* the master config generated by configure */

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/sysmacros.h>
#include <string.h>
/*
 *  PBS header files
 */
#include "pbs_error.h"
#include "pbs_ifl.h"
#include "log.h"
#include "resmon.h"
/*
 *  Scheduler header files
 */
#include "gblxvars.h"
#include "toolkit.h"

extern	int	 connector;
extern	long long	LARGE_MEM_LIM;
char	*getreq();

int	Large_Running;

struct	resutl * get_resutl()
{
char	*id = "get_resutl";
int	rm;
long long	tmpmem;
char	*tmpq;
char	*ptr1, *ptr2;
char    *response;
char    tmpname[PBS_MAXSERVERNAME];
struct	batch_status *bs;
struct	batch_status *bs1;
char	membuf[32];
static	struct resutl rsrcs;

static	struct attrl alist[] = {
        {&alist[1],ATTR_count,   "",   ""},
	{NULL,     ATTR_rescassn,"mem",""}
};

static	struct attropl qlist[] = {
	{&qlist[1],ATTR_l,    "mem","                ",GE},
	{NULL,     ATTR_state,NULL,"R",EQ}
};

char *buf[100];
char tstbuf[100];
int badreply;


	badreply = 0;
	/*
	 *  Establish a connection to the resource monitor
	 */
	if((rm=openrm(SCHED_HOST,0)) == -1) {
		(void) sprintf(log_buffer,
			"Unable to contact resource monitor, %s",SCHED_HOST);
		log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
		return((struct resutl *)0L);
	}

	/*
	 *  Turn off full response, responses will be in the order
	 *  in which they are sent.
	 */
	fullresp(0);

	/*
	 *  Build a list of all the resources we want information on
	 */
		/*addreq(rm,"swapinrate"); */
	addreq(rm,"loadave");
	addreq(rm,"availmem");
	addreq(rm,"physmem");
	addreq(rm,"ncpus");

	/* these are available under UNICOS, but not IRIX
	 * would need to be added back in...
	 *
	 * addreq(rm,"cpuidle");  
	 * addreq(rm,"cpuunix"); 
	 * addreq(rm,"cpuuser"); 
	 * addreq(rm,"cpuguest");
	 */

	/*
	 *  Get the values back from the resource monitor, round up
	 */
		/* rsrcs.swapinrate = (int)(atof(getreq(rm)) + 0.5);   */

	response = getreq(rm);
	if (response != NULL) {
		rsrcs.loadave    = (int)(atof(response) + 0.5);
	} else {
		(void) sprintf(log_buffer,"bad return from getreq(loadave), %d, %d", pbs_errno, errno);
		log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
		badreply = 1;
	}
	response = getreq(rm);
	if (response != NULL) {
		rsrcs.freemem    = (long long)((atof(response) + 0.5) * 16 * 1024);
	} else {
		(void) sprintf(log_buffer,"bad return from getreq(freemem), %d, %d", pbs_errno, errno);
		log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
		badreply = 1;
	}
		

	response = getreq(rm);
	if (response != NULL) {
		rsrcs.realmem    = (long long)((atof(response) + 0.5) * 16 * 1024);
	} else {
		(void) sprintf(log_buffer,"bad return from getreq(realmem), %d, %d", pbs_errno, errno);
		log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
		badreply = 1;
	}
		

	response = getreq(rm);
	if (response != NULL) {
		rsrcs.ncpus_total    = atoi(response);
	} else {
		(void) sprintf(log_buffer,"bad return from getreq(ncpus), %d, %d", pbs_errno, errno);
		log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
		badreply = 1;
	}
		
	/*
	 *  Disconnect from the resource monitor
	 */
	closerm(rm);

	/* 
	 * Verify all the data came back as expected; 
	 * if not, abort this scheduler iteration.
	 */

	if (badreply) {
		(void) sprintf(log_buffer,"aborting scheduling iteration");
		log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
		return((struct resutl *)0L);
	}

	/*
	 *  How much batch memory and NCPUs are actually in use and how
	 *  many jobs are running?
	 *
	 *
	 *  Get all running jobs
	 */
	bs = getjobs(NULL,"R");
	bs1 = bs;

	rsrcs.njobs = 0;
	rsrcs.batchmem = 0L;
	rsrcs.ncpus_alloc = 0;
	while(bs1 != (struct batch_status *)0L) {
		/*
		 *  Only count jobs running on this host
		 */
		ptr1 = getat(ATTR_exechost,bs1,NULL);
		strcpy(tmpname, SCHED_HOST);
		ptr2 = strtok(tmpname, ".");
		if(!strncmp(ptr1,ptr2,strlen(ptr2))) {
			rsrcs.njobs++;
			rsrcs.batchmem += val2byte(getat(ATTR_l,bs1,"mem"));
			rsrcs.ncpus_alloc += atoi(getat(ATTR_l,bs1,"ncpus"));
		}
		bs1 = bs1->next;
	}

	pbs_statfree(bs);

	/*
	 *  How many large memory jobs are running that are mt
	 *  and how many large memory jobs are running that are
	 *  not mt.  This is used by the selection algorithm to
	 *  help maintain a balance of jobs to run by keeping too
	 *  few jobs using all of memory from using too few processors.
	 */
	sprintf(qlist[0].value,"%lld\0",LARGE_MEM_LIM);

	rsrcs.lrgmem = rsrcs.lrgmtcnt = rsrcs.lrgcnt = 0L;

	if((bs=pbs_selstat(connector,qlist,NULL)) ==
	   (struct batch_status *)0L) {
		if(pbs_errno) {
			(void) sprintf(log_buffer,"selstat failed, %d",
				pbs_errno);
			log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,
				id,log_buffer);
			return((struct resutl *)0L);
		}
	} else {
		/*
		 *  Go count up and classify the large memory jobs
		 */
		bs1 = bs;
		tmpmem = 0L;
		tmpq = NULL;
	
		while(bs != (struct batch_status *)0L) {
			ptr1 = getat(ATTR_exechost,bs,NULL);
			strcpy(tmpname, SCHED_HOST);
			ptr2 = strtok(tmpname, ".");
			if(strncmp(ptr1,ptr2,strlen(ptr2))) {
				bs = bs->next;
				continue;
			}

			tmpmem = 0L;
			if(tmpq != NULL) {
				free(tmpq);
				tmpq = NULL;
			}
		
			rsrcs.lrgmem += val2byte(getat(ATTR_l,bs,"mem"));
			tmpq = strdup(getat(ATTR_queue,bs,NULL));

			if(!strncmp(tmpq,"mt",2)){
				/*
				 *  This is an mt job
				 */
				rsrcs.lrgmtcnt++;
			} else {
				rsrcs.lrgcnt++;
			}
	
			/*
			 *  clear out the temporary fields
			 */
			tmpmem = 0L;
			if(tmpq != NULL) {
				free(tmpq);
				tmpq = NULL;
			}
	
			bs = bs->next;
		}
	
		pbs_statfree(bs1);
	}

	Large_Running = rsrcs.lrgmtcnt + rsrcs.lrgcnt;

	/*
	 *  Log the system's status
	 */
	/*
	(void) sprintf(log_buffer,
		" %d%% usr, %d%% sys, %d%% idl, %d%% gst",
		rsrcs.usrtime,rsrcs.systime,rsrcs.idltime,rsrcs.gsttime);
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
	*/

	(void) sprintf(log_buffer,
		" Memory (free) %lld bytes",rsrcs.freemem);
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);

	(void) sprintf(log_buffer,
		" Memory (real) %lld bytes",rsrcs.realmem);
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);

	(void) sprintf(log_buffer,
		" Memory allocated %lld bytes",rsrcs.batchmem);
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);

	(void) sprintf(log_buffer,
		" CPUs allocated: %d / %d", rsrcs.ncpus_alloc, rsrcs.ncpus_total);
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);

	(void) sprintf(log_buffer,
		" CPU Load average %d",rsrcs.loadave);
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);

/*
	(void) sprintf(log_buffer,
		" There %s %d job%srunning", rsrcs.njobs == 1 ? "is" : "are",
		rsrcs.njobs, rsrcs.njobs == 1 ? " " : "s ");
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
*/

	(void) sprintf(log_buffer,
		" Jobs: %d mt, %d non-mt, %d total",
		rsrcs.lrgmtcnt, rsrcs.lrgcnt, rsrcs.njobs);
		/*(rsrcs.njobs - rsrcs.lrgcnt), rsrcs.lrgcnt, rsrcs.njobs);*/
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);

	(void) sprintf(log_buffer,
		" Large memory used %lld bytes",rsrcs.lrgmem);
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);

	/*
	 *  Return the resource structure
	 */
	return(&rsrcs);
}
