/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
static char USMID[] = "%Z%%M%	%I%	%G% %U%";

/*

    Program:
       File: 

        Author : Nicholas P. Cardo
                 Sterling Software
                 NAS Facility
                 NASA Ames Research Center

  Description:

*/

#include <pbs_config.h>   /* the master config generated by configure */

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/sysmacros.h>
#include <string.h>
#ifdef UNICOS
#include <msg_funcs.h>
#endif

/*
 *  PBS header files
 */
#include "pbs_error.h"
#include "pbs_ifl.h"
#include "log.h"
/*
 *  Scheduler header files
 */
#include "toolkit.h"
#include "gblxvars.h"

extern	int connector;


/*
 *  Check queue limits to see if a job should be
 *  run out of this job.
 */
int
run_ded_queue()
{
char	*id = "run_ded_queue";
struct	batch_status *bs;
struct	attrl *aptr;
int	running,queued,started,maxrun;
int	msg;
char	*r;
int	x;
int	badnum;
char	*ptr;
static	int jobs2run=0;

static	struct attrl alist[] = {
	{&alist[1],ATTR_start, "",""},
	{&alist[2],ATTR_maxrun,"",""},
	{NULL,     ATTR_count, "",""}
};

	/*
	 *  Safety Precaution to insure we
	 *  don't run too many
	 */
	if(jobs2run == -1) 
		return(-1);

	/*
	 *  Initialize to UNSPECIFIED
	 */
	running=queued=started=maxrun = UNSPECIFIED;

	/*
	 *  Ask the server for queue information
	 */
	if((bs=pbs_statque(connector,"ded",alist,NULL)) == 
	   (struct batch_status *)0L) {
		sprintf(log_buffer,"pbs_statque failed, %d",pbs_errno);
		log_record(PBSEVENT_ERROR,PBS_EVENTCLASS_SERVER,id,log_buffer);
		return(-1);
	} 

	/*
	 *  Process the results
	 */
	aptr = bs->attribs;
	while(aptr != (struct attrl *)0L) {
		if(!strcmp(aptr->name,ATTR_start)) {
			if(!strcmp(aptr->value,"True"))
				started = 1;
			else if(!strcmp(aptr->value,"False"))
				started = 0;

		} else if(!strcmp(aptr->name,ATTR_count)) {
			running = how_many(aptr->value,SC_RUNNING);
			queued  = how_many(aptr->value,SC_QUEUED);
		} else if(!strcmp(aptr->name,ATTR_maxrun)) {
			maxrun = atoi(aptr->value);
		}

		aptr = aptr->next;
	}

	pbs_statfree(bs);

	/*
	 *  Is the queue started
	 */
	if((started == UNSPECIFIED) || (started == 0)) 
		return(0);

	/*
	 *  Is the queue below the run limit
	 */
	if((maxrun != UNSPECIFIED) && (maxrun <= running))
		return(-1);

#ifdef UNICOS
	/*
	 *  If there are more than MAX_DED_JOBS queued, then
	 *  ask how many to run
	 */
	if(!running && (queued > MAX_DED_JOBS) && !jobs2run) {
		badnum=1;
		while(badnum) {
			(void) sprintf(log_buffer,
		"PBS: There are %d queued ded jobs, how many should run? ",
				queued);

			msg=send_message(log_buffer,MSG_SEND_REP);
			receive_reply(msg,&r,1);
			/*
			 *  Check the response
			 */
			ptr = r;
			badnum = 0;
			for(x=0;x<strlen(ptr);x++) {
				if(!isdigit(*ptr))  {
					badnum = 1;
					break;
				}
				ptr++;
			}

			if(!badnum)
				jobs2run = atoi(r);
		}
	} else {
		jobs2run = MAX_DED_JOBS;
	}
#else
	jobs2run = MAX_DED_JOBS;

#endif /* UNICOS */

	/*
	 *  Are there any jobs to run or did we run as many
	 *  as we should
	 */
	if((!queued && !running) || (!jobs2run)) {
		/*
		 *  No more to run
		 */
#ifdef UNICOS
		msg=send_message("PBS: Dedicated Job Processing Completed",
			MSG_SEND_REP);
		receive_reply(msg,&r,1);
#endif /* UNICOS */
		jobs2run = -1;
		return(-1);
	}

	/*
	 *  Get the a job to run from the ded queue
	 */
	if((bs=getjobs("ded","Q")) == (struct batch_status *)0L)
		return(-1);

	pbs_runjob(connector,bs->name,NULL,NULL);
	jobs2run--;

	pbs_statfree(bs);

	return(-1);
}

