/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
/* $Id: evaluate_system.c,v 1.1 1999/04/16 19:32:35 hender Exp $ */

/*
 * Perform some basic analysis of system resources to determine if 
 * jobs should be run.
 */

#include <stdio.h>
#include <stdlib.h>

/* PBS header files */
#include "pbs_error.h"
#include "pbs_ifl.h"
#include "log.h"

/*  Scheduler head files */
#include "toolkit.h"
#include "gblxvars.h"

#define DONT_START_JOB 0
#define	START_JOB 1

int
schd_evaluate_system(Resources *rsrcs, char *reason)
{
    double loadchk;

    if (rsrcs->nodes_alloc == rsrcs->nodes_total) {
	if (reason)
	    (void)sprintf(reason, "No nodes available to schedule");
	return (DONT_START_JOB);
    }

    /* Are there enough jobs running?  If not, start one. */
    if ((rsrcs->njobs < schd_MIN_JOBS) && schd_MIN_JOBS) {
	if (reason)
	    (void)sprintf(reason, "Not running enough jobs (%d/%d min)",
		rsrcs->njobs, schd_MIN_JOBS);
	return (START_JOB);
    }

    /* Are there too many jobs running?  If so, don't start one. */
    if ((rsrcs->njobs >= schd_MAX_JOBS) && schd_MAX_JOBS) {
	if (reason)
	    (void)sprintf(reason, "Maximum number of jobs running (%d/%d max)",
		rsrcs->njobs, schd_MAX_JOBS);
	return (DONT_START_JOB);
    }

    /* Does the system have a sufficient load?  If not, start a job. */
    loadchk = (double)((schd_TARGET_LOAD_PCT - schd_TARGET_LOAD_MINUS) * 
	(rsrcs->nodes_total * PE_PER_NODE));
    if (rsrcs->loadave < loadchk) {
	if (reason)
	    (void)sprintf(log_buffer, "Load Average too low (%.02f/%.02f min)",
		rsrcs->loadave, loadchk);
	return (START_JOB);
    }

    /* Does the system have too much work to do?  Don't start a new job. */
    loadchk = (double)((schd_TARGET_LOAD_PCT + schd_TARGET_LOAD_PLUS) * 
	rsrcs->nodes_total * PE_PER_NODE);
    if (rsrcs->loadave > loadchk) {
	if (reason)
	    (void)sprintf(reason, "Load Average too high (%.02f/%.02f max)",
		rsrcs->loadave, loadchk);
	return (DONT_START_JOB);
    }

    /*
     * If system time is high and we are permitting deferred jobs then 
     * change our mind and don't permit deferred jobs.  The high system 
     * time could be a short duration anomaly due to several jobs all 
     * doing I/O.  However, we should let regular jobs start if we are 
     * taking idle time.
     */
    if ((rsrcs->systime >= schd_HIGH_SYSTIME) && schd_HIGH_SYSTIME) {
	if (reason)
	    (void)sprintf(reason, "High system time (%d/%d max)",
		rsrcs->systime, schd_HIGH_SYSTIME);
	return (DONT_START_JOB);
    }

    /*
     *  If we are not running the nominal number of jobs or if
     *  we are taking idle, then it's okay to start jobs.
     */
    loadchk = (double)(schd_TARGET_LOAD_PCT * rsrcs->nodes_total * PE_PER_NODE);
    if ((rsrcs->loadave < loadchk) || rsrcs->idltime) {
	if (reason)
	    (void)sprintf(log_buffer, 
		"Load average below nominal (%.02f/%.02f nom).  Start jobs.",
		rsrcs->loadave, loadchk);
	return (START_JOB);
    }

    if (reason)
	(void)sprintf(reason, "Done.  Don't need to run any jobs.");

    /*
     *  If we reached here, then we really don't need to start 
     *  another job.  If we do, then the system could become over
     *  burdened with work.
     */
    return (DONT_START_JOB);
}
