/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
/* $Id: fragments.c,v 1.1 1999/04/16 19:32:41 hender Exp $ */

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/sysmacros.h>
#include <string.h>

/* PBS header files */

#include "pbs_error.h"
#include "pbs_ifl.h"
#include "log.h"

/* Scheduler header files */

#include "toolkit.h"
#include "gblxvars.h"

int
schd_fragment_okay(Job *job, Queue *queue, char *reason)
{
    /* char   *id = "fragment_okay"; */
    int     nodes_avail, frag_size, avg_nodes;

    /*
     * If this job has been waiting for a long time, don't allow anything to
     * interfere with it's execution.
     */
    if (job->flags & JFLAGS_WAITING)
	return (1);

    /* 
     * "Fragments" only make sense if there is a node limit and a maximum
     * number of running jobs.  If not, fragmentation is not really an issue.
     */
    if ((queue->nodes_max == UNSPECIFIED) ||
	(queue->maxrun == UNSPECIFIED) ||
	(queue->maxrun < 2))
    {
	return (1);
    }

    frag_size = (queue->nodes_max / queue->maxrun);
    nodes_avail = (queue->nodes_max - queue->nodes_assn);

    /*
     * See if the queue has fragments of less than 2 nodes, and return if
     * so -- fragmentation is nonsensical for <2 nodes/frag.
     */
    if (frag_size < 2)
	return (1);

    /*
     * If there is less than a fragment left, do not run a job that would
     * cause this queue to be fragmented when the other jobs exit.
     */
    if (nodes_avail < frag_size) {
	if ((job->walltime + schd_TimeNow) > queue->empty_by) {
	    if (reason)
		sprintf(reason, 
		    "Would leave fragments in %s (must be <= %s walltime)", 
		    queue->qname, schd_sec2val(queue->empty_by - schd_TimeNow));
	    return (0);
	}

	/*
	 * This job will end before the queue is drained, so go ahead and
	 * run it.  It may temporarily increase fragmentation, but it will
	 * not be a problem for long.
	 */
	return (1);
    }

    /*
     * Find the average number of nodes consumed by jobs in this queue.  This
     * number is rounded to the nearest fragment size.  If there are no 
     * running jobs, allow anything to run and deal with the consequences
     * later.  This should help prevent starvation of very small jobs.
     */
    if (queue->running)
	avg_nodes = ((queue->nodes_assn + (frag_size / 2)) / queue->running);
    else
	return (1);

    /*
     * If the average number of nodes in use by the processes running in the
     * queues is less than the fragment size, then run a job that is smaller
     * than the fragment size, as long as it won't go beyond the time limit
     * of the longest running job.  If there are no running jobs, than let 
     * the fragment run.
     */
    if (job->nodes < frag_size) {

	/* This job is a fragment, and the average job in the queue is not. */

	if (avg_nodes >= frag_size) {
	    if (reason) {
		sprintf(reason, 
		    "Would fragment queue %s (must be >= %d nodes)",
		    queue->qname, frag_size);
	    }
	    return (0);
	}

	if (queue->running && 
	    ((job->walltime + schd_TimeNow) > queue->empty_by))
	{
	    if (reason) {
		sprintf(reason, 
		    "Would perpetuate fragmentation (must be < %d sec)",
		    (queue->empty_by - schd_TimeNow));
	    }
	    return (0);
	}
    }

    return (1);
}
