/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
/* $Id: getjobs.c,v 1.1 1999/04/16 19:32:53 hender Exp $ */

/* Get all the jobs queued in the specified queue. */

#include	<stdio.h>
#include	<stdlib.h>

#include        "pbs_error.h"
#include        "pbs_ifl.h"
#include        "log.h"


extern int connector;

#include "toolkit.h"

Job *
schd_get_jobs(char *qname, char *state)
{
    char   *id = "schd_get_jobs";
    int     idx;
    Job    *joblist = NULL, *jobtail = NULL, *new;
    Batch_Status *pbs_head, *pbs_ptr;
    AttrOpList *attr;

    static AttrOpList alist[] =
    {
	{NULL, NULL, NULL, NULL, EQ},
	{NULL, NULL, NULL, NULL, EQ}
    };

    if ((qname == NULL) && (state == NULL)) {
	attr = NULL; /* Caller requested all jobs in all queues. */
    } else {
	/*
	 * Initialize the search criteria since alist is a static struct 
	 * and it will retain the previous search when repeatedly called.
	 */
	for (idx = 0; idx < (sizeof (alist) / sizeof (AttrOpList)); idx++) {
	    alist[idx].next  = NULL;
	    alist[idx].name  = NULL;
	    alist[idx].value = NULL;
	}

	idx = 0;

	/* Was a specific queue requested? */
	if (qname != NULL) {
	    alist[idx].name = ATTR_queue;
	    alist[idx].value = qname;
	    idx++;
	}

	/* Was a specific state requested? */
	if (state != NULL) {
	    alist[idx].name = ATTR_state;
	    alist[idx].value = state;
	    if (idx > 0)
		alist[idx - 1].next = &alist[idx];
	    idx++;
	}

	/* (More tests can be added here.) */

	attr = alist;
    }

    /* Ask PBS for the list of jobs requested */
    pbs_head = pbs_selstat(connector, attr, NULL);

    if ((pbs_head == NULL) && (pbs_errno)) {
	(void)sprintf(log_buffer, "pbs_selstat failed, %d", pbs_errno);
	log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, id, log_buffer);
	return (NULL);
    }

    for (pbs_ptr = pbs_head; pbs_ptr != NULL; pbs_ptr = pbs_ptr->next) {
	/*
	 * If there is no list yet, create one.  If there is already a list,
	 * create a new element and place it after the current tail.  The new
	 * element then becomes the tail.
	 */
	new = (Job *)malloc(sizeof(Job));

	if (new == NULL) {
	    log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, id,
		"malloc(new Job)");
	    /*
	     * Free any allocated storage, set joblist to NULL, and break.
	     * By doing this, the PBS batch_struct list will be freed,
	     * and the NULL joblist returned to the caller.
	     */
	    if (joblist)
		schd_free_jobs(joblist);
	    joblist = NULL;
	    break;
	}

	new->next = NULL;

	if (!joblist) {
	    joblist = new;
	    jobtail = joblist;
	} else {
	    jobtail->next = new;
	    jobtail = jobtail->next;
	}

	/*
	 * 'jobtail' now points to a newly-created Job at the end of the
	 * list of jobs.  Call get_jobinfo() to fill it in with the contents
	 * of this PBS batch_struct description.
	 */
	
	schd_get_jobinfo(pbs_ptr, jobtail);
    }

    /*
     * We are left with a list of Job's that was created the from the list 
     * of Batch_Structs we got from pbs_selstat().  The Job list should
     * contain everything we need to know about the jobs.  It is okay to 
     * free the list returned by PBS, and return the list of Job's.
     */

    pbs_statfree(pbs_head);

    return (joblist);
}
