/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
/*
 * accounting.c - contains functions to record accounting information
 *
 * Functions included are:
 *	acct_open()
 *	acct_record()
 *	acct_close()
 */

static char ident[] = "@(#) $RCSfile: accounting.c,v $ $Revision: 2.2 $";


#include <pbs_config.h>   /* the master config generated by configure */
#include "portability.h"
#include <sys/param.h>
#include <sys/types.h>
#include <string.h>
#include <errno.h>
#include <limits.h>
#include <stdio.h>
#include <time.h>
#include "list_link.h"
#include "attribute.h"
#include "server_limits.h"
#include "job.h"
#include "queue.h"
#include "log.h"
#include "acct.h"

/* Local Data */

static FILE	    *acctfile;		/* open stream for log file */
static volatile int  acct_opened = 0;
static int	     acct_opened_day;
static int	     acct_auto_switch = 0;

/* Global Data */

extern attribute_def job_attr_def[];
extern char	    *path_acct;
extern int	     resc_access_perm;
extern time_t	     time_now;


/*
 * acct_job - build common data for start/end job accounting record
 *	Used by account_jobstr() and account_jobend()
 */

static char *acct_job(pjob, pb)
	job  *pjob;
	char *pb;	/* buffer in which data is to be placed */
{
	list_head attrlist;
	svrattrl *pal;

	CLEAR_HEAD(attrlist);

	/* user */
	(void)sprintf(pb, "user=%s ",
		      pjob->ji_wattr[(int)JOB_ATR_euser].at_val.at_str);
	pb += strlen(pb);

	/* group */
	(void)sprintf(pb, "group=%s ",
		      pjob->ji_wattr[(int)JOB_ATR_egroup].at_val.at_str);
	pb += strlen(pb);

	/* account */
	if (pjob->ji_wattr[(int)JOB_ATR_account].at_flags & ATR_VFLAG_SET) {
		(void)sprintf(pb, "account=%s ",
			    pjob->ji_wattr[(int)JOB_ATR_account].at_val.at_str);
		pb += strlen(pb);
	}

	/* job name */
	(void)sprintf(pb, "jobname=%s ",
		      pjob->ji_wattr[(int)JOB_ATR_jobname].at_val.at_str);
	pb += strlen(pb);

	/* queue name */
	(void)sprintf(pb, "queue=%s ",
		      pjob->ji_qhdr->qu_qs.qu_name);
	pb += strlen(pb);

	/* create time */
	(void)sprintf(pb, "ctime=%ld ",
		      pjob->ji_wattr[(int)JOB_ATR_ctime].at_val.at_long);
	pb += strlen(pb);

	/* queued time */
	(void)sprintf(pb, "qtime=%ld ",
		      pjob->ji_wattr[(int)JOB_ATR_qtime].at_val.at_long);
	pb += strlen(pb);

	/* eligible time, how long ready to run */
	(void)sprintf(pb, "etime=%ld ", pjob->ji_wattr[(int)JOB_ATR_etime].at_val.at_long);
	pb += strlen(pb);

	/* execution start time */
	(void)sprintf(pb, "start=%d ", pjob->ji_qs.ji_stime);
	pb += strlen(pb);

	/* execution host name */
	(void)sprintf(pb, "exec_host=%s ",
		      pjob->ji_wattr[(int)JOB_ATR_exec_host].at_val.at_str);
	pb += strlen(pb);

	/* now encode the job's resource_list attribute */

	resc_access_perm = READ_ONLY;
	(void)job_attr_def[(int)JOB_ATR_resource].at_encode(
			&pjob->ji_wattr[(int)JOB_ATR_resource],
			&attrlist,
			job_attr_def[(int)JOB_ATR_resource].at_name,
			(char *)0,
			ATR_ENCODE_CLIENT);

	while (pal = GET_NEXT(attrlist)) {
		(void)strcat(pb, pal->al_name);
		if (pal->al_resc) {
			(void)strcat(pb, ".");
			(void)strcat(pb, pal->al_resc);
		}
		(void)strcat(pb, "=");
		(void)strcat(pb, pal->al_value);
		(void)strcat(pb, " ");
		delete_link(&pal->al_link);
		(void)free(pal);
		pb += strlen(pb);
	}
	return (pb);
}
/*
 * acct_open() - open the acct file for append.
 *
 *	Opens a (new) acct file. 
 *	If a acct file is already open, and the new file is successfully opened,
 *	the old file is closed.  Otherwise the old file is left open.
 */

int acct_open(filename)
	char *filename;		/* abs pathname or NULL */
{
	char  filen[_POSIX_PATH_MAX];
	char  logmsg[_POSIX_PATH_MAX+80];
	FILE *newacct;
	time_t now;
	struct tm *ptm;

	if (filename == (char *)0) {	/* go with default */
		now = time(0);
		ptm = localtime(&now);
		(void)sprintf(filen, "%s%04d%02d%02d",
			      path_acct,
			      ptm->tm_year+1900, ptm->tm_mon+1, ptm->tm_mday);
		filename = filen;
		acct_auto_switch = 1;
		acct_opened_day = ptm->tm_yday;
	} else if (*filename == '\0') {	/* a null name is not an error */
		return (0);		/* turns off account logging.  */
	} else if (*filename != '/') {
		return (-1);		/* not absolute */
	}
	if ((newacct = fopen(filename, "a")) == NULL) {
		log_err(errno, "acct_open", filename);
		return (-1);
	}
	setbuf(newacct, NULL);		/* set no buffering */

	if (acct_opened > 0) 		/* if acct was open, close it */
		(void)fclose(acctfile);
		
	acctfile = newacct;
	acct_opened = 1;			/* note that file is open */
	(void)sprintf(logmsg, "Account file %s opened", filename);
	log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, "Act", logmsg);

	return (0);
}

/*
 * acct_close - close the current open log file
 */

void acct_close()
{
	if (acct_opened == 1) {
		(void)fclose(acctfile);
		acct_opened = 0;
	}
}

/*
 * account_record - write basic accounting record
 */

void account_record(acctype, pjob, text)
	int 	 acctype;	/* accounting record type */
	job	*pjob;
	char	*text;		/* text to log, may be null */
{
	struct tm *ptm;

	if (acct_opened == 0)
		return;		/* file not open, don't bother */

	ptm = localtime(&time_now);

	/* Do we need to switch files */

	if (acct_auto_switch && (acct_opened_day != ptm->tm_yday)) {
		acct_close();
		acct_open((char *)0);
	}
	if (text == (char *)0)
		text = "";

	(void)fprintf(acctfile,
		      "%02d/%02d/%04d %02d:%02d:%02d;%c;%s;%s\n",
		      ptm->tm_mon+1, ptm->tm_mday, ptm->tm_year+1900,
		      ptm->tm_hour, ptm->tm_min, ptm->tm_sec,
		      (char)acctype,
		      pjob->ji_qs.ji_jobid,
		      text);
}

/*
 * account_jobstr - write a job start record
 */

void account_jobstr(pjob)
	job	*pjob;
{
	char	  buf[PBS_ACCT_MAX_RCD+1];

	/* pack in general information about the job */

	(void)acct_job(pjob, buf);
	buf[PBS_ACCT_MAX_RCD] = '\0';
	account_record(PBS_ACCT_RUN, pjob, buf);
}

/*
 * account_jobend - write a job termination/resource usage record
 */

void account_jobend(pjob, used)
	job	*pjob;
	char	*used;	/* job usage information, see req_jobobit() */
{
	char	  buf[PBS_ACCT_MAX_RCD+1];
	char	 *pb;
	
	/* pack in general information about the job */

	pb = acct_job(pjob, buf);

	/* session */
	(void)sprintf(pb, "session=%ld ",
		      pjob->ji_wattr[(int)JOB_ATR_session_id].at_val.at_long);
	pb += strlen(pb);

	/* Alternate id if present */

	if (pjob->ji_wattr[(int)JOB_ATR_altid].at_flags & ATR_VFLAG_SET) {
		(void)sprintf(pb, "alt_id=%s ",
			pjob->ji_wattr[(int)JOB_ATR_altid].at_val.at_str);
		pb += strlen(pb);
	}

	/* add the execution ended time */
	(void)sprintf(pb, "end=%d ", time_now);
	pb += strlen(pb);

	/* finally add on resources used from req_jobobit() */

	(void)strncat(pb, used, PBS_ACCT_MAX_RCD - (pb - buf));
	buf[PBS_ACCT_MAX_RCD] = '\0';
	
	account_record(PBS_ACCT_END, pjob, buf);
}
