/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/

#include <pbs_config.h>   /* the master config generated by configure */

#include "portability.h"
#include "list_link.h"
#ifndef NDEBUG
#include <stdio.h>
#endif

static char ident[] = "@(#) $RCSfile: list_link.c,v $ $Revision: 2.1 $";

/*
 * list_link.c - general routines for maintenance of a double
 *	linked list.  A user defined structure can be managed as
 *	a double linked list if the first element in the user structure
 *	is the "list_link" struct defined in list_link.h and the list
 *	is headed by a "list_link" struct also defined in list_link.h.
 *
 *	There are the routines provided:
 *		insert_link - inserts a new entry before or after an old
 *		append_link - adds a new entry to the end of the list
 *		delete_link - removes an entry from the list
 *		is_linked   - returns 1 if entry is in the list
 */


/*
 * insert_link - adds a new entry to a list.
 *	Entry is added either before (position=0) or after (position !=0)
 *	an old entry.
 */

void insert_link(old, new, pobj, position)
	struct list_link *old;			/* ptr to old entry in list */
	struct list_link *new;			/* ptr to new link entry    */
	void		 *pobj;			/* ptr to object to link in */
	int 		  position;		/* 0=before old, else after */
{

#ifndef NDEBUG
	/* first make sure unlinked entries are pointing to themselves	    */

	if ( (pobj          == (void *)0)	      ||
	     (old           == (struct list_link *)0) ||
	     (old->ll_prior == (list_link *)0)        ||
	     (old->ll_next  == (list_link *)0)        ||
	     (new->ll_prior != (list_link *)new)      ||
	     (new->ll_next  != (list_link *)new) )  {
		(void)fprintf(stderr, "Assertion failed, bad pointer in insert_link\n");
		abort();
	}
#endif

	if (position == LINK_INSET_AFTER) {  /* insert new after old */
		new->ll_prior = old;
		new->ll_next  = old->ll_next;
		(old->ll_next)->ll_prior = new;
		old->ll_next = new;
	} else {				/* insert new before old */
		new->ll_next = old;
		new->ll_prior = old->ll_prior;
		(old->ll_prior)->ll_next = new;
		old->ll_prior = new;
	}
	/*
	 * its big trouble if ll_struct is null, it would make this
	 * entry appear to be the head, so we never let that happen
	 */
	if (pobj)
		new->ll_struct = pobj;
	else
		new->ll_struct = (void *)new;
}

/*
 * append_link - append a new entry to the end of the list
 */

void append_link(head, new, pobj)
	list_head *head;	/* ptr to head of list	    */
	list_link *new;		/* ptr to new entry	    */
	void	  *pobj;	/* ptr to object to link in */
{

#ifndef NDEBUG
	/* first make sure unlinked entries are pointing to themselves	    */

	if ( (pobj == (void *)0) 		  ||
	     (head->ll_prior == (list_link *)0)   ||
	     (head->ll_next  == (list_link *)0)   ||
	     (new->ll_prior  != (list_link *)new) ||
	     (new->ll_next   != (list_link *)new) )   {
		(void)fprintf(stderr, "Assertion failed, bad pointer in insert_link\n");
		abort();
	}
#endif

	(head->ll_prior)->ll_next = new;
	new->ll_prior = head->ll_prior;
	new->ll_next  = head;
	head->ll_prior = new;
	/*
	 * its big trouble if ll_struct is null, it would make this
	 * entry appear to be the head, so we never let that happen
	 */
	if (pobj)
		new->ll_struct = pobj;
	else
		new->ll_struct = (void *)new;
}

/*
 * delete_link - delete an entry from the list
 *
 *	Checks to be sure links exist before breaking them
 *	Note: the old entry is unchanged other than the list links
 *	are cleared.
 */

void delete_link(old)
	struct list_link *old;		/* ptr to link to delete */
{

	if ((old->ll_prior != (list_link *)0) && 
	    (old->ll_prior != old) && (old->ll_prior->ll_next == old))
		(old->ll_prior)->ll_next = old->ll_next;

	if ((old->ll_next != (list_link *)0) &&
	    (old->ll_next != old) && (old->ll_next->ll_prior == old))
		(old->ll_next)->ll_prior = old->ll_prior;

	old->ll_next  = old;
	old->ll_prior = old;
}

/*
 * swap_link - swap the positions of members of a list
 */

void swap_link(pone, ptwo)
	list_link *pone;
	list_link *ptwo;
{
	list_link *p1p;
	list_link *p2p;


	if (pone->ll_next == ptwo) {
		delete_link(pone);
		insert_link(ptwo, pone, pone->ll_struct, LINK_INSET_AFTER);
	} else if (ptwo->ll_next == pone) {
		delete_link(ptwo);
		insert_link(pone, ptwo, ptwo->ll_struct, LINK_INSET_AFTER);
	} else {
		p1p = pone->ll_prior;
		p2p = ptwo->ll_prior;
		delete_link(pone);
		insert_link(p2p, pone, pone->ll_struct, LINK_INSET_AFTER);
		delete_link(ptwo);
		insert_link(p1p, ptwo, ptwo->ll_struct, LINK_INSET_AFTER);
	}
}

/*
 * is_linked - determine if entry is in the list 
 *
 * Returns: 1 if in list
 *	    0 if not in list
 */

int is_linked(head, entry)
	list_link *head;
	list_link *entry;
{
	list_link *pl;

	pl = head->ll_next;
	while (pl != head) {
		if (pl == entry)
			return (1);
		pl = pl->ll_next;
	}
	return (0);
}

/*
 * The following routines are replaced by in-line code with the
 * GET_NEXT / GET_PRIOR macroes when NDEBUG is defined, see list_link.h
 */

#ifndef NDEBUG

void *get_next(pl, file, line)
	list_link  pl;
	char	  *file;
	int	   line;
{
	if ((pl.ll_next == (list_link *)0) ||
	    ((pl.ll_next == &pl) && (pl.ll_struct != (void *)0))) {
		(void)fprintf(stderr, "Assertion failed, bad pointer in link: file \"%s\", line %d\n", file, line);
		abort();
	}
	return (pl.ll_next->ll_struct);
}

void *get_prior(pl, file, line)
	list_link  pl;
	char	  *file;
	int	   line;
{
	if ((pl.ll_prior == (list_link *)0) ||
	    ((pl.ll_prior == &pl) && (pl.ll_struct != (void *)0))) {
		(void)fprintf(stderr, "Assertion failed, null pointer in link: file \"%s\", line %d\n", file, line);
		abort();
	}
	return (pl.ll_prior->ll_struct);
}
#endif

/*
 * list_move - move an entire list from one head to another 
 */

void list_move(from, to)
	list_head *from;
	list_head *to;
{
	if (from->ll_next == from) {
		to->ll_next = to;
		to->ll_prior = to;
	} else {
		to->ll_next = from->ll_next;
		to->ll_next->ll_prior = to;
		to->ll_prior = from->ll_prior;
		to->ll_prior->ll_next = to;
		CLEAR_HEAD((*from));
	}
}
