/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/




/*
 * node_func.c - various functions dealing with nodes, properties and
 *		 the following global variables:
 *	pbsnlist     -	the server's global node list 
 *	svr_totnodes -	total number of pbshost entries
 *	svr_clnodes  -	number of cluster (space-shared) nodes 
 *	svr_tsnodes  -	number of time-shared nodes, one per host
 *
 * Included functions are:
 *	find_nodehbyname() - find a node host with a given name
 */
#include <pbs_config.h>   /* the master config generated by configure */

#include <stdio.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include "pbs_ifl.h"
#include "libpbs.h"
#include "list_link.h"
#include "attribute.h"
#include "credential.h"
#include "batch_request.h"
#include "server_limits.h"
#include "server.h"
#include "job.h"
#include "pbs_nodes.h"
#include "pbs_error.h"
#include "log.h"

#if !defined(H_ERRNO_DECLARED)
extern int h_errno;
#endif


static char ident[] = "@(#) $RCSfile: node_func.c,v $ $Revision: 2.8 $";

/* Global Data */

extern int	 svr_totnodes;
extern int	 svr_tsnodes;
extern int	 svr_clnodes;
extern char	*path_nodes_new;
extern char	*path_nodes;
extern char	*path_nodestate;


/* Functions in this file
 * find_nodehbyname()   -     given a node host name, search pbsndlist 
 * find_subnodebyname() -     given a subnode name
 * save_characteristic() - save the the characteristics of the node along with
 *		the address of the node
 * chk_characteristic() -  check for changes to the node's set of
 *		characteristics and set appropriate flag bits in the "need_todo"
 *		location depending on which characteristics changed
 * status_nodeattrib() -    add status of each requested (or all) node-attribute
 *		to the status reply
 * initialize_pbsnode() -   performs node initialization on a new node
 * effective_node_delete() -  effectively deletes a node from the server's node
 *		list by setting the node's "deleted" bit
 * setup_notification() -   sets mechanism for notifying other hosts about a new
 *		host
 * process_host_name_part() - processes hostname part of a batch request into a
 *		prop structure, host's IP addresses into an array, and node
 *		node type (cluster/time-shared) into an int variable
 * update_nodes_file() -    used to update the nodes file when certain changes
 *		occur to the server's internal nodes list
 * recompute_ntype_cnts -   Recomputes the current number of cluster nodes and
 *		current number of time-shared nodes
 * create_pbs_node -	create basic node structure for adding a node
 */



/*
 * find_nodehbyname() - find a node host by its name
 */

struct pbsnode	  *find_nodebyname(nodename)
	char *nodename;
{
	int		i;
	char		*pslash;
	struct pbsnode *pnode;

	if (pslash = strchr(nodename, (int)'/'))
		*pslash = '\0';
	pnode = (struct pbsnode *)0;
	for(i=0; i<svr_totnodes; i++ ) {
		if(pbsndlist[i]->nd_state & INUSE_DELETED)
			continue;

		if( strcasecmp(nodename, pbsndlist[i]->nd_name) == 0 ) {
		    pnode = pbsndlist[i];
		    break;
		}
	}
	if (pslash)
		*pslash = '/';	/* restore the slash */
	return (pnode);
}

/*
 * find_subnodebyname() - find a subnode by its (parent's) name
 *
 *	returns first of N subnodes for a given node name
 */

struct pbssubn *find_subnodebyname(nodename)
	char *nodename;
{
	struct pbsnode  *pnode;

	if ((pnode = find_nodebyname(nodename)) == NULL)
		return (NULL);

	return pnode->nd_psn;
}

	


static struct pbsnode	*old_address = 0;			/*node in question */
static struct prop	*old_first = (struct prop *)0xdead;	/*node's first prop*/
static short		old_state = (short)0xdead;	/*node's   state   */
static short		old_ntype = (short)0xdead;	/*node's   ntype   */
static int		old_nprops = 0xdead;		/*node's   nprops  */


/*
 * save_characteristic() -  save the characteristic values of the node along
 *			    with the address of the node
 */

void	save_characteristic( pnode )
	struct pbsnode	*pnode;
{
	if( pnode == (struct pbsnode *)0 )
	    return;

	old_address = pnode;
	old_state =   pnode->nd_state;
	old_ntype =   pnode->nd_ntype;
	old_nprops =  pnode->nd_nprops;
	old_first =   pnode->nd_first;
	
}

/*
 * chk_characteristic() -  check the value of the characteristics against
 *		   that which was saved earlier.
 *		   Returns:
 *		   -1  if parent address doesn't match saved parent address
 *		    0  if successful check.  *pneed_todo gets appropriate
 *		       bit(s) set depending on the results of the check.
 *		       The "returned" bits get used by the caller.
 */


int	chk_characteristic( pnode, pneed_todo )
	struct pbsnode	*pnode;
	int		*pneed_todo;
{
	int		rc;
	short		tmp;

	if( pnode != old_address || pnode == (struct pbsnode *)0 ) {
	  /*didn't do save_characteristic() before issuing chk_characteristic()*/

	    old_address = (struct pbsnode *)0;
	    return (-1);
	}

	tmp = pnode -> nd_state; 
	if (tmp != old_state) {

	   if (tmp & INUSE_OFFLINE && !(old_state & INUSE_OFFLINE))
	       *pneed_todo |= WRITENODE_STATE;		/*marked offline */

	   if (!(tmp & INUSE_OFFLINE) && old_state & INUSE_OFFLINE)
	       *pneed_todo |= WRITENODE_STATE;		/*removed offline*/

	   if (tmp & INUSE_DELETED && old_state & INUSE_OFFLINE)
	       *pneed_todo |= WRITENODE_STATE;	       /*offline & now deleted*/

	   if (tmp & INUSE_DELETED && !(old_state & INUSE_DELETED))
	       *pneed_todo |= WRITE_NEW_NODESFILE;	/*node being deleted*/
	}

	tmp = pnode -> nd_ntype;
	if (tmp != old_ntype )
	    *pneed_todo |= WRITE_NEW_NODESFILE;

	if (old_nprops != pnode->nd_nprops || old_first != pnode->nd_first)
	    *pneed_todo |= WRITE_NEW_NODESFILE;

	old_address = (struct pbsnode *)0;
	return  0;
}



/* status_nodeattrib() - add status of each requested (or all) node-attribute to
 *			 the status reply
 *
 *      Returns:     0	is success
 *                != 0	is error, if a node-attribute is incorrectly specified, *bad is
 *			set to the node-attribute's ordinal position 
 */

int status_nodeattrib(pal, padef, pnode, limit, priv, phead, bad)
        svrattrl        *pal;	/*an svrattrl from the request  */
        attribute_def   *padef;	/*the defined node attributes   */
        struct pbsnode  *pnode;	/*no longer an attribute ptr	*/
        int              limit;	/*number of array elts in padef */
        int              priv;	/*requester's privilege		*/

        list_head       *phead;	/*heads list of svrattrl structs that hang */
				/*off the brp_attr member of the status sub*/
				/*structure in the request's "reply area"  */

        int             *bad;	/*if node-attribute error record it's*/
				/*list position here                 */
{
	int   i;
	int   rc = 0;		/*return code, 0 == success*/
        int   index;
        int   nth;		/*tracks list position (ordinal tacker)   */

	attribute	atemp[ND_ATR_LAST];	/*temporary array of attributes	  */



        priv &= ATR_DFLAG_RDACC;  		/* user-client privilege          */
	
	for (i=0; i<ND_ATR_LAST; i++) {   /*set up attributes using data from node*/

	     if ( !strcmp ((padef + i)->at_name, ATTR_NODE_state) )
		  atemp[i].at_val.at_short = pnode->nd_state;
	     else if ( !strcmp ((padef + i)->at_name, ATTR_NODE_properties) )
		  atemp[i].at_val.at_arst = pnode->nd_prop;
	     else if ( !strcmp ((padef + i)->at_name, ATTR_NODE_ntype) )
		  atemp[i].at_val.at_short = pnode->nd_ntype;
	     else if ( !strcmp ((padef + i)->at_name, ATTR_NODE_jobs) )
		  atemp[i].at_val.at_jinfo = pnode;
	     else if ( !strcmp ((padef + i)->at_name, ATTR_NODE_np) )
		  atemp[i].at_val.at_long = pnode->nd_nsn;
	     else {/*we don't ever expect this*/
		  *bad = 0; 
		  return (PBSE_UNKNODEATR);
	     }
	     atemp[i].at_flags = ATR_VFLAG_SET; /*artificially set the value's flags*/
	}


        if (pal) {        /*caller has requested status on specific node-attributes*/
	    nth = 0;
            while (pal) {
                ++nth;
                index = find_attr(padef, pal->al_name, limit);
                if (index < 0) {
                        *bad = nth;		/*name in this position can't be found*/
                        rc = PBSE_UNKNODEATR;
			break;
                }
                if ((padef+index)->at_flags & priv) {
                        if ( rc = (  (padef+index)->at_encode( &atemp[index], phead,
                                        (padef+index)->at_name,
                                        (char *)0, ATR_ENCODE_CLIENT )  ) < 0  ) {
			      
			      rc = -rc;
			      break;
			}
                }
                pal = (svrattrl *)GET_NEXT(pal->al_link);
            }

        } else {        /* non-specific request, return all readable attributes    */

                for (index = 0; index < limit; index++) {
                    if (((padef+index)->at_flags & priv) &&
                        !((padef+index)->at_flags & ATR_DFLAG_NOSTAT)) {
                           if ( rc = (  (padef+index)->at_encode( &atemp[index], phead,
                                        (padef+index)->at_name,
                                        (char *)0, ATR_ENCODE_CLIENT )  ) < 0  ) {
			      
			      rc = -rc;
			      break;
			   }
                    }
                }
        }

        return (rc);
}


/*
 * initialize_pbsnode - carries out initialization on a new
 *	pbs node.  The assumption is that all the parameters are valid.
*/

static void initialize_pbsnode (pnode, pname, pul, ntype)
	struct pbsnode	*pnode;
	char		*pname;     /* node name */
	u_long		*pul;	    /*host byte order array */
				    /*ipaddrs for this node */
	int		ntype;	    /*time-shared or cluster*/
{
	
	pnode->nd_name    = pname;
	pnode->nd_stream  = -1;
	pnode->nd_addrs   = pul;	    /*list of host byte order */
	pnode->nd_ntype   = ntype;
	pnode->nd_nsn     = 0;
	pnode->nd_nsnfree = 0;
	pnode->nd_needed  = 0;
	pnode->nd_order   = 0;
	pnode->nd_prop	  = (struct array_strings *)0;
	pnode->nd_psn     = NULL;
	pnode->nd_state   = INUSE_UNKNOWN | INUSE_NEEDS_HELLO_PING | INUSE_DOWN;
	pnode->nd_first	  = init_prop(pnode->nd_name);
	pnode->nd_last	  = pnode->nd_first;
	pnode->nd_nprops  = 0;
}

/*
 * subnode_delete - delete the specified subnode
 *	by marking it deleted
 */

static void subnode_delete(psubn)
	struct pbssubn *psubn;
{
	struct jobinfo	*jip, *jipt;

	for (jip = psubn->jobs; jip; jip = jipt) {
       		jipt = jip->next;
       		free (jip); 
	}
	psubn->host  = (struct pbsnode *)0;
	psubn->jobs  = (struct jobinfo *)0;
	psubn->next  = (struct pbssubn *)0;
	psubn->inuse = INUSE_DELETED; 
}


void	effective_node_delete (pnode)
	struct pbsnode	*pnode;
{
	struct pbssubn  *psubn;
	struct pbssubn  *pnxt;
	u_long		*up;

	psubn = pnode->nd_psn;
	while (psubn) {
		pnxt = psubn->next;
		subnode_delete(psubn);
		psubn = pnxt;
	}

	pnode->nd_last->next = (struct prop *)0; /*just in case*/
	pnode->nd_last = (struct prop *)0;
	free_prop_list (pnode->nd_first);
	pnode->nd_first = (struct prop *)0;

	for (up = pnode->nd_addrs; *up; up++) {
	    /* del node's IP addresses from tree  */
	    tdelete(*up, &ipaddrs);
	}
	if (pnode->nd_addrs) {	/* remove array of IP addresses */
	    free (pnode->nd_addrs);
	    pnode->nd_addrs = (u_long *)0;
	}

	tdelete((u_long)pnode->nd_stream, &streams); /*take stream out of tree*/
	(void )rpp_close (pnode->nd_stream);
	(void)free(pnode->nd_name);
	pnode->nd_name = NULL;
	pnode->nd_stream = -1;
	pnode->nd_state  = INUSE_DELETED;
	pnode->nd_nsn    = 0;
	pnode->nd_nsnfree= 0;
}



/*
 *	setup_notification -  Sets up the  mechanism for notifying
 *			      other members of the server's node
 *			      pool that a new node was added manually
 *			      via qmgr.  Actual notification occurs some
 *			      time later through the ping_nodes mechanism
*/

void	setup_notification ()
{
	int	i;
	for (i=0; i<svr_totnodes; i++) {
	     if (pbsndlist[i]->nd_state & INUSE_DELETED)
		 continue;
	     else
		 pbsndlist[i]->nd_state |= INUSE_NEEDS_HELLO_PING;
	}
}



static int process_host_name_part (objname, pul, pname, ntype)

	char		 *objname;	/* node to be's name */
					/*THINGS RETURNED*/
	u_long		**pul;		/*0 terminated host adrs array*/
	char		**pname;	/*node name w/o any :ts       */
	int		 *ntype;	/*node type; time-shared, not */
{
	struct hostent	*hp;
	struct in_addr  addr;
	char		*phostname;     /*caller supplied hostname   */
	u_long		ipaddr;		/*ip address in host order   */
	int		len, i;


	len = strlen(objname);
	if (len == 0)
	    return  (PBSE_UNKNODE);

	phostname = strdup (objname);

	if (phostname == (char *)0 )
            return  (PBSE_SYSTEM);

	*ntype = NTYPE_CLUSTER;
	if (len >= 3 && !strcmp (&phostname[len-3], ":ts")) {
            phostname[len-3] = '\0';
	    *ntype = NTYPE_TIMESHARED;
	}

	if ((hp = gethostbyname(phostname)) == NULL) {
	     sprintf(log_buffer, "host %s not found", objname);
	     free (phostname);
	     return  (PBSE_UNKNODE);
	}

	memcpy((char *)&addr, hp->h_addr, hp->h_length);
	if (hp->h_addr_list[1] == NULL) {/*weren't given cannonical name*/
	    char	*hname;

	    if ((hp = gethostbyaddr((void *)&addr,  sizeof(struct in_addr),
						hp->h_addrtype)) == NULL) {

	         sprintf(log_buffer, "addr not found h_errno=%d errno=%d",
				     h_errno, errno);
	         free (phostname);
	         return  (PBSE_UNKNODE);
	    }
	
	    hname = (char *)strdup(hp->h_name); /*cannonical name in theory*/
	    if (hname == (char *)0) {
	        free (phostname);
	        return  (PBSE_SYSTEM);
	    }

	    if ((hp = gethostbyname(hname)) == NULL) {
	         sprintf(log_buffer, "bad cname %s, h_errno=%d errno=%d",
				     hname, h_errno, errno);
		 free (hname);
	         free (phostname);
	         return  (PBSE_UNKNODE);
	    }
	    free(hname);
	}

	for (i=0; hp->h_addr_list[i]; i++);	/*count how many ipadrs*/

	*pul = (u_long *)malloc(sizeof(u_long) * (i+1));  /*null end it*/
	if (*pul == (u_long *)0) {
	     free (phostname);
	}

	for (i=0; hp->h_addr_list[i]; i++) {
	     u_long	ipaddr;

	     memcpy((char *)&addr, hp->h_addr_list[i], hp->h_length);
	     ipaddr = ntohl(addr.s_addr);
	     (*pul)[i] = ipaddr;
	}

	(*pul)[i] = 0;				/* null term array ip adrs*/
	*pname = phostname;			/* return node name	  */

	return  (0);				/* function successful    */
}



/*
 *	update_nodes_file - When called, this function will update
 *			    the nodes file.  Specifically, it will
 *			    walk the server's array of pbsnodes
 *			    constructing for each entry a nodes file
 *			    line if that entry is not marked as deleted.
 *			    These are written to a temporary file.
 *			    Upon successful conclusion that file replaces
 *			    the nodes file.
*/			    

int	update_nodes_file ()
{
	static	char	id[] = "update_nodes_file";
	struct pbsnode  *np;
	int	i, j;
	FILE	*nin;

	DBPRT(("%s: entered\n", id))


	if ((nin = fopen(path_nodes_new, "w")) == NULL) {
		log_event(PBSEVENT_ADMIN, PBS_EVENTCLASS_SERVER, "nodes",
			  "Node description file update failed");
		return (-1);
	}

	if (svr_totnodes == 0 || pbsndmast == (struct pbsnode **)0) {
		log_event(PBSEVENT_ADMIN, PBS_EVENTCLASS_SERVER, "nodes",
			  "Server has empty nodes list");
		fclose (nin);
		return (-1);
	}

	/* for each node ... */

	for (i=0; i<svr_totnodes; ++i) {

	    np = pbsndmast[i];
	    if (np->nd_state & INUSE_DELETED)
		 continue;

	    /* ... write its name, and if time-shared, append :ts */

	    fprintf(nin, "%s", np->nd_name);	/* write name */
	    if (np->nd_ntype == NTYPE_TIMESHARED)
		fprintf(nin, ":ts"); 

	    /* if number of subnodes is gt 1, write that; if only one,   */
	    /* don't write to maintain compatability with old style file */

	    if (np->nd_nsn > 1)
	    	fprintf(nin, " %s=%d", ATTR_NODE_np, np->nd_nsn);

	    /* write out properties */

	    for (j=0; j < np->nd_nprops-1; ++j) {
		fprintf(nin, " %s", np->nd_prop->as_string[j]);
	    }
	    /* finish off line with new-line */
	    fprintf(nin, "\n");
	    fflush (nin);

	    if (ferror(nin)) {

		log_event(PBSEVENT_ADMIN, PBS_EVENTCLASS_SERVER, "nodes",
			  "Node description file update failed");
		fclose (nin);
		return (-1);
	     }
	}
	fclose (nin);

        (void)unlink(path_nodes);
        (void)link(path_nodes_new, path_nodes);
        (void)unlink(path_nodes_new);
	return (0);
}


/*
 *	recompute_ntype_cnts - Recomputes the current number of cluster
 *			       nodes and current number of time-shared nodes
 */
void	recompute_ntype_cnts ()
{
	int		 svr_loc_clnodes = 0;
	int		 svr_loc_tsnodes = 0;
	int		 i;
	struct	pbsnode  *pnode;

	if (svr_totnodes) {
	    for (i=0; i<svr_totnodes; ++i) {

		 pnode = pbsndlist[i];
		 if (pnode->nd_state & INUSE_DELETED)
		     continue;

		 if (pnode->nd_ntype == NTYPE_CLUSTER)
		     svr_loc_clnodes += pnode->nd_nsn;
		 else if (pnode->nd_ntype == NTYPE_TIMESHARED)
		     svr_loc_tsnodes++;
	    }
	    svr_clnodes = svr_loc_clnodes;
	    svr_tsnodes = svr_loc_tsnodes;
	}
}

/*
 * init_prop - allocate and initialize a prop struct
 *
 *	pname points to the property string
 */

struct prop *init_prop(pname)
	char	    *pname;
{
	struct prop *pp;

	if ((pp = (struct prop *)malloc(sizeof(struct prop))) != NULL) {
		pp->name    = pname;
		pp->mark    = 0;
		pp->next    = 0;
	}
	
	return (pp);
}
	

/*
 * create_subnode - create a subnode entry and link to parent node
 *
 *	Note, pname arg must be a copy of prop list as it is linked directly in
 */
static struct pbssubn *create_subnode(pnode)
	struct pbsnode	*pnode;
{
	struct pbssubn  *psubn;
	struct pbssubn **nxtsn;
	struct prop	*plastp;
	

	psubn = (struct pbssubn *)malloc(sizeof(struct pbssubn));
	if (psubn == (struct pbssubn *)0) {
		return (NULL);
	}

	/* initialize the subnode and link into the parent node */

	psubn->host  = pnode;
	psubn->next  = (struct pbssubn *)NULL;
	psubn->jobs  = (struct jobinfo *)NULL;
	psubn->flag  = okay;
	psubn->inuse = 0;
	psubn->index = pnode->nd_nsn++;
	pnode->nd_nsnfree++;
	if ((pnode->nd_state & (INUSE_JOB|INUSE_JOBSHARE)) != 0)
		pnode->nd_state = INUSE_FREE;
	psubn->allocto = (resource_t)0;

	nxtsn = &pnode->nd_psn;	   /* link subnode onto parent node's list */
	while (*nxtsn)
		nxtsn = &((*nxtsn)->next);
	*nxtsn = psubn;

	return (psubn);
}

/*
 * create_pbs_node - create pbs node structure, i.e. add a node
 */

int create_pbs_node(objname, plist, perms, bad)
	char		*objname;
	svrattrl	*plist;
	int		 perms;
	int		*bad;
{
	struct pbsnode	*pnode;
	struct pbsnode **tmpndlist;
	int		ntype;		/*node type; time-shared, not */
	char		*pname;		/*node name w/o any :ts       */
	u_long		*pul;		/*0 terminated host adrs array*/
	int		 rc;
	int		 iht;

	if (rc = process_host_name_part (objname, &pul, &pname, &ntype)) {
	    return (rc);
	}

	if (find_nodebyname(pname)) {
		free (pname);
		free (pul);
		return (PBSE_NODEEXIST);
	}

	for (iht=0; iht<svr_totnodes; iht++) {
	     if (pbsndmast[iht]->nd_state & INUSE_DELETED) { /*available, use*/
		 pnode = pbsndmast[iht];
		 break;
	     }
	}

	if (iht == svr_totnodes) {      /*no unused entry, make an entry*/

	    pnode = (struct pbsnode *)malloc(sizeof(struct pbsnode));
	    if (pnode == (struct pbsnode *)0) {
	        free(pul);
		free (pname);

		return (PBSE_SYSTEM);
	    }
	    pnode->nd_state = INUSE_DELETED;

	    /* expand pbsndmast array exactly svr_totnodes long*/

	    tmpndlist = (struct pbsnode **)realloc(pbsndmast,
		       sizeof(struct pbsnode*) * (svr_totnodes + 1) );

	    if (tmpndlist != (struct pbsnode **)0) {/*add in the new entry etc*/
		pbsndmast = tmpndlist;
		pbsndmast[svr_totnodes++] = pnode;
	    } else {
		free(pnode);
		free(pul);
		free (pname);
		return (PBSE_SYSTEM);
	    }
	    tmpndlist = (struct pbsnode **)realloc(pbsndlist, 
			sizeof(struct pbsnode*) * (svr_totnodes + 1) );
	    if (tmpndlist != (struct pbsnode **)0) {
		memcpy(tmpndlist, pbsndmast, svr_totnodes * sizeof (struct pbsnode*));
		pbsndlist = tmpndlist;
	    } else {
		free(pnode);
		free(pul);
		free (pname);
		return (PBSE_SYSTEM);
	    }
	}
	initialize_pbsnode (pnode, pname, pul, ntype);


	/* create and initialize the first subnode to go with the parent node */

	if (create_subnode(pnode) == NULL) {
		pnode->nd_state = INUSE_DELETED;
		free(pul);
		free (pname);
		return (PBSE_SYSTEM);
	}

	rc = mgr_set_node_attr (pnode, node_attr_def, ND_ATR_LAST,
			        plist, perms, bad, (void *)pnode,
				ATR_ACTION_ALTER);

	if (rc != 0) {
	    effective_node_delete (pnode);
	    return (rc);
	}

	recompute_ntype_cnts ();

	return(PBSE_NONE);	    /*create completely successful*/
}


/*
 * parse_node_token - parse tokens in the nodes file
 *
 *	Token is returned, if null then there was none.
 *	If there is an error, then "err" is set non-zero.
 *	On following call, with argument "start" as null pointer, then 
 *	resume where left off.
 *
 *	If "cok" is true, then this is first token (node name) and ':' is
 *	allowed and '=' is not.   For following tokens, allow '=' as separator
 *	between "keyword" and "value".  Will get value as next token.
 */

static char *parse_node_token(start, cok, err, term)
	char *start;	/* if null, restart where left off */
	int   cok;	/* flag - non-zero if colon ":" allowed in token */
	int  *err;	/* RETURN: non-zero if error */
	char *term;	/* RETURN: character terminating token */
{
	static char *pt;
	char        *ts;

	*err = 0;
	if (start)
		pt = start;

	while (*pt && isspace((int)*pt))	/* skip leading whitespace */
		pt++;
	if (*pt == '\0')
		return (NULL);		/* no token */

	ts = pt;
	
	/* test for legal characters in token */

	for ( ; *pt; pt++) {
		if ( isalnum((int)*pt) || (*pt == '-') || (*pt == '.') )
			continue;
		else if (isspace((int)*pt))
			break;
		else if (cok && (*pt == ':'))
			continue;
		else if ( !cok && (*pt == '='))
			break;
		else
			*err = 1;
	}
	*term = *pt;
	*pt++   = '\0';
	return (ts);
}
			




/*
**	Read the file, "nodes", containing the list of properties for each node.
**	The list of nodes is formed with pbsndmast (also pbsndlist) as the head.
**	Return -1 on error, 0 otherwise.
**
**	Read the node state file, "node_state", for any "offline"
**	conditions which should be set in the nodes. 
**
**	If routine returns -1, then "log_buffer" contains a message to
**	be logged.
*/

int
setup_nodes()
{
	static	char	id[] = "setup_nodes";
	FILE	 *nin;
	char	  line[256];
	char	 *nodename;
	char	  propstr[256];
	char	 *token;
	int	  bad, i, num, linenum;
	int	  err;
	struct pbsnode *np;
	char     *val;
	char      xchar;
	svrattrl *pal;
	int	  perm = ATR_DFLAG_MGRD|ATR_DFLAG_MGWR;
	list_head atrlist;

	extern char server_name[];
	extern resource_t next_resource_tag;

	DBPRT(("%s: entered\n", id))
	CLEAR_HEAD(atrlist);

	if ((nin = fopen(path_nodes, "r")) == NULL) {
		log_event(PBSEVENT_ADMIN, PBS_EVENTCLASS_SERVER, server_name,
			  "No Node description file found in setup_nodes");
		return(0);
	}

	next_resource_tag = time(0);	/* initialize next resource handle */

	tfree(&streams);
	tfree(&ipaddrs);

	svr_totnodes = 0;
	for (linenum=1; fgets(line, sizeof(line), nin); linenum++) {
		if (line[0] == '#')	/* comment */
			continue;

		/* first token is the node name, may have ":ts" appended */

		propstr[0] = '\0';
		token = parse_node_token(line, 1, &err, &xchar);
		if (token == NULL)
			continue;	/* blank line */
		if (err) {
			sprintf(log_buffer,
			      "invalid character in token \"%s\" on line %d",
			      token, linenum);
			goto errtoken2;
		}
		if (!isalpha((int)*token)) {
			sprintf(log_buffer,
			     "token \"%s\" doesn't start with alpha on line %d",
			     token, linenum);
			goto errtoken2;
		}
		nodename = token;

		/* now process remaining tokes (if any), they may be either */
		/* attributes (keyword=value) or old style properties       */

		while (1) {

		    token = parse_node_token(NULL, 0, &err, &xchar);
		    if (err)
			goto errtoken1;
		    if (token == NULL)
			break;

		    if (xchar == '=') {

			/* have new style attribute, keyword=value */

			val = parse_node_token(NULL, 0, &err, &xchar);
			if ((val==NULL) || err || (xchar=='='))
				goto errtoken1;

			pal = attrlist_create(token, 0, strlen(val)+1);
			if (pal == (svrattrl *)0) {
			    strcpy(log_buffer, "cannot create node attribute");
			    goto errtoken2;
			}
			(void)strcpy(pal->al_value, val);
			pal->al_flags = SET;
			append_link(&atrlist, &pal->al_link, pal);

		    } else {

			/* old style properity */

			if (propstr[0] != '\0')
				(void)strcat(propstr, ",");
			(void)strcat(propstr, token);
		    }

		}

		/* if any properities, create properity attr and add to list */

		if (propstr[0] != '\0') {
		    pal = attrlist_create(ATTR_NODE_properties,0,strlen(propstr)+1);
		    if (pal == (svrattrl *)0) {
		        strcpy(log_buffer, "cannot create node attribute");
		        return -1;
		    }
		    (void)strcpy(pal->al_value, propstr);
		    pal->al_flags = SET;
		    append_link(&atrlist, &pal->al_link, pal);
	        }

		/* now create node and subnodes */

		pal = GET_NEXT(atrlist);
		err = create_pbs_node(nodename, pal, perm, &bad);
		if (err == PBSE_NODEEXIST) {
			sprintf(log_buffer,"duplicate node \"%s\"on line %d",
				nodename, linenum);
			goto errtoken2;
		} else if (err) {
			sprintf(log_buffer, "could not create node \"%s\", error = %d", nodename, err);
			goto errtoken2;
		}
		free_attrlist(&atrlist);
	}

	fclose(nin);
	nin = fopen(path_nodestate, "r");
	if (nin != NULL) {
		while (fscanf(nin, "%s %d", line, &num) == 2) {
			for (i=0; i<svr_totnodes; i++) {
				np = pbsndmast[i];

				if (strcmp(np->nd_name, line) == 0) {
					np->nd_state = num;
					break;
				}
			}
		}	
		(void)fclose(nin);
	}

	return(0);

errtoken1:
	sprintf(log_buffer, "token \"%s\" in error on line %d of file nodes",
		token, linenum);
errtoken2:
	free_attrlist(&atrlist);
	fclose(nin);
	return -1;
}


/*
 * delete_a_subnode - mark a (last) single subnode entry as deleted
 */
static void delete_a_subnode(pnode)
	struct pbsnode *pnode;
{
	struct pbssubn *psubn;
	struct pbssubn *pprior = 0;

	psubn = pnode->nd_psn;

	while (psubn->next) {
		pprior = psubn;
		psubn = psubn->next;
	}
	
	/*
	 * found last subnode in list for given node, mark it deleted
	 * note, have to update nd_nsnfree using pnode rather than psubn->host
	 * because it point to the real node rather than the the copy (pnode)
	 * and the real node is overwritten by the copy
	 */
	
	if ((psubn->inuse & (INUSE_JOB|INUSE_JOBSHARE)) == 0)
		pnode->nd_nsnfree--;

	subnode_delete(psubn);
	if (pprior)
		pprior->next = (struct pbssubn *)0;
}

/*
 * node_np_action - action routine for node's np attribute
 */

int	node_np_action( new, pobj, actmode)
	attribute	*new;		/*derive props into this attribute*/
	void		*pobj;		/*pointer to a pbsnode struct     */
	int		 actmode;	/*action mode; "NEW" or "ALTER"   */
{
	struct pbsnode *pnode = (struct pbsnode *)pobj;
	short		old_np;
	short		new_np;

	switch( actmode ) {
 
	    case ATR_ACTION_NEW:        
		new->at_val.at_long = pnode->nd_nsn;
		break;

	    case ATR_ACTION_ALTER:
		old_np = pnode->nd_nsn;
		new_np = (short)new->at_val.at_long;
		if (new_np <= 0)
			return PBSE_BADATVAL;

		while (new_np != old_np) {

		    if (new_np < old_np) {
			delete_a_subnode(pnode); 
			old_np--;
		    } else {
			create_subnode(pnode);
			old_np++;
		    }
		}
		pnode->nd_nsn = old_np;
		break;
	}

	return 0;
}
