/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/

#include <pbs_config.h>   /* the master config generated by configure */

#include 	<stdio.h>
#include 	<stdlib.h>
#include	<unistd.h>
#include 	<string.h>
#include 	<ctype.h>
#include 	<errno.h>
#include	<fcntl.h>
#include	<netdb.h>
#include 	<sys/types.h>
#include	<netinet/in.h>

#include	"portability.h"
#include	"libpbs.h"
#include	"server_limits.h"
#include 	"svrfunc.h"
#include	"list_link.h"
#include	"attribute.h"
#include	"resource.h"
#include	"server.h"
#include	"net_connect.h"
#include	"work_task.h"
#include	"job.h"
#include	"log.h"
#include	"pbs_nodes.h"
#include	"rpp.h"
#include	"dis.h"
#include	"dis_init.h"
#include	"resmon.h"

#if !defined(H_ERRNO_DECLARED)
extern int h_errno;
#endif
static char ident[] = "@(#) $RCSfile: node_manager.c,v $ $Revision: 2.14.2.3 $";

int	 	 svr_totnodes = 0;	/* total number nodes defined       */
int		 svr_clnodes  = 0;	/* number of cluster nodes	    */
int		 svr_tsnodes  = 0;	/* number of time shared nodes	    */
int		 svr_chngNodesfile = 0;	/* 1 signals want nodes file update */
					/* on server shutdown, (qmgr mods)  */
struct pbsnode **pbsndlist = NULL;
struct pbsnode **pbsndmast = NULL;
static int	 svr_numnodes = 0;	/* number nodes currently available */
static int	 exclusive;		/* node allocation type */

static char      extra_parm[] = "extra parameter(s)";
static char      no_parm[]    = "required parameter not found";
static FILE 	*nstatef = NULL;

extern int	 server_init_type;
extern int	 has_nodes;

extern int	ctnodes A_((char *));
extern char	*path_home;
extern char	*path_nodes;
extern char	*path_nodes_new;
extern char	*path_nodestate;

extern struct server server;

#define		SKIP_NONE	0
#define		SKIP_EXCLUSIVE	1
#define		SKIP_ANYINUSE	2

static int hasprop(struct pbsnode *, struct prop *);

/*
 * Tree search generalized from Knuth (6.2.2) Algorithm T just like
 * the AT&T man page says.
 *
 * The tree_t structure is for internal use only, lint doesn't grok it.
 *
 * Written by reading the System V Interface Definition, not the code.
 *
 * Totally public domain.
 */
/*LINTLIBRARY*/

/*
**      Modified by Tom Proett <proett@nas.nasa.gov> for PBS.
*/

typedef struct tree_t {
	u_long		key;
	struct pbsnode	*nodep;
	struct tree_t	*left, *right;
} tree;

tree	*ipaddrs = NULL;	/* tree of ip addrs */
tree	*streams = NULL;	/* tree of stream numbers */

/* find value in tree, return NULL if not found */
struct	pbsnode	*
tfind(key, rootp)
	const u_long	key;		/* key to be located */
	tree		**rootp;	/* address of tree root */
{
	register tree	*q;

	if (rootp == (struct tree_t **)0)
		return NULL;
	while (*rootp != NULL) {		/* Knuth's T1: */
		if (key == (*rootp)->key)	/* T2: */
			return (*rootp)->nodep;	/* we found it! */
		rootp = (key < (*rootp)->key) ?
			&(*rootp)->left :	/* T3: follow left branch */
			&(*rootp)->right;	/* T4: follow right branch */
	}
	return NULL;
}

void
tinsert(key, nodep, rootp)
	const	u_long	key;		/* key to be located */
	struct	pbsnode	*nodep;
	tree		**rootp;	/* address of tree root */
{
	register tree	*q;

	DBPRT(("tinsert: %lu %s stream %d\n", key,
		nodep->nd_name, nodep->nd_stream))

	if (rootp == (struct tree_t **)0)
		return;
	while (*rootp != (struct tree_t *)0) {	/* Knuth's T1: */
		if (key == (*rootp)->key)	/* T2: */
		    return;			/* we found it! */
		rootp = (key < (*rootp)->key) ?
			&(*rootp)->left :	/* T3: follow left branch */
			&(*rootp)->right;	/* T4: follow right branch */
	}
	q = (tree *) malloc(sizeof(tree));	/* T5: key not found */
	if (q != NULL) {			/* make new node */
		*rootp = q;			/* link new node to old */
		q->key = key;			/* initialize new node */
		q->nodep = nodep;
		q->left = q->right = NULL;
	}
	return;
}

/* delete node with given key */
void *
tdelete(key, rootp)
	const u_long	key;		/* key to be deleted */
	tree		**rootp;	/* address of the root of tree */
{
	tree		*p;
	register tree	*q;
	register tree	*r;

	DBPRT(("tdelete: %lu\n", key))
	if (rootp == (struct tree_t **)0 || (p = *rootp) == (struct tree_t *)0)
		return NULL;
	while (key != (*rootp)->key) {
		p = *rootp;
		rootp = (key < (*rootp)->key) ?
			&(*rootp)->left :		/* left branch */
			&(*rootp)->right;		/* right branch */
		if (*rootp == NULL)
			return ((void *)0);		/* key not found */
	}
	r = (*rootp)->right;				/* D1: */
	if ((q = (*rootp)->left) == NULL)		/* Left */
		q = r;
	else if (r != (struct tree_t *)0) {		/* Right is null? */
		if (r->left == (struct tree_t *)0) {	/* D2: Find successor */
			r->left = q;
			q = r;
		}
		else {		/* D3: Find (struct tree_t *)0 link */
			for (q = r->left; q->left != NULL; q = r->left)
				r = q;
			r->left = q->right;
			q->left = (*rootp)->left;
			q->right = (*rootp)->right;
		}
	}
	free((struct tree_t *) *rootp);     /* D4: Free node */
	*rootp = q;                         /* link parent to new node */
	return(p);
}

void
tfree(rootp)
	tree            **rootp;
{
	if (rootp == NULL || *rootp == NULL)
		return;
	tfree(&(*rootp)->left);
	tfree(&(*rootp)->right);
	free(*rootp);
	*rootp = NULL;
}

/*
**	Start a standard inter-server message.
*/
int
is_compose(stream, command)
     int	stream;
     int	command;
{
	int	ret;

	if (stream < 0)
		return DIS_EOF;
	DIS_rpp_reset();

	ret = diswsi(stream, IS_PROTOCOL);
	if (ret != DIS_SUCCESS)
		goto done;
	ret = diswsi(stream, IS_PROTOCOL_VER);
	if (ret != DIS_SUCCESS)
		goto done;
	ret = diswsi(stream, command);
	if (ret != DIS_SUCCESS)
		goto done;
	return DIS_SUCCESS;

 done:
	DBPRT(("is_compose: send error %s\n", dis_emsg[ret]))
	return ret;
}

/*
**	We got an EOF on a stream.
*/
void
stream_eof(stream, ret)
    int		stream;
    int		ret;
{
	static	char	id[] = "stream_eof";
	struct	pbsnode		*np;
	struct  pbssubn		*sp;
	extern	int		pbs_rm_port;

	rpp_close(stream);

	/* find who the stream belongs to and mark down */
	if ((np = tfind((u_long)stream, &streams)) != NULL) {
		DBPRT(("%s: %s down\n", id, np->nd_name))
		/* Down node and all subnodes */
		np->nd_state |= INUSE_DOWN;
		np->nd_stream = -1;
		tdelete((u_long)stream, &streams);
		for (sp = np->nd_psn; sp; sp = sp->next)
			sp->inuse |= INUSE_DOWN;
	}
	return;
}

/*
**	Send a ping to any node that is in an unknown stat.
**	If wt_parm1 is NULL, set up a worktask to ping again.
*/
void
ping_nodes(ptask)
    struct	work_task	*ptask;
{
	static	char	id[] = "ping_nodes";
	struct	pbsnode		*np;
	struct  pbssubn		*sp;
	struct	sockaddr_in	*addr;
	int			i, ret, com;
	extern	time_t		time_now;
	extern	int		pbs_rm_port;

	DBPRT(("%s: entered\n", id))
	for (i=0; i<svr_totnodes; i++) {
		np = pbsndmast[i];

		if (np->nd_state & (INUSE_DELETED|INUSE_OFFLINE))
		    continue;

		if (np->nd_state & (INUSE_JOB|INUSE_JOBSHARE)) {
		   if (!(np->nd_state & INUSE_NEEDS_HELLO_PING))
			continue;
		}

		if (np->nd_stream < 0) {
			np->nd_stream = rpp_open(np->nd_name, pbs_rm_port);
			np->nd_state |= INUSE_DOWN;
			for (sp = np->nd_psn; sp; sp = sp->next)
				sp->inuse |= INUSE_DOWN;
			if (np->nd_stream == -1) {
				sprintf(log_buffer, "rpp_open to %s",
					np->nd_name);
				log_err(errno, id, log_buffer);
				continue;
			}
			com = IS_HELLO;
			tinsert((u_long)np->nd_stream, np, &streams);
		}
		else
			com = IS_NULL;

		DBPRT(("%s: ping %s\n", id, np->nd_name))
		if (np->nd_state & INUSE_NEEDS_HELLO_PING)
		    com = IS_HELLO;

		ret = is_compose(np->nd_stream, com);
		if (ret == DIS_SUCCESS) {
			if (rpp_flush(np->nd_stream) == 0)
			    continue;
			ret = DIS_NOCOMMIT;
		}

		addr = rpp_getaddr(np->nd_stream);
		sprintf(log_buffer, "%s %d to %s(%s)",
			dis_emsg[ret], errno, np->nd_name,
			(addr == NULL) ? "unknown" : netaddr(addr));
		log_err(-1, id, log_buffer);
		rpp_close(np->nd_stream);
		tdelete((u_long)np->nd_stream, &streams);
		np->nd_stream = -1;
		np->nd_state |= INUSE_DOWN;
		for (sp = np->nd_psn; sp; sp = sp->next)
			sp->inuse |= INUSE_DOWN;
	}

	if (ptask->wt_parm1 == NULL) {
		if (server_init_type == RECOV_HOT)
			i = 15;		/* rapid ping rate while hot restart */
		else
			i = 300;	/* relaxed ping rate for normal run  */
		(void)set_task(WORK_Timed, time_now+i, ping_nodes, NULL);
	}
}

/*
**	Input is coming from another server (MOM) over a DIS rpp stream.
**	Read the stream to get a Inter-Server request.
*/
void
is_request(stream, version)
     int	stream;
     int	version;
{
	static	char		id[] = "is_request";
	int			command = 0;
	int			ret = DIS_SUCCESS;
	int			i, j;
	unsigned long		ipaddr;
	struct	sockaddr_in	*addr;
	struct	pbsnode		*np, *node;
	struct  pbssubn		*sp;
	extern	time_t		time_now;

	DBPRT(("%s: stream %d version %d\n", id, stream, version))
	addr = rpp_getaddr(stream);
	if (version != IS_PROTOCOL_VER) {
		sprintf(log_buffer, "protocol version %d unknown from %s",
			version, netaddr(addr));
		log_err(-1, id, log_buffer);
		rpp_close(stream);
		return;
	}

	/* check that machine is known */
	DBPRT(("%s: connect from %s\n", id, netaddr(addr)))
	if ((node = tfind((u_long)stream, &streams)) != NULL)
		goto found;

	ipaddr = ntohl(addr->sin_addr.s_addr);
	if ((node = tfind(ipaddr, &ipaddrs)) != NULL) {
		if (node->nd_stream >= 0) {
			DBPRT(("%s: stream %d from %s already open on %d\n",
				id, stream, node->nd_name, node->nd_stream))
			rpp_close(stream);
			rpp_close(node->nd_stream);
			tdelete((u_long)node->nd_stream, &streams);
			node->nd_state = INUSE_UNKNOWN;
			node->nd_stream = -1;
			/* do a ping in 5 seconds */
			(void)set_task(WORK_Timed, time_now+5,
				ping_nodes, node);
			return;
		}
		node->nd_stream = stream;
		tinsert((u_long)stream, node, &streams);
		goto found;
	}
	sprintf(log_buffer, "bad attempt to connect from %s", netaddr(addr));
	log_err(-1, id, log_buffer);
	rpp_close(stream);
	return;

 found:
	command = disrsi(stream, &ret);
	if (ret != DIS_SUCCESS)
		goto err;

	switch (command) {

	case IS_NULL:		/* a ping from server */
		DBPRT(("%s: IS_NULL\n", id))
		break;

	case IS_HELLO:
		DBPRT(("%s: IS_HELLO\n", id))
		node->nd_state &=
			~(INUSE_DOWN|INUSE_UNKNOWN|INUSE_NEEDS_HELLO_PING);
		for (sp = node->nd_psn; sp; sp = sp->next)
			sp->inuse &= ~INUSE_DOWN;
		ret = is_compose(stream, IS_CLUSTER_ADDRS);
		if (ret != DIS_SUCCESS)
			goto err;
		for (i=0; i<svr_totnodes; i++) {
			np = pbsndmast[i];

			if (np->nd_state & INUSE_DELETED)
			    continue;

			DBPRT(("%s: %s(%d)\n", id, np->nd_name, i))

			for (j=0; np->nd_addrs[j]; j++) {
				u_long	ipaddr = np->nd_addrs[j];

				DBPRT(("%s: %d\t%ld.%ld.%ld.%ld\n", id, j,
					(ipaddr & 0xff000000) >> 24,
					(ipaddr & 0x00ff0000) >> 16,
					(ipaddr & 0x0000ff00) >> 8,
					(ipaddr & 0x000000ff) ))
				ret = diswul(stream, ipaddr);
				if (ret != DIS_SUCCESS)
					goto err;
			}
		}
		rpp_flush(stream);
		break;
	case IS_UPDATE:
		DBPRT(("%s: IS_UPDATE\n", id))
		i = disrui(stream, &ret);
		if (ret == DIS_SUCCESS) {
			DBPRT(("%s: IS_UPDATE %s 0x%x\n", id, node->nd_name,i))
			i &= (INUSE_DOWN|INUSE_BUSY);
			node->nd_state &= ~(INUSE_DOWN|INUSE_BUSY);
			node->nd_state |= i;
		}
		break;


	default:
		sprintf(log_buffer, "unknown command %d sent from %s",
			command, node->nd_name);
		log_err(-1, id, log_buffer);
		goto err;
	}

	rpp_eom(stream);
	return;

 err:
	/*
	** We come here if we got a DIS write error.
	*/
	sprintf(log_buffer, "%s from %s(%s)", dis_emsg[ret],
		node->nd_name, netaddr(addr));
	log_err(-1, id, log_buffer);
	rpp_close(stream);
	node->nd_state |= INUSE_DOWN;
	for (sp = node->nd_psn; sp; sp = sp->next)
		sp->inuse |= INUSE_DOWN;

	return;
}

void write_node_state()
{
	struct pbsnode *np;
	static char *fmt = "%s %d\n";
	int	i;

	DBPRT(("write_node_state: entered\n"))
	if (nstatef != NULL) {
		(void)fseek(nstatef, 0L, SEEK_SET);	/* rewind and clear */
		(void)ftruncate(fileno(nstatef), (off_t)0);
	} else {
		/* need to open for first time, temporary-move to pbsd_init */
	
		if ((nstatef = fopen(path_nodestate, "w+")) == NULL) {
			log_err(errno, "write_node_state", "could open file");
			return;
		}
	}
	/*
	**	The only state that carries forward is if the
	**	node has been marked offline.
	*/
	for (i=0; i<svr_totnodes; i++) {
		np = pbsndmast[i];

		if (np->nd_state & INUSE_DELETED)
			continue;
		if (np->nd_state & INUSE_OFFLINE)
			fprintf(nstatef, fmt, np->nd_name, np->nd_state);
	}
	(void)fflush(nstatef);
	
}
	
/*
 * free_prop - free list of prop structures created by proplist()
 */

static void free_prop(prop)
	struct prop *prop;
{
	struct prop *pp;

	for (pp = prop; pp; pp = prop) {
		prop = pp->next;
		free(pp->name);
		free(pp);
	}
}
	

/*
 * unreserve - unreserve nodes
 *
 *	If handle is set to a existing resource_t, then release all nodes
 *	associated with that handle, otherwise, (this is dangerous) 
 *	if handle == RESOURCE_T_ALL, release all nodes period.
 */

void
node_unreserve(handle)
	resource_t handle;
{
	struct	pbsnode *np;
	struct  pbssubn *sp;
	int		i;

	/* clear old reserve */
	for (i=0; i<svr_totnodes; i++) {
	    np = pbsndlist[i];
	    for (sp=np->nd_psn; sp; sp=sp->next) {

		if (sp->inuse & INUSE_RESERVE) {
			if ((handle==RESOURCE_T_ALL) || (handle==sp->allocto)) {
				sp->inuse &= ~INUSE_RESERVE;
				np->nd_nsnfree++;
			}
		}
	    }
	}
}

/*
**	Look through the property list and make sure that all
**	those marked are contained in the node.
*/
static
int
hasprop(pnode, props)
    struct	pbsnode	*pnode;
    struct	prop	*props;
{
	struct  prop    *need;

	for (need=props; need; need=need->next) {
		struct	prop	*pp;

		if (need->mark == 0)	/* not marked, skip */
			continue;

		for (pp=pnode->nd_first; pp; pp=pp->next) {
			if (strcmp(pp->name, need->name) == 0)
				break;		/* found it */
		}
		if (pp == NULL)
			return 0;
	}
	return 1;
}


/*
 * see if node has the number of processors required 
 *	if free == SKIP_NONE,  check against total number of processors, else
 *	if free != SKIP_NONE,  check against number free
 *
 *	Return 1 if possible, 0 if not
 */

static int hasppn(pnode, node_req, free)
	struct pbsnode  *pnode;
	int		 node_req;
{
	if ( (free != SKIP_NONE) && (pnode->nd_nsnfree >= node_req) )
		return 1;
	if (!free && (pnode->nd_nsn >= node_req) )
		return 1;
	return 0;
}

/*
**	Mark the properties of a node that match the marked
**	properties given.
*/
static
void
mark(pnode, props)
    struct	pbsnode	*pnode;
    struct	prop	*props;
{
	struct  prop    *set, *pp;

	for (pp=pnode->nd_first; pp; pp=pp->next) {
		pp->mark = 0;
		for (set=props; set; set=set->next) {
			if (set->mark == 0)
				continue;
			if (strcmp(pp->name, set->name) == 0) {
				pp->mark = 1;
				break;
			}
		}
	}
}

#define	RECURSIVE_LIMIT	3
/*
**	Search for a node which contains properties glorf and the requirements.
**	skip indicates which nodes to pass over for this search.
**	Don't do any recursive calls deeper than RECURSIVE_LIMIT.
*/
static
int
search(glorf, vpreq, skip, order, depth)
    struct prop		*glorf;		/* properites */
    int			 vpreq;		/* VPs needed */
    int			 skip;
    int			 order;
    int			 depth;
{
	static	int	pass = INUSE_OFFLINE|INUSE_DOWN|
				INUSE_RESERVE|INUSE_UNKNOWN|INUSE_DELETED;
	struct	pbsnode	*pnode;
	struct	pbssubn	*np;
	int	found;
	int	i;

	if (++depth == RECURSIVE_LIMIT)
		return 0;

	/* look for nodes we haven't picked already */

	for (i=0; i<svr_totnodes; i++) {
	    pnode = pbsndlist[i];

	    if (pnode->nd_ntype  == NTYPE_CLUSTER) {
		if (pnode->nd_flag != okay)
		    continue;
		if (!hasprop(pnode, glorf))
			continue;
		if (skip == SKIP_NONE)  {
		    if (vpreq > pnode->nd_nsn)
			continue;
		} else if ( (skip == SKIP_ANYINUSE) && 
			   ((pnode->nd_state & INUSE_SUBNODE_MASK) ||
			    (vpreq > pnode->nd_nsnfree)) ) {
			continue;
		} else if ( (skip == SKIP_EXCLUSIVE) && 
			   ((pnode->nd_state & INUSE_SUBNODE_MASK) ||
			    (vpreq > (pnode->nd_nsnfree + pnode->nd_nsnshared))) ) {
			continue;
		}

		pnode->nd_flag = thinking;
		mark(pnode, glorf);
		pnode->nd_needed = vpreq;
		pnode->nd_order  = order;
		return 1;
	    }
	}

	if (glorf == NULL)			/* no property */
		return 0;			/* can't retry */

	/* try re-shuffling the nodes to get what we want */

	for (i=0; i<svr_totnodes; i++) {
	    pnode = pbsndlist[i];
	    if (pnode->nd_ntype == NTYPE_CLUSTER) {

		    if (pnode->nd_flag != thinking)
			continue;
		    if (pnode->nd_state & pass)
			continue;
		    if ( (skip == SKIP_EXCLUSIVE) &&
			 (vpreq < pnode->nd_nsnfree) )
			continue;
		    if ( (skip == SKIP_ANYINUSE) &&
				(vpreq < (pnode->nd_nsnfree + pnode->nd_nsnshared)) )
			continue;
		    if (!hasprop(pnode, glorf))
			continue;

		    pnode->nd_flag = conflict;
		    found = search(pnode->nd_first, vpreq, skip, pnode->nd_order, depth);
		    pnode->nd_flag = thinking;
		    if (found) {
			mark(pnode, glorf);
			pnode->nd_needed = vpreq;
			pnode->nd_order  = order;
			
			return 1;
		    }
	    }
	}
	return 0;	/* not found */
}

/*
**	Parse a number in a spec.
**	Return 0 if okay, 1 if no number exists, -1 on error
*/
static
int
number(ptr, num)
    char	**ptr;
    int		*num;
{
	char	holder[80];
	int	i = 0;
	char	*str = *ptr;

	while (isdigit(*str))
		holder[i++] = *str++;

	if (i == 0)
		return 1;

	holder[i] = '\0';
	if ((i = atoi(holder)) == 0) {
		sprintf(log_buffer, "zero illegal");
		return -1;
	}

	*ptr = str;
	*num = i;
	return 0;
}

/*
**	Check string to see if it is a legal property name.
**	If not, return 1.
**	*prop set to static char array containing the properity,
**	must be copied.
*/
static
int
property(ptr, prop)
    char	**ptr;
    char	**prop;
{
	static	char	name[80];
	char*	str = *ptr;
	int	i = 0;

	if (!isalpha(*str)) {
		sprintf(log_buffer,
			"first character of property (%s) not a letter", str);
		return 1;
	}
	while (isalnum(*str) || *str == '-' || *str == '.' || *str == '=')
		name[i++] = *str++;

	name[i] = '\0';
	*prop = (i == 0) ? NULL : name;

	/* skip over "/vp_number" */
	if (*str == '/') {
		do {
			str++;
		} while (isdigit(*str));
	}
	*ptr = str;
	return 0;
}

/*
**	Create a property list from a string.
**	Return 0 if all is well, 1 otherwise.
*/
static
int
proplist(str, plist, node_req)
    char		**str;
    struct	prop	**plist;
    int			 *node_req;
{
	struct	prop	*pp;
	char		*pname;
	char		*pequal;

	*node_req = 1;	/* default to 1 processor per node */

	for (;;) {
		if (property(str, &pname))
			return 1;
		if (pname == NULL)
			break;
		if (pequal = strchr(pname, (int)'=')) {  /* special properity */
			
		    /* identify the special property and place its value */
		    /* into node_req 					 */

		    *pequal = '\0';

		    if (strcmp(pname, "ppn") == 0) {
			pequal++;
			if ((number(&pequal, node_req) != 0) ||
			    (*pequal != '\0'))
				return 1;
		    } else 
			return 1;	/* not recognized - error */
		} else {
		    pp = (struct prop *)malloc(sizeof(struct prop));
		    pp->mark = 1;
		    pp->name = strdup(pname);
		    pp->next = *plist;
		    *plist = pp;
		}
		if  (**str != ':')
			break;
		(*str)++;
	}
	return 0;
}


/*
 *	Evaluate one element in a node spec.
 *
 *	Return 1 if it can be satisfied
 *	       0 if it cannot be completly satisfied. (not used now)
 *	      -1 if error - can never be satisfied.
 */
static
int
listelem(str, order)
    char	**str;
    int		order;
{
	int	num = 1;
	int	i, hit, j;
	int	ret = -1;
	struct	prop	*pp, *prop = NULL;
	struct  pbsnode *pnode;
	struct	pbssubn	*np;
	int	node_req = 1;

	if ((i = number(str, &num)) == -1)	/* get number */
		return ret;

	if (i == 0) {				/* number exists */
		if (**str == ':') {		/* there are properties */
			(*str)++;
			if (proplist(str, &prop, &node_req))
				return ret;
		}
	}
	else {					/* no number */
		if (proplist(str, &prop, &node_req))  /* must be a prop list */
			return ret;		/* with no number in front */
	}

	/* count number of nodes with the requested property */
	hit = 0;
	for (i=0; i<svr_totnodes; i++) {
	    pnode = pbsndlist[i];
	    if (pnode->nd_ntype == NTYPE_CLUSTER) {
		if ( hasprop(pnode, prop) && 
		     hasppn(pnode, node_req, SKIP_NONE) )
			hit++;
		if (hit == num) {
			break;		/* found enough  */
		}
	    }
	}

	if (hit < num)			/* can never be satisfied */
		goto done;

	/*
	** Find an initial set of nodes to satisfy the request.
	** Go ahead and use any nodes no mater what state they are in.
	*/
	for (i=0; i<num; i++) {
		if (search(prop, node_req, SKIP_NONE, order, 0))
			continue;

		goto done;		/* can never be satisfied */
	}
	ret = 1;

done:
	free_prop(prop);
	return ret;
}

/*
**	Add the "global" spec to every sub-spec in "spec".
*/
static
char    *
mod_spec(spec, global)
    char	*spec;
    char	*global;
{
	static	char	line[512];
	char	*cp;
	int	len;

	len = strlen(global);
	cp = line;
	while (*spec) {
		if (*spec == '+') {
			*cp++ = ':';
			strcpy(cp, global);
			cp += len;
		}
		*cp++ = *spec++;
	}
	*cp++ = ':';
	strcpy(cp, global);

	return line;
}

/* cntjons - count jobs on (shared) nodes */

static int
cntjons(pn)
    struct pbsnode *pn;
{
    	struct pbssubn *psn;
	int ct = 0;
	int n;
	struct jobinfo *pj;

	psn = pn->nd_psn;
	for (n=0; n<pn->nd_nsn; ++n) {

	    pj = psn->jobs;
	    while (pj) {
		++ct;
		pj = pj->next;
	    }
	    psn = psn->next;
	}
	return (ct);
}

/*
 * nodecmp - compare two nodes for sorting 
 *	For "exclusive", depending on setting of node_order attribute:
 *	    pack:    put free node with fewest non-zero free VPs in node first
 *	    scatter: put free node with most fre VPs first
 *	For "shared", put current shared with fewest jobs first,
 *		then free nodes, and others last
 */

#define BIG_NUM 4096;

static int
nodecmp(aa, bb)
    const	void	*aa;
    const	void	*bb;
{
	struct	pbsnode	*a = *(struct pbsnode **)aa;
	struct	pbsnode	*b = *(struct pbsnode **)bb;
	int	aprim, bprim;
	int	i;

	if (exclusive) {	/* best is free */
	    if (server.sv_attr[(int)SVR_ATR_NodePack].at_val.at_long) {
		/* pack - fill up nodes first */
		aprim = (a->nd_nsnfree > 0) ? a->nd_nsnfree : BIG_NUM;
		bprim = (b->nd_nsnfree > 0) ? b->nd_nsnfree : BIG_NUM;
	    } else {
		/* scatter - spread amoung nodes first */
		aprim = a->nd_nsn - a->nd_nsnfree;
		bprim = b->nd_nsn - b->nd_nsnfree;
	    }

	} else {		/* best is shared with fewest jobs */
		aprim = (a->nd_state == INUSE_JOBSHARE) ? cntjons(a) :
				((a->nd_state == INUSE_FREE) ? 5 : 1000);
		bprim = (b->nd_state == INUSE_JOBSHARE) ? cntjons(b) :
				((b->nd_state == INUSE_FREE) ? 5 : 1000);
	}
	if (aprim == bprim) {
		return (a->nd_nprops - b->nd_nprops);
	} else {
		return (aprim - bprim);
	}
}

/*
 *	Test a node specification.  
 *
 *	Return >0 - number of nodes counted in the spec if it works,
 *	        0 - if it cannot be satisfied,
 *	       -1 - if it can never be satisfied.
 *	Okay to bail early if "early" is true.
 *	VPs selected are marked "thinking"
 */
static int
node_spec(spec, early)
    char	*spec;
    int		 early;
{
	static	char	id[] = "node_spec";
	struct	pbsnode	*pnode;
	struct	pbssubn	*snp;
	char	*str, *globs, *cp, *hold;
	int	i, num;
	int	rv;
	static char shared[] = "shared";

	DBPRT(("%s: entered spec=%s\n", id, spec))
	exclusive = 1;	/* by default, nodes (VPs) are requested exclusively */
	spec = strdup(spec);
	if ((globs = strchr(spec, '#')) != NULL) {
		*globs++ = '\0';
		globs = strdup(globs);
		while ((cp = strrchr(globs, '#')) != NULL) {
			*cp++ = '\0';
			if (strcmp(cp, shared) != 0) {
				hold = strdup(mod_spec(spec, cp));
				free(spec);
				spec = hold;
			} else {
				exclusive = 0;
			}
		}
		if (strcmp(globs, shared) != 0) {
			hold = strdup(mod_spec(spec, globs));
			free(spec);
			spec = hold;
		} else {
			exclusive = 0;
		}
		free(globs);
	}
	str = spec;
		
	num = ctnodes(str);
	if (num > svr_clnodes) {
		free(spec);
		return -1;
	}

	/*
	 * if SVR_ATR_NodePack set (true or false), then
	 * sort nodes by state, number of VPs and number of attributes;
	 * otherwise, leave unsorted
	 */
	if (server.sv_attr[(int)SVR_ATR_NodePack].at_flags & ATR_VFLAG_SET) {
	    qsort(pbsndlist, svr_totnodes, sizeof(struct pbsnode *), nodecmp);
	}

	/* reset subnodes (VPs) to okay */

	svr_numnodes = 0;
	for (i=0; i<svr_totnodes; i++) {
	    pnode = pbsndlist[i];
	    DBPRT(("%s: %s nsn %d, nsnfree %d\n", id, pnode->nd_name,
		   pnode->nd_nsn, pnode->nd_nsnfree))
	    pnode->nd_flag   = okay;
	    pnode->nd_needed = 0;
	    for (snp = pnode->nd_psn; snp; snp = snp->next) {
		snp->flag = okay;
		DBPRT(("%s: %s/%d inuse 0x%x nprops %d\n", id,
		pnode->nd_name,snp->index,snp->inuse,pnode->nd_nprops))
	    }

	    if ( (pnode->nd_ntype == NTYPE_CLUSTER) &&
	         ((pnode->nd_state & (INUSE_OFFLINE|
				     INUSE_DOWN|
				     INUSE_RESERVE|
				     INUSE_JOB)) == 0) )
			svr_numnodes++;
	}

	/*
	 * Make first pass at finding nodes to allocate.
	 * process each subspec (piece between '+'s)
	 */

	for (i=1;; i++) {
		if ((rv = listelem(&str, i)) <= 0) {
			free(spec);
			return rv;
		}
		if (*str != '+')
			break;
		str++;
	}
	i = (int)*str;
	free(spec);

	if (i)					/* garbled list */
		return -1;
	if (num > svr_numnodes && early)	/* temp fail, not available */
		return 0;
/*
 * 	At this point we know the spec is legal.
 *	Here we find a replacement for any nodes chosen above
 *	that are already inuse.
 */
	for (i=0; i<svr_totnodes; i++) {
	    pnode = pbsndlist[i];
	    if (pnode->nd_ntype != NTYPE_CLUSTER)
			continue;	
	    if (pnode->nd_flag != thinking)
		continue;
	    if (pnode->nd_state == INUSE_FREE)  {
		if (pnode->nd_needed <= pnode->nd_nsnfree) 
		    continue;
		if ( !exclusive && 
	            (pnode->nd_needed < pnode->nd_nsnfree+pnode->nd_nsnshared) )
		    continue;		/* shared node */
	    } else {
		if ( !exclusive && 
	            (pnode->nd_needed <= pnode->nd_nsnfree+pnode->nd_nsnshared) )
		    continue;		/* shared node */
	    }

	    /* otherwise find replacement node */
	    if ( search(pnode->nd_first, pnode->nd_needed, 
				exclusive ? SKIP_ANYINUSE : SKIP_EXCLUSIVE,
				pnode->nd_order, 0)) {
			pnode->nd_flag = okay;
			continue;
	    }
	    if (early)
		return 0;
	    num = 0;
	}

	return num;	/* spec ok */
}

/*
**	set_nodes() - Call node_spec() to allocate nodes then set them inuse.
**	Build list of allocated nodes to pass back in rtnlist.
*/
int set_nodes(pjob, spec, rtnlist)
    job		*pjob;
    char	*spec;
    char       **rtnlist;
{
	struct	howl {
		char	*name;
		int	order;
		int	index;
		struct	howl *	next;
	} *hp, *hlist, *curr, *prev, *nxt;
	int     i;
	short   newstate;

	static	char	*id = "set_nodes";
	struct  pbsnode *pnode;
	struct	pbssubn	*snp;
	struct	prop	*pp;
	char		*nodelist;
	resource	*pres;

	DBPRT(("%s: entered jid %s spec %s\n",

			id, pjob->ji_qs.ji_jobid, spec))
	/*
	 * allocate nodes 
	 */
	if ((i = node_spec(spec, 1)) == 0)	/* check spec */
		return PBSE_RESCUNAV;
	else if (i == -1)
		return PBSE_UNKNODE;

	if (exclusive)
		svr_numnodes -= i;
	hlist = NULL;

	newstate = exclusive ? INUSE_JOB : INUSE_JOBSHARE;
	for (i=0; i<svr_totnodes; i++) {
	    struct	jobinfo	*jp;

	    pnode = pbsndlist[i];
	    if (pnode->nd_flag != thinking)
		continue;			/* skip this one */

	    /* within the node, check each subnode */

	    for (snp=pnode->nd_psn; snp && pnode->nd_needed; snp = snp->next) {

		if (exclusive) {
			if (snp->inuse != INUSE_FREE)
				continue;
		} else {
			if ((snp->inuse != INUSE_FREE) && (snp->inuse != INUSE_JOBSHARE))
				continue;
		}

		/* Mark subnode as being IN USE */

		DBPRT(("%s: allocated node %s/%d\n", id, pnode->nd_name, snp->index))
		if (snp->inuse == INUSE_FREE) {
			snp->inuse = newstate;
			pnode->nd_nsnfree--;		/* reduce free count */
			if (!exclusive)
				pnode->nd_nsnshared++;
		}
		jp  = (struct jobinfo *)malloc(sizeof(struct jobinfo));
		jp->next = snp->jobs;
		snp->jobs = jp;
		jp->job = pjob;

		/* build list of nodes ordered to match request */

		curr = (struct howl *)malloc(sizeof(struct howl));
		curr->order = pnode->nd_order;
		curr->name  = pnode->nd_name;
		curr->index = snp->index;

		for (prev=NULL, hp=hlist; hp; prev=hp, hp=hp->next) {
			if (curr->order <= hp->order)
				break;
		}
		curr->next = hp;
		if (prev == NULL)
			hlist = curr;
		else
			prev->next = curr;

		--pnode->nd_needed;
	    }
	    if (pnode->nd_nsnfree == 0)	    /* if no free VPs, set node state */
		pnode->nd_state = newstate;
	}

	pjob->ji_qs.ji_svrflags |= JOB_SVFLG_HasNodes;	/* indicate has nodes */

	/* build list of allocated nodes */

	i = 1;			/* first, size list */
	for (hp=hlist; hp; hp=hp->next) {
		i += (strlen(hp->name) + 4);
	}
	nodelist = malloc(++i);
	*nodelist = '\0';
	/* now copy in name+name+... */
	for (hp=hlist; hp; hp=nxt) {
		(void)sprintf(nodelist+strlen(nodelist), "%s/%d+",
				hp->name, hp->index);
		nxt = hp->next;
		(void)free(hp);
	}
	*(nodelist + strlen(nodelist) - 1) = '\0';	/* strip trailing + */

	*rtnlist = nodelist;
	DBPRT(("Allocated to %s nodes: %s\n",
		pjob->ji_qs.ji_jobid, nodelist))
	return (PBSE_NONE);
}


/*
 * node_avail_complex - 
 *		*navail is set to number available
 *		*nalloc is set to number allocated
 *		*nresvd is set to number reserved 
 *		*ndown  is set to number down/offline
 */

int node_avail_complex(spec, navail, nalloc, nresvd, ndown)
	char	*spec;		/* In  - node spec */
	int 	*navail;	/* Out - number available */
	int	*nalloc;	/* Out - number allocated */
	int	*nresvd;	/* Out - number reserved  */
	int	*ndown;		/* Out - number down	  */
{
	int	holdnum;
	int	ret;

	holdnum = svr_numnodes;
	ret = node_spec(spec, 1);
	svr_numnodes = holdnum;

	*navail = ret;
	*nalloc = 0;
	*nresvd = 0;
	*ndown  = 0;

	return (ret);
}

/*
 * node_avail - report if nodes requested is available
 *	Does NOT even consider Time Shared Nodes 
 *
 *	Return 0 when no error in request and
 *		*navail is set to number available
 *		*nalloc is set to number allocated
 *		*nresvd is set to number reserved 
 *		*ndown  is set to number down/offline
 *	     !=0 error number when error in request
 */

int node_avail(spec, navail, nalloc, nresvd, ndown)
	char	*spec;		/* In  - node spec */
	int 	*navail;	/* Out - number available */
	int	*nalloc;	/* Out - number allocated */
	int	*nresvd;	/* Out - number reserved  */
	int	*ndown;		/* Out - number down	  */
{
	char	*id = "node_avail";
	int	i;
	int	j;
	int	holdnum;
	struct	pbsnode	*pn;
	struct  pbssubn *psn;
	char    *pc;
	struct prop *prop = NULL;
	register int xavail;
	register int xalloc;
	register int xresvd;
	register int xdown;
	int          node_req = 1;


	if (spec == NULL) {
		log_event(PBSEVENT_ADMIN, PBS_EVENTCLASS_SERVER, id, "no spec");
		return (RM_ERR_NOPARAM);
	}

	pc = spec;
	if ( (strchr(spec, (int)'+') == NULL) &&
	     (number(&pc, &holdnum) == 1) ) {

		/* A simple node spec - reply with numbers of avaiable,	*/
		/* allocated, reserved, and down nodes that match the	*/
		/* the spec, null or simple number means all		*/


		xavail = 0;
		xalloc = 0;
		xresvd = 0;
		xdown  = 0;

		/* find number of a specific type of node */

		if (*pc)
			if (proplist(&pc, &prop, &node_req))
				return (RM_ERR_BADPARAM);

		for (i=0; i<svr_totnodes; i++) {
			pn = pbsndlist[i];

		     if ((pn->nd_ntype == NTYPE_CLUSTER) && hasprop(pn, prop)) {
			if (pn->nd_state &
				     (INUSE_OFFLINE|INUSE_DOWN))
				++xdown;
			else if (hasppn(pn, node_req, SKIP_ANYINUSE))
				++xavail;
			else if (hasppn(pn, node_req, SKIP_NONE)) {
			    /*
			     * has enought processor, are they busy or reserved
			     */
			    j = 0;
			    for (psn=pn->nd_psn; psn; psn=psn->next) {
				if (psn->inuse  & INUSE_RESERVE)
					j++;
			    }
			    if (j >= node_req)
				++xresvd;
			    else
				++xalloc;
			}
		    }
		}
		free_prop(prop);

		*navail = xavail;
		*nalloc = xalloc;
		*nresvd = xresvd;
		*ndown  = xdown;
		return 0;

	} else if (number(&pc, &holdnum) == -1) {
		    /* invalid spec */
		    return (RM_ERR_BADPARAM);
	}


	/* not a simple spec - determine if supplied complex	*/
	/* node spec can be satisified from avail nodes		*/
	/* navail set to >0 if can be satified now		*/
	/*		  0 if not now but possible		*/
	/*		 -l if never possible			*/

	(void)node_avail_complex(spec, navail, nalloc, nresvd, ndown);
	return (0);
}


/*
 * node_reserve - Reserve nodes 
 *	Cannot reserve Time Shared Nodes
 *
 *	Returns: >0 - reservation suceeded, number of nodes reserved
 *		  0 - None or partial reservation
 *		 -1 - requested reservation impossible
 */

int node_reserve(nspec, tag)
	char		*nspec; /* In     - a node specification */
	resource_t	 tag;	   /* In/Out - tag for resource if reserved */
{
	static	char	id[] = "node_reserve";
	struct  pbsnode *pnode;
	struct	pbssubn	*snp;
	char		*spec;
	int		 ret_val;
	int		 i;

	DBPRT(("%s: entered\n", id))
	if ((nspec == NULL) || (*nspec == '\0')) {
		log_event(PBSEVENT_ADMIN, PBS_EVENTCLASS_SERVER, id, "no spec");
		return -1;
	}

	if (tag)
		node_unreserve(tag);    /* release so can rereserve */

	if ((ret_val = node_spec(nspec, 0)) >= 0) {
		/*
		** Zero or more of the needed Nodes are available to be 
		** reserved.
		*/
		for (i=0; i<svr_totnodes; i++) {
		    pnode = pbsndlist[i];
		    for (snp=pnode->nd_psn; snp; snp=snp->next) {

			if ( (snp->flag == thinking) &&
			     (snp->inuse == INUSE_FREE) ) {
				DBPRT(("hold %s/%d\n", pnode->nd_name, snp->index))
				snp->inuse |= INUSE_RESERVE;
				snp->allocto = tag;
				pnode->nd_nsnfree--;
				if (--pnode->nd_needed == 0)
					break;	     /* marked all VP needed */
			}
		    }
		}

	} else {

	    /* could never satisfy the reservation */

	    (void)sprintf(log_buffer, "can never reserve %s",nspec);
	    log_record(PBSEVENT_SCHED, PBS_EVENTCLASS_REQUEST, id,
			log_buffer);

	}

	return ret_val;
}

/*
 * is_ts_node - does the nodestr specify a single time share node?
 *	0 - yes
 *	1 - no, not a ts node or more than one node (name will not match)
 */

int is_ts_node(nodestr)
	char *nodestr;
{
	struct pbsnode *np;
	int		i;

	for (i=0; i<svr_totnodes; i++) {
		np = pbsndmast[i];

		if ( ((np->nd_state & INUSE_DELETED) == 0) &&
		     (np->nd_ntype == NTYPE_TIMESHARED) ) {
			if (strcmp(nodestr, np->nd_name) == 0)
				return 0;
		}
	}
	return 1;
}

/*
 * find_ts_node - find first up time-shared node
 *
 *	returns name of node or null
 */
char *find_ts_node()
{
	struct pbsnode *np;
	int		i;

	for (i=0; i<svr_totnodes; i++) {
	    np = pbsndmast[i];
	    if ( (np->nd_ntype == NTYPE_TIMESHARED) &&
	         ((np->nd_state & (INUSE_DOWN|INUSE_DELETED|INUSE_OFFLINE)) ==0)) {
		return (np->nd_name);
	    }
	}
	return ((char *)0);
}

/*
 * free_nodes - free nodes allocated to a job
 */

void free_nodes(pjob)
	job *pjob;
{
	static	char	id[] = "free_nodes";
	struct	pbssubn	*np;
	struct  pbssubn *xnp;
	struct  pbsnode *pnode;
	struct	jobinfo	*jp, *prev;
	int		i;

	DBPRT(("%s: entered\n", id))
	for (i=0; i<svr_totnodes; i++) {
	    pnode = pbsndlist[i];
	    for (np = pnode->nd_psn; np; np = np->next) {

		for (prev=NULL, jp=np->jobs; jp; prev=jp, jp=jp->next) {
			if (jp->job != pjob)
				continue;

			DBPRT(("Freeing node %s/%d from job %s\n", pnode->nd_name, np->index,pjob->ji_qs.ji_jobid))
			if (prev == NULL)
				np->jobs = jp->next;
			else
				prev->next = jp->next;
			free(jp);
			if (np->jobs == NULL) {
			    pnode->nd_nsnfree++;	/* up count of free */
			    DBPRT(("%s: upping free count to %d\n", id, pnode->nd_nsnfree))
			    if (np->inuse & INUSE_JOBSHARE)
				pnode->nd_nsnshared--;
			    np->inuse &= ~(INUSE_JOB|INUSE_JOBSHARE);
			    for (xnp=pnode->nd_psn; xnp; xnp = xnp->next) {
				if (xnp->inuse & (INUSE_JOB|INUSE_JOBSHARE))
				    break;
			    }
			    pnode->nd_state &= ~(INUSE_JOB|INUSE_JOBSHARE);
			}
			break;
		}
	    }
	}
	pjob->ji_qs.ji_svrflags &= ~JOB_SVFLG_HasNodes;
}

/*
 * set_one_old - set a named node as allocated to a a job
 */

static void set_one_old(name, pjob, shared)
	char *name;
	job  *pjob;
	int   shared;	/* how used flag, either INUSE_JOB or INUSE_JOBSHARE */
{
	int		i;
	int		index;
	struct pbsnode *pnode;
	struct pbssubn *snp;
	struct jobinfo *jp;
	char	       *pc;

	
	if (pc = strchr(name, (int)'/'))  {
		index = atoi(pc+1);
		*pc = '\0';
	} else
		index = 0;

	for (i=0; i<svr_totnodes; i++) {
	    pnode = pbsndmast[i];

	    if (strcmp(name, pnode->nd_name) == 0) {

		/* Mark node as being IN USE ...  */

		 if (pnode->nd_ntype == NTYPE_CLUSTER) {
		    for (snp = pnode->nd_psn; snp; snp = snp->next) {
			if (snp->index == index) {

				snp->inuse = shared;
				jp  = (struct jobinfo *)malloc(sizeof(struct jobinfo));
				jp->next = snp->jobs;
				snp->jobs = jp;
				jp->job = pjob;

				if (--pnode->nd_nsnfree == 0)
				    pnode->nd_state = shared;
				return;
			}

		    }
		}
	    }
	}
}
	
/*
 * set_old_nodes - set "old" nodes as in use - called from pbsd_init
 *	when recovering a job in the running state.
 */
void set_old_nodes(pjob)
	job *pjob;
{
	char *old;
	char *po;
	struct pbssubn *np;
	resource *presc;
	int   shared = INUSE_JOB;

	if ((pbsndmast != NULL) && 
	    (pjob->ji_wattr[(int)JOB_ATR_exec_host].at_flags & ATR_VFLAG_SET)) {

		/* are the nodes being used shared? Look in "neednodes" */

		presc = find_resc_entry(&pjob->ji_wattr[(int)JOB_ATR_resource],
					find_resc_def(svr_resc_def, "neednodes",
					svr_resc_size) );
		if ( (presc != NULL) && 
		     (presc->rs_value.at_flags & ATR_VFLAG_SET) ) {
			if (po = strchr(presc->rs_value.at_val.at_str, '#')) {
				if (strstr(++po, "shared") != NULL) 
					shared = INUSE_JOBSHARE;
			}
		}

		
		old = strdup(pjob->ji_wattr[(int)JOB_ATR_exec_host].at_val.at_str);
		while ((po = strrchr(old, (int)'+')) != NULL) {

			*po++ = '\0';
			set_one_old(po, pjob, shared);
		}
		set_one_old(old, pjob, shared);

		free(old);
	}
}
