/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
/*
 * The entry point function for pbs_daemon.
 *
 * Included public functions re:
 *
 *	main	initialization and main loop of pbs_daemon
 */

#include <pbs_config.h>   /* the master config generated by configure */

#include <sys/types.h>
#if (PLOCK_DAEMONS & 1)
#include <sys/lock.h>
#endif	/* PLOCK_DAEMONS */
#include <netinet/in.h>
#include "pbs_ifl.h"
#include <assert.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <netdb.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include "list_link.h"
#include "work_task.h"
#include "log.h"
#include "server_limits.h"
#include "attribute.h"
#include "job.h"
#include "queue.h"
#include "server.h"
#include "net_connect.h"
#include "credential.h"
#include "svrfunc.h"
#include "tracking.h"
#include "acct.h"
#include "sched_cmds.h"
#include "rpp.h"
#include "dis.h"
#include "dis_init.h"

static char ident[] = "@(#) $RCSfile: pbsd_main.c,v $ $Revision: 2.4.8.1 $";

/* External functions called */

extern int  pbsd_init A_((int));
extern void shutdown_ack ();
extern void update_nodes_file A_((void));

/* External data items */
extern	int	svr_chngNodesfile;


/* Local Private Functions */

static int    get_port A_((char *, unsigned int *, pbs_net_t *));
static time_t next_task A_(());
static int    start_hot_jobs();
static void   lock_out A_((int, int));

/* Global Data Items */

char	       *acct_file = (char *)0;
char	       *log_file  = (char *)0;
char	       *path_home = PBS_SERVER_HOME;
char	       *path_acct;
char	        path_log[MAXPATHLEN+1];
char	       *path_priv;
char	       *path_jobs;
char	       *path_queues;
char	       *path_spool;
char	       *path_svrdb;
char	       *path_svrdb_new;
char 	       *path_track;
char	       *path_nodes;
char	       *path_nodes_new;
char	       *path_nodestate;
char		pbs_current_user[PBS_MAXUSER]="PBS_Server"; /* for libpbs.a */
char	       *msg_daemonname    = pbs_current_user;
int		pbs_errno;
char	       *pbs_o_host = "PBS_O_HOST";
pbs_net_t	pbs_mom_addr;
unsigned int	pbs_mom_port;
unsigned int	pbs_rm_port;
pbs_net_t	pbs_scheduler_addr;
unsigned int	pbs_scheduler_port;
pbs_net_t	pbs_server_addr;
unsigned int	pbs_server_port_dis;
int		queue_rank = 0;
struct server	server;		/* the server structure */
char	        server_host[PBS_MAXHOSTNAME+1];	/* host_name */
char	       *mom_host = server_host;
int	 	server_init_type = RECOV_WARM;
char	        server_name[PBS_MAXSERVERNAME+1]; /* host_name[:service|port] */
int		svr_delay_entry = 0;
int		svr_do_schedule = SCH_SCHEDULE_NULL;
list_head	svr_queues;            /* list of queues                   */
list_head	svr_alljobs;           /* list of all jobs in server       */
list_head	svr_newjobs;           /* list of incomming new jobs       */
list_head	task_list_immed;
list_head	task_list_timed;
list_head	task_list_event;
time_t		time_now;

void
DIS_rpp_reset()
{
	if (dis_getc != rpp_getc) {
		dis_getc = rpp_getc;
		dis_puts = (int (*) A_((int, const char *, size_t)) )rpp_write;
		dis_gets = (int (*) A_((int, char *, size_t)) )rpp_read;
		disr_skip   = (int (*) A_((int, size_t)) )rpp_skip;
		disr_commit = rpp_rcommit;
		disw_commit = rpp_wcommit;
	}
}

/*
** Read a RPP message from a stream.  Only one kind of message
** is expected -- Inter Server requests from MOM's.
*/
void
do_rpp(stream)
    int		stream;
{
	static	char		id[] = "do_rpp";
	int			ret, proto, version;
	void			is_request A_((int, int));
	void			stream_eof A_((int, int));

	DIS_rpp_reset();
	proto = disrsi(stream, &ret);
	if (ret != DIS_SUCCESS) {
		stream_eof(stream, ret);
		return;
	}
	version = disrsi(stream, &ret);
	if (ret != DIS_SUCCESS) {
		DBPRT(("%s: no protocol version number %s\n",
				id, dis_emsg[ret]))
		stream_eof(stream, ret);
		return;
	}

	switch (proto) {
	case	IS_PROTOCOL:
		DBPRT(("%s: got an inter-server request\n", id))
		is_request(stream, version);
		break;

	default:
		DBPRT(("%s: unknown request %d\n", id, proto))
		rpp_close(stream);
		break;
	}
	return;
}

void
rpp_request(fd)
    int		fd;	/* not used */
{
	static	char	id[] = "rpp_request";
	int	stream;

	for (;;) {
		if ((stream = rpp_poll()) == -1) {
			log_err(errno, id, "rpp_poll");
			break;
		}
		if (stream == -2)
			break;
		do_rpp(stream);
	}
	return;
}

/*
 * main - the initialization and main loop of pbs_daemon
 */

main(argc, argv)
	int	 argc;
	char	*argv[];
{
	int	 c;
	FILE    *dummyfile;
	int	 i;
	int	 lockfds;
	int	 rppfd;			/* fd to receive is HELLO's */
	int	 privfd;		/* fd to send is messages */
	uint	 tryport;
	char	 lockfile[MAXPATHLEN+1];
	char	*pc;
	job	*pjob;
	pbs_queue *pque;
	char	*servicename;
	pid_t	 sid;
	long    *state;
	time_t	 waittime;
	void	 ping_nodes A_((struct work_task *ptask));

	static struct {
		char *it_name;
		int   it_type;
	} init_name_type[] = {
		{ "hot",	RECOV_HOT },
		{ "warm",	RECOV_WARM },
		{ "cold",	RECOV_COLD },
		{ "create",	RECOV_CREATE },
		{ "",		RECOV_Invalid }
	};

	extern  int   optind;
	extern	char *optarg;
	extern  char *msg_daemonname;
	extern  char *msg_svrdown;	/* log message   */
	extern  char *msg_startup1;	/* log message   */
	extern  char *msg_startup2;	/* log message   */
	extern  char *msg_shutdnsig;	/* log message   */
	extern  char *msg_init_chdir;	/* error message */


	/* If we are not run with real and effective uid of 0, forget it */

	if ((getuid() != 0) || (geteuid() != 0)) {
		fprintf(stderr, "%s: Must be run by root\n", argv[0]);
		return (1);
	}

	/* set standard umask */

	umask(022);

	/* find out who we are (hostname) */

	if ((gethostname(server_host, PBS_MAXHOSTNAME) == -1) ||
	    (get_fullhostname(server_host,server_host,PBS_MAXHOSTNAME) == -1)) {
		log_err(-1, "pbsd_main", "Unable to get my host name");
		return (-1);
	}

	/* initialize service port numbers for self, Scheduler, and MOM */

	pbs_server_port_dis = get_svrport(PBS_BATCH_SERVICE_NAME, "tcp", 
				PBS_BATCH_SERVICE_PORT_DIS);
	pbs_scheduler_port = get_svrport(PBS_SCHEDULER_SERVICE_NAME, "tcp",
				PBS_SCHEDULER_SERVICE_PORT);
	pbs_mom_port = get_svrport(PBS_MOM_SERVICE_NAME, "tcp",
				PBS_MOM_SERVICE_PORT);
	pbs_rm_port = get_svrport(PBS_MANAGER_SERVICE_NAME, "tcp",
				PBS_MANAGER_SERVICE_PORT);

	(void)strcpy(server_name, server_host);	/* by default server = host */
	pbs_server_addr    = get_hostaddr(server_host);
	pbs_mom_addr 	   = pbs_server_addr;   /* assume on same host */
	pbs_scheduler_addr = pbs_server_addr;   /* assume on same host */

	/* parse the parameters from the command line */

	while ((c = getopt(argc, argv, "A:a:d:p:t:L:M:R:S:")) != -1) {
	    switch (c) {
		case 'a':
		    if (decode_b(&server.sv_attr[(int)SRV_ATR_scheduling], NULL,
			NULL, optarg) != 0) { 
			(void)fprintf(stderr, "%s: bad -a option\n", argv[0]);
			return (1);
		    }
		    break;
		case 'd':
		    path_home = optarg;
		    break;
		case 'p':
		    servicename = optarg;
		    if (strlen(server_name) + strlen(servicename) + 1 >
			(size_t)PBS_MAXSERVERNAME) {
			    (void)fprintf(stderr,
				   "%s: -p host:port too long\n", argv[0]);
			    return (1);
			}
		    (void)strcat(server_name, ":");
		    (void)strcat(server_name, servicename);
		    if ((pbs_server_port_dis = atoi(servicename)) == 0) {
			(void)fprintf(stderr,
				   "%s: -p host:port invalid\n", argv[0]);
			return (1);
		    }
		    break;
		case 't':
		    for (i = RECOV_HOT; i < RECOV_Invalid; i++) {
			if (strcmp(optarg, init_name_type[i].it_name) == 0) {
			    server_init_type = init_name_type[i].it_type;
			    break;
			}
		    }
		    if (i == RECOV_Invalid) {
			(void)fprintf(stderr, "%s -t bad recovery type\n",
				     argv[0]);
			return (1);
		    }
		    break;
		case 'A':
		    acct_file = optarg;
		    break;
		case 'L':
		    log_file = optarg;
		    break;
		case 'M':
		    if (get_port(optarg, &pbs_mom_port, &pbs_mom_addr)) {
			(void)fprintf(stderr,"%s: bad -M %s\n", argv[0],optarg);
			return (1);
		    }
		    if (isalpha((int)*optarg)) {
			if ((pc = strchr(optarg, (int)':')) != NULL)
				*pc = '\0';
		    	mom_host = optarg;
		    }
		    break;
		case 'R':
		    if ((pbs_rm_port = atoi(optarg)) == 0) {
			(void)fprintf(stderr, "%s: bad -R %s\n",
				argv[0], optarg);
			return 1;
		    }
		    break;
		case 'S':
		    if(get_port(optarg, &pbs_scheduler_port,
							&pbs_scheduler_addr)) {
			(void)fprintf(stderr,"%s: bad -S %s\n", argv[0],optarg);
			return (1);
		    }
		    break;
		default:
		    (void)fprintf(stderr,"%s: unknown option: %c\n",argv[0],c);
		    return (1);
	    }
	}

	if (optind < argc) {
		(void)fprintf(stderr, "%s: invalid operand\n", argv[0]);
		return (1);
	}

	/* make sure no other server is running with this home directory */

	(void)sprintf(lockfile, "%s/%s/server.lock", path_home,PBS_SVR_PRIVATE);
	if ((lockfds = open(lockfile,O_CREAT | O_TRUNC | O_WRONLY, 0600)) < 0) {
		(void)sprintf(log_buffer, "%s: unable to open lock file",
			      msg_daemonname);
		(void)fprintf(stderr, "%s\n", log_buffer);
		log_err(errno, msg_daemonname, log_buffer);
		return (2);
	}
	lock_out(lockfds, F_WRLCK);
	
	server.sv_started = time(&time_now);	/* time server started */

	/*
	 * Open the log file so we can start recording events 
	 *
	 * set log_event_mask to point to the log_event attribute value so
	 * it controls which events are logged.
	 */
	log_event_mask = &server.sv_attr[SRV_ATR_log_events].at_val.at_long;
	(void)sprintf(path_log, "%s/%s", path_home, PBS_LOGFILES);
	(void)log_open(log_file, path_log);
	(void)sprintf(log_buffer, msg_startup1, server_name, server_init_type);
	log_event(PBSEVENT_SYSTEM | PBSEVENT_ADMIN | PBSEVENT_FORCE,
		  PBS_EVENTCLASS_SERVER, msg_daemonname, log_buffer);

	/* initialize the server objects and perform specified recovery */
	/* will be left in the server's private directory		*/

	if (pbsd_init(server_init_type) != 0) {
		log_err(-1, msg_daemonname, "pbsd_init failed");
		return (3);
	}

	/* initialize the network interface */

	(void)sprintf(log_buffer, "Using ports Server:%d Scheduler:%d MOM:%d",
		      pbs_server_port_dis, pbs_scheduler_port, pbs_mom_port);
	log_event(PBSEVENT_SYSTEM|PBSEVENT_ADMIN, PBS_EVENTCLASS_SERVER,
		  msg_daemonname, log_buffer);

	if (init_network(pbs_server_port_dis, process_request) != 0) {
		perror("pbs_server: network");
		log_err(-1, msg_daemonname, "init_network failed dis");
		return (3);
	}
	net_set_type(Secondary, FromClientDIS);	/* "there" */

	/* go into the background and become own session/process group */

#ifndef DEBUG
	
	lock_out(lockfds, F_UNLCK);
	if (fork() > 0)
		exit(0);	/* parent goes away */

	if ((sid = setsid()) == -1) {
		log_err(errno, msg_daemonname, "setsid failed");
		return (2);
	}
	lock_out(lockfds, F_WRLCK);
	(void)fclose(stdin);
	(void)fclose(stdout);
	(void)fclose(stderr);
	dummyfile = fopen("/dev/null", "r");
	assert( (dummyfile != 0) && (fileno(dummyfile) == 0) );
	dummyfile = fopen("/dev/null", "w");
	assert( (dummyfile != 0) && (fileno(dummyfile) == 1) );
	dummyfile = fopen("/dev/null", "w");
	assert( (dummyfile != 0) && (fileno(dummyfile) == 2) );

#else
	sid = getpid();
	(void)setvbuf(stdout, NULL, _IOLBF, 0);
	(void)setvbuf(stderr, NULL, _IOLBF, 0);
#endif
	(void)sprintf(log_buffer, "%d\n", sid);
	(void)write(lockfds, log_buffer, strlen(log_buffer));
#if (PLOCK_DAEMONS & 1)
	(void)plock(PROCLOCK);
#endif

	if ((rppfd = rpp_bind(pbs_server_port_dis)) == -1) {
		log_err(errno, msg_daemonname, "rpp_bind");
		return (1);
	}
	rpp_fd = -1;		/* force rpp_bind() to get another socket */
	tryport = IPPORT_RESERVED;
	while (--tryport > 0) {
		if ((privfd = rpp_bind(tryport)) != -1)
			break;
		if ((errno != EADDRINUSE) && (errno != EADDRNOTAVAIL))
			break;
	}
	if (privfd == -1) {
		log_err(errno, msg_daemonname, "no privileged ports");
		return(1);
	}

	add_conn(rppfd, Primary, (pbs_net_t)0, 0, rpp_request);
	add_conn(privfd, Primary, (pbs_net_t)0, 0, rpp_request);

	/* record the fact that we are up and running */

	sprintf(log_buffer, msg_startup2, sid);
	log_event(PBSEVENT_SYSTEM | PBSEVENT_FORCE, PBS_EVENTCLASS_SERVER,
		  msg_daemonname, log_buffer);

	(void)set_task(WORK_Immed, 0, ping_nodes, NULL);
	/*
	 * Now at last, we are read to do some batch work, the
	 * following section constitutes the "main" loop of the server
	 */

	state  = &server.sv_attr[(int)SRV_ATR_State].at_val.at_long;
	if (server_init_type == RECOV_HOT)
		*state = SV_STATE_HOT;
	else
		*state = SV_STATE_RUN;

	while (*state != SV_STATE_DOWN) {

		/* first process any task whose time delay has expired */

		waittime = next_task();

		if (*state == SV_STATE_RUN) {	/* In normal Run State */

			/* if time or event says to run scheduler, do it */

			if ((svr_do_schedule != SCH_SCHEDULE_NULL) && server.sv_attr[(int)SRV_ATR_scheduling].at_val.at_long) {
				server.sv_next_schedule = time_now + server.sv_attr[(int)SRV_ATR_schedule_iteration].at_val.at_long;
				(void)schedule_jobs();
			}
	
		} else if (*state == SV_STATE_HOT) {

			/* Are there HOT jobs to rerun */
			/* only try every _CYCLE seconds */

			if (time_now > server.sv_hotcycle + SVR_HOT_CYCLE) {
				server.sv_hotcycle = time_now + SVR_HOT_CYCLE;
				c = start_hot_jobs();
			}

			/* If more than _LIMIT seconds since start, stop */

			if ((c == 0) || 
			    (time_now > server.sv_started + SVR_HOT_LIMIT)) {
				server_init_type = RECOV_WARM;
				*state = SV_STATE_RUN;
			}
		}

		/* any jobs to route today */

		pque = (pbs_queue *)GET_NEXT(svr_queues);
		while (pque) {
			if (pque->qu_qs.qu_type == QTYPE_RoutePush)
				queue_route(pque);
			pque = (pbs_queue *)GET_NEXT(pque->qu_link);
		}

		/* touch the rpp streams that need to send */

		rpp_request(42);

		/* wait for a request and process it */

		if (wait_request(waittime) != 0) {
			log_err(-1, msg_daemonname, "wait_requst failed");
		}

		if (*state == SV_STATE_SHUTSIG) 
			(void)svr_shutdown(SHUT_SIG);	/* caught sig */

		/*
		 * if in process of shuting down and all running jobs
		 * and all children are done, change state to DOWN
		 */

		if ((*state > SV_STATE_RUN) &&
		    (server.sv_jobstates[JOB_STATE_RUNNING] == 0) &&
		    (server.sv_jobstates[JOB_STATE_EXITING] == 0) &&
		    ((void *)GET_NEXT(task_list_event) == (void *)0))
			*state = SV_STATE_DOWN;

	}

	svr_save(&server, SVR_SAVE_FULL);	/* final recording of server */
	track_save((struct work_task *)0);	/* save tracking data	     */

	/* save any jobs that need saving */
	for ( pjob = (job *)GET_NEXT(svr_alljobs); 
	      pjob;
	      pjob = (job *)GET_NEXT(pjob->ji_alljobs) ) {
		if (pjob->ji_modified)
			(void)job_save(pjob, SAVEJOB_FULL);
	}

	if (svr_chngNodesfile) {/*nodes created/deleted, or props changed and*/
				/*update in req_manager failed; try again    */
	   (void)update_nodes_file ();
	}

	rpp_shutdown();
	shutdown_ack();
	net_close(-1);		/* close all network connections */
	log_event(PBSEVENT_SYSTEM | PBSEVENT_FORCE, PBS_EVENTCLASS_SERVER,
		  msg_daemonname, msg_svrdown);
	acct_close();
	log_close(1);
	return (0);
}

/*
 * get_port - parse host:port for -M and -S option
 *	Returns into *port and *addr if and only if that part is specified
 *	Both port and addr are returned in HOST byte order.
 *	Function return: 0=ok, -1=error
 */

static int get_port(arg, port, addr)
	char	  *arg;		/* "host", "port", ":port", or "host:port" */
	unsigned int *port;	/* RETURN: new port iff one given 	   */
	pbs_net_t    *addr;	/* RETURN: daemon's address iff host given */
{
	char *name;

	if (*arg == ':')
		++arg;
	if (isdigit((int)*arg)) {	/* port only specified */
		*port = (unsigned int)atoi(arg);
	} else {
		name = parse_servername(arg, port);
		if (name) {
			*addr = get_hostaddr(name);
		} else {
			return (-1);
		}
	}
	if ((*port <= 0) || (*addr == 0))
		return (-1);
	return 0;
}


/*
 * next_task - look for the next work task to perform:
 *	1. If svr_delay_entry is set, then a delayed task is ready so
 *	   find and process it.
 *	2. All items on the immediate list, then
 *	3. All items on the timed task list which have expired times
 *
 *	Returns: amount of time till next task
 */

static time_t next_task()
{
	
	time_t		   delay;
	struct work_task  *nxt;
	struct work_task  *ptask;
	time_t		   tilwhen = server.sv_attr[(int)SRV_ATR_schedule_iteration].at_val.at_long;


	time_now = time((time_t *)0);

	if (svr_delay_entry) {
		ptask = (struct work_task *)GET_NEXT(task_list_event);
		while (ptask) {
			nxt = (struct work_task *)GET_NEXT(ptask->wt_linkall);
			if (ptask->wt_type == WORK_Deferred_Cmp)
				dispatch_task(ptask);
			ptask = nxt;
		}
		svr_delay_entry = 0;
	}

	while (ptask = (struct work_task *)GET_NEXT(task_list_immed)) 
		dispatch_task(ptask);

	while (ptask = (struct work_task *)GET_NEXT(task_list_timed)) {
		if ((delay = ptask->wt_event - time_now) > 0) {
			if (tilwhen > delay) 
				tilwhen = delay;
			break;
		} else {
			dispatch_task(ptask);	/* will delete link */
		}
		
	}

	/* should the scheduler be run?  If so, adjust the delay time  */

	if ((delay = server.sv_next_schedule - time_now) <= 0)
		svr_do_schedule = SCH_SCHEDULE_TIME;
	else if (delay < tilwhen)
			tilwhen = delay;

	return (tilwhen);
}

/*
 * start_hot_jobs - place any job which is state QUEUED and has the
 *	HOT start flag set into execution.
 * 
 *	Returns the number of jobs to be hot started.
 */

static int start_hot_jobs()
{
	int  ct = 0;
	job *pjob;

	pjob = (job *)GET_NEXT(svr_alljobs);
	while (pjob) {
		if ((pjob->ji_qs.ji_substate == JOB_SUBSTATE_QUEUED) &&
		    (pjob->ji_qs.ji_svrflags & JOB_SVFLG_HOTSTART)) {
			log_event(PBSEVENT_SYSTEM, PBS_EVENTCLASS_JOB,
				  pjob->ji_qs.ji_jobid, 
				  "attempting to hot start job");
			(void)svr_startjob(pjob, 0);
			ct++;
		}
		pjob = (job *)GET_NEXT(pjob->ji_alljobs);
	}
	return (ct);
}

/*
 * lock_out - lock out other daemons from this directory.
 */

static void lock_out(fds, op)
	int fds;
	int op;		/* F_WRLCK  or  F_UNLCK */
{
	struct flock flock;

	flock.l_type   = op;
	flock.l_whence = SEEK_SET;
	flock.l_start  = 0;
	flock.l_len    = 0;
	if (fcntl(fds, F_SETLK, &flock) < 0) {
	    (void)strcpy(log_buffer, "pbs_server: another server running\n");
	    log_err(errno, msg_daemonname, log_buffer);
	    fprintf(stderr, log_buffer);
	    exit (1);
	}
}
