/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/

#include <pbs_config.h>   /* the master config generated by configure */

#include <errno.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <memory.h>
#include "libpbs.h"
#include "pbs_error.h"
#include "server_limits.h"
#include "list_link.h"
#include "attribute.h"
#include "job.h"
#include "server.h"
#include "credential.h"
#include "batch_request.h"
#include "net_connect.h"
#include "log.h"
#include "svrfunc.h"

static char ident[] = "@(#) $RCSfile: process_request.c,v $ $Revision: 2.2 $";

/*
 * process_request - this function gets, checks, and invokes the proper
 *	function to deal with a batch request received over the network.
 *
 *	All data encoding/decoding dependencies are moved to a lower level
 *	routine.  That routine must convert
 *	the data received into the internal server structures regardless of
 *	the data structures used by the encode/decode routines.  This provides
 *	the "protocol" and "protocol generation tool" freedom to the bulk
 *	of the server.
 */

/* global data items */

list_head svr_requests;

extern struct    connection svr_conn[];
extern struct    credential conn_credent[PBS_NET_MAX_CONNECTIONS];
extern struct server server;
extern char      server_host[];
extern list_head svr_newjobs;
extern time_t    time_now;
extern char  *msg_err_noqueue;
extern char  *msg_err_malloc;
extern char  *msg_reqbadhost;
extern char  *msg_request;

/* Private functions local to this file */

static void close_client A_((int sfds));
static void free_rescrq A_((struct rq_rescq *));
static void freebr_manage A_((struct rq_manage *)); 
static void freebr_cpyfile A_((struct rq_cpyfile *));
static void close_quejob A_((int sfds));



/*
 * process_request - process an request from the network:
 *	Call function to read in the request and decode it.
 *	Validate requesting host and user.
 *	Call function to process request based on type.
 *		That function MUST free the request by calling free_br()
 */

void process_request(sfds)
	int	sfds;		/* file descriptor (socket) to get request */
{
	int		      rc;
	struct batch_request *request;

	time_now = time((time_t *)0);

	request = alloc_br(0);
	request->rq_conn = sfds;

	/*
	 * Read in the request and decode it to the internal request structure.
	 */

#ifndef PBS_MOM

	if (svr_conn[sfds].cn_active == FromClientDIS) {
		rc = dis_request_read(sfds, request);
	} else {
		LOG_EVENT(PBSEVENT_SYSTEM, PBS_EVENTCLASS_REQUEST,"process_req",
			  "request on invalid type of connection");
		close_conn(sfds);
		free_br(request);
		return;
	}
#else	/* PBS_MOM */
	rc = dis_request_read(sfds, request);
#endif	/* PBS_MOM */

	if (rc == -1) {		/* End of file */
		close_client(sfds);
		free_br(request);
		return;

	} else if ((rc == PBSE_SYSTEM) || (rc == PBSE_INTERNAL)) {

		/* read error, likely cannot send reply so just disconnect */

		/* ??? not sure about this ??? */

		close_client(sfds);
		free_br(request);
		return;

	} else if (rc > 0) {

		/*
		 * request didn't decode, either garbage or  unknown
		 * request type, in ether case, return reject-reply
		 */

		req_reject(rc, 0, request);
		close_client(sfds);
		return;
	}

	if (get_connecthost(sfds, request->rq_host, PBS_MAXHOSTNAME)) {

		(void)sprintf(log_buffer, "%s: %lu", msg_reqbadhost,
			      get_connectaddr(sfds));
		LOG_EVENT(PBSEVENT_DEBUG, PBS_EVENTCLASS_REQUEST,"",log_buffer);
		req_reject(PBSE_BADHOST, 0, request);
		return;
	}

	(void)sprintf(log_buffer, msg_request, request->rq_type,
			request->rq_user, request->rq_host,sfds);
	LOG_EVENT(PBSEVENT_DEBUG2, PBS_EVENTCLASS_REQUEST, "", log_buffer);

	/* is the request from a host acceptable to the server */

#ifndef PBS_MOM
	if (server.sv_attr[(int)SRV_ATR_acl_host_enable].at_val.at_long) {

		/* acl enabled, check it; always allow myself	*/

		if ((acl_check(&server.sv_attr[(int)SRV_ATR_acl_hosts],
		     request->rq_host, ACL_Host) == 0) &&
		    (strcmp(server_host, request->rq_host) != 0)) {

			req_reject(PBSE_BADHOST, 0, request);
			close_client(sfds);
			return;
		}
	}

	/*
	 * determine source (user client or another server) of request.
	 * set the permissions granted to the client
	 */

	if (svr_conn[sfds].cn_authen == PBS_NET_CONN_FROM_PRIVIL) {

		/* request came from another server */

		request->rq_fromsvr = 1;
		request->rq_perm = ATR_DFLAG_USRD | ATR_DFLAG_USWR |
				   ATR_DFLAG_OPRD | ATR_DFLAG_OPWR |
				   ATR_DFLAG_MGRD | ATR_DFLAG_MGWR |
				   ATR_DFLAG_SvWR;

	} else {

		/* request not from another server */

		request->rq_fromsvr = 0;

		/*
		 * Client must be authenticated by a Authenticate User Request,
		 * if not, reject request and close connection.
		 * -- The following is retained for compat with old cmds --
		 * The exception to this is of course the Connect Request which
		 * cannot have been authenticated, because it contains the 
		 * needed ticket; so trap it here.  Of course, there is no
		 * prior authentication on the Authenticate User request either,
		 * but it comes over a reserved port and appears from another
		 * server, hence is automatically granted authorization.

		 */
	
		if (request->rq_type == PBS_BATCH_Connect) {
			req_connect(request);
			return;
		}
		if (svr_conn[sfds].cn_authen != PBS_NET_CONN_AUTHENTICATED)
			rc = PBSE_BADCRED;
		else
			rc = authenticate_user(request, &conn_credent[sfds]);
		if (rc != 0) {
			req_reject(rc, 0, request);
			close_client(sfds);
			return;
		}
	
		request->rq_perm = 
			  svr_get_privilege(request->rq_user, request->rq_host);
	}

	/* if server shutting down, disallow new jobs and new running */

	if (server.sv_attr[(int)SRV_ATR_State].at_val.at_long > SV_STATE_RUN) {
		switch (request->rq_type) {
		    case PBS_BATCH_AsyrunJob:
		    case PBS_BATCH_JobCred:
		    case PBS_BATCH_MoveJob:
		    case PBS_BATCH_QueueJob:
		    case PBS_BATCH_RunJob:
		    case PBS_BATCH_StageIn:
		    case PBS_BATCH_jobscript:
			req_reject(PBSE_SVRDOWN, 0, request);
			return;
		}
	}
	

#else	/* THIS CODE FOR MOM ONLY */

	{
		/* check connecting host against allowed list of ok clients */

		extern void	*okclients;
	
		if (!tfind(svr_conn[sfds].cn_addr, &okclients)) {
			req_reject(PBSE_BADHOST, 0, request);
			close_client(sfds);
			return;
		}
	}
		
	request->rq_fromsvr = 1;
	request->rq_perm = ATR_DFLAG_USRD | ATR_DFLAG_USWR |
			   ATR_DFLAG_OPRD | ATR_DFLAG_OPWR |
			   ATR_DFLAG_MGRD | ATR_DFLAG_MGWR |
			   ATR_DFLAG_SvWR | ATR_DFLAG_MOM;
#endif

	/*
	 * dispatch the request to the correct processing function.
	 * The processing function must call reply_send() to free
	 * the request struture.
	 */

	dispatch_request(sfds, request);
	return;
}

/*
 * dispatch_request - Determine the request type and invoke the corresponding
 *	function.  The function will perform the request action and return the
 *	reply.  The function MUST also reply and free the request by calling
 *	reply_send().
 */

void dispatch_request(sfds, request)
	int		      sfds;
	struct batch_request *request;
{

	switch (request->rq_type) {

 		case PBS_BATCH_QueueJob:
			net_add_close_func(sfds, close_quejob);
 			req_quejob(request);
 			break; 

		case PBS_BATCH_JobCred:
			req_jobcredential(request);
			break;

 		case PBS_BATCH_jobscript:
 			req_jobscript(request);
 			break;

 		case PBS_BATCH_RdytoCommit:
 			req_rdytocommit(request);
 			break;

 		case PBS_BATCH_Commit:
 			req_commit(request);
			net_add_close_func(sfds, (void (*)())0);
 			break; 

 		case PBS_BATCH_DeleteJob: 
 			req_deletejob(request); 
 			break; 

 		case PBS_BATCH_HoldJob: 
 			req_holdjob(request);
 			break;
#ifndef PBS_MOM
 		case PBS_BATCH_LocateJob: 
 			req_locatejob(request); 
 			break;

 		case PBS_BATCH_Manager: 
 			req_manager(request);
 			break;
#endif
 		case PBS_BATCH_MessJob: 
 			req_messagejob(request); 
 			break;

 		case PBS_BATCH_ModifyJob:
 			req_modifyjob(request);
 			break; 

 		case PBS_BATCH_Rerun:
 			req_rerunjob(request); 
 			break;
#ifndef PBS_MOM
 		case PBS_BATCH_MoveJob:
 			req_movejob(request); 
 			break;

 		case PBS_BATCH_OrderJob:
 			req_orderjob(request); 
 			break;

		case PBS_BATCH_Rescq:
			req_rescq(request);
			break;

		case PBS_BATCH_ReserveResc:
			req_rescreserve(request);
			break;

		case PBS_BATCH_ReleaseResc:
			req_rescfree(request);
			break;

 		case PBS_BATCH_ReleaseJob:
 			req_releasejob(request);
 			break;

 		case PBS_BATCH_RunJob:
		case PBS_BATCH_AsyrunJob:
 			req_runjob(request); 
 			break;

 		case PBS_BATCH_SelectJobs:
 		case PBS_BATCH_SelStat:
 			req_selectjobs(request); 
 			break;
#endif
 		case PBS_BATCH_Shutdown: 
 			req_shutdown(request); 
 			break;

 		case PBS_BATCH_SignalJob:
 			req_signaljob(request); 
 			break;

 		case PBS_BATCH_StatusJob:
 			req_stat_job(request);
 			break;

		case PBS_BATCH_MvJobFile:
			req_mvjobfile(request);
			break;

#ifndef PBS_MOM		/* Server Only Functions */

 		case PBS_BATCH_StatusQue: 
 			req_stat_que(request); 
 			break;

 		case PBS_BATCH_StatusNode: 
 			req_stat_node(request); 
 			break;

 		case PBS_BATCH_StatusSvr: 
 			req_stat_svr(request);
 			break;

 		case PBS_BATCH_TrackJob:
			req_track(request); 
 			break;

		case PBS_BATCH_RegistDep:
			req_register(request);
			break;

		case PBS_BATCH_AuthenUser:
			req_authenuser(request);
			break;

		case PBS_BATCH_JobObit:
			req_jobobit(request);
			break;

		case PBS_BATCH_StageIn:
			req_stagein(request);
			break;

#else			/* MOM only functions */

		case PBS_BATCH_CopyFiles:
			req_cpyfile(request);
			break;

		case PBS_BATCH_DelFiles:
			req_delfile(request);
			break;
#endif
		default:
			req_reject(PBSE_UNKREQ, 0, request);
			break;
	}
	return;
}

/*
 * close_client - close a connection to a client, also "inactivate"
 *		  any outstanding batch requests on that connection.
 */

static void close_client(sfds)
	int sfds;	/* connection socket */
{
	struct batch_request *preq;

	close_conn(sfds);	/* close the connection */
	preq = (struct batch_request *)GET_NEXT(svr_requests);	
	while (preq) {			/* list of outstanding requests */
		if (preq->rq_conn == sfds)
			preq->rq_conn = -1;
		if (preq->rq_orgconn == sfds)
			preq->rq_orgconn = -1;
		preq = (struct batch_request *)GET_NEXT(preq->rq_link);
	}
}

/*
 * alloc_br - allocate and clear a batch_request structure
 */

struct batch_request *alloc_br(type)
	int type;
{
	struct batch_request *req;

	req= (struct batch_request *)malloc(sizeof (struct batch_request));
	if (req== (struct batch_request *)0)
		log_err(errno, "alloc_br", msg_err_malloc);
	else {
		memset((void *)req, (int)0, sizeof (struct batch_request));
		req->rq_type = type;
		CLEAR_LINK(req->rq_link);
		req->rq_conn = -1;		/* indicate not connected */
		req->rq_orgconn = -1;		/* indicate not connected */
		req->rq_time = time_now;
		req->rq_reply.brp_choice = BATCH_REPLY_CHOICE_NULL;
		append_link(&svr_requests, &req->rq_link, req);
	}
	return (req);
}

/*
 * close_quejob - locate and deal with the new job that was being recevied
 *		  when the net connection closed.
 */

static void close_quejob(sfds)
	int sfds;
{
	job *pjob;

	pjob = (job *)GET_NEXT(svr_newjobs);
	while (pjob  != (struct job *)0) {
	    if (pjob->ji_qs.ji_un.ji_newt.ji_fromsock == sfds) {
		if (pjob->ji_qs.ji_substate == JOB_SUBSTATE_TRANSICM) {

#ifndef PBS_MOM
			if (pjob->ji_qs.ji_svrflags & JOB_SVFLG_HERE) {
	
			/*
			 * the job was being created here for the first time 
			 * go ahead and enqueue it as QUEUED; otherwise, hold
			 * it here as TRANSICM until we hear from the sending
			 * server again to commit.
			 */
				delete_link(&pjob->ji_alljobs);
				pjob->ji_qs.ji_state = JOB_STATE_QUEUED;
				pjob->ji_qs.ji_substate = JOB_SUBSTATE_QUEUED;
				if (svr_enquejob(pjob))
					(void)job_abt(pjob, msg_err_noqueue);
					
			}
#endif	/* PBS_MOM */

		} else {

			/* else delete the job */

			delete_link(&pjob->ji_alljobs);
			job_purge(pjob);
		}
		break;
	    }
	    pjob = GET_NEXT(pjob->ji_alljobs);
	}
	return;
}


/*
 * free_br - free space allocated to a batch_request structure
 *	including any sub-structures
 */

void free_br(preq)
	struct batch_request *preq;
{
	delete_link(&preq->rq_link);
	reply_free(&preq->rq_reply);
	if (preq->rq_extend)
		(void)free(preq->rq_extend);

	switch (preq->rq_type) {
	    case PBS_BATCH_QueueJob:
		free_attrlist(&preq->rq_ind.rq_queuejob.rq_attr);
		break;
	    case PBS_BATCH_JobCred:
		if (preq->rq_ind.rq_jobcred.rq_data)
			(void)free(preq->rq_ind.rq_jobcred.rq_data);
		break;
	    case PBS_BATCH_jobscript:
		if (preq->rq_ind.rq_jobfile.rq_data)
			(void)free(preq->rq_ind.rq_jobfile.rq_data);
		break;
	    case PBS_BATCH_HoldJob:
		freebr_manage(&preq->rq_ind.rq_hold.rq_orig);
		break;
	    case PBS_BATCH_MessJob:
		if (preq->rq_ind.rq_message.rq_text)
			(void)free(preq->rq_ind.rq_message.rq_text);
		break;
	    case PBS_BATCH_ModifyJob:
		freebr_manage(&preq->rq_ind.rq_modify);
		break;
	    case PBS_BATCH_StatusJob:
	    case PBS_BATCH_StatusQue:
	    case PBS_BATCH_StatusNode:
	    case PBS_BATCH_StatusSvr:
		free_attrlist(&preq->rq_ind.rq_status.rq_attr);
		break;
	    case PBS_BATCH_JobObit:
		free_attrlist(&preq->rq_ind.rq_jobobit.rq_attr);
		break;
	    case PBS_BATCH_CopyFiles:
	    case PBS_BATCH_DelFiles:
		freebr_cpyfile(&preq->rq_ind.rq_cpyfile);
		break;

#ifndef PBS_MOM		/* Server Only */

	    case PBS_BATCH_Manager:
		freebr_manage(&preq->rq_ind.rq_manager);
		break;
	    case PBS_BATCH_ReleaseJob:
		freebr_manage(&preq->rq_ind.rq_release);
		break;
	    case PBS_BATCH_Rescq:
		free_rescrq(&preq->rq_ind.rq_rescq);
		break;
	    case PBS_BATCH_SelectJobs:
 	    case PBS_BATCH_SelStat:
		free_attrlist(&preq->rq_ind.rq_select);
		break;
#endif
	}
	(void)free(preq);
}

static void freebr_manage(pmgr) 
	struct rq_manage *pmgr;
{
	free_attrlist(&pmgr->rq_attr);
}

static void freebr_cpyfile(pcf)
	struct rq_cpyfile *pcf;
{
	struct rqfpair *ppair;

	while (ppair = (struct rqfpair *)GET_NEXT(pcf->rq_pair)) {
		delete_link(&ppair->fp_link);
		if (ppair->fp_local)
			(void)free(ppair->fp_local);
		if (ppair->fp_rmt)
			(void)free(ppair->fp_rmt);
		(void)free(ppair);
	}
}

static void free_rescrq(pq)
	struct rq_rescq *pq;
{
	int i;

	i = pq->rq_num;
	while (i--) {
		if (*(pq->rq_list + i))
			(void)free(*(pq->rq_list + i));
	}
	if (pq->rq_list)
		(void)free(pq->rq_list);
}
