/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
/*
 * This file contains the routines used to send a reply to a client following
 * the processing of a request.  The following routines are provided here:
 *
 *	reply_send()  - the main routine, used by all reply senders
 *	reply_ack()   - send a basic no error acknowledgement
 *	req_reject()  - send a basic error return 
 *	reply_text()  - send a return with a supplied text string
 *	reply_jobid() - used by several requests where the job id must be sent
 *	reply_free()  - free the substructure that might hang from a reply
 */

#include <pbs_config.h>   /* the master config generated by configure */

#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include "libpbs.h"
#include "dis.h"
#include "log.h"
#include "pbs_error.h"
#include "server_limits.h"
#include "list_link.h"
#include "net_connect.h"
#include "attribute.h"
#include "credential.h"
#include "batch_request.h"
#include "work_task.h"

static char ident[] = "@(#) $RCSfile: reply_send.c,v $ $Revision: 2.1 $";

/* External Globals */

extern struct connection svr_conn[];

extern char *msg_daemonname;
extern char *msg_system;

#ifndef PBS_MOM
extern list_head task_list_event;
extern list_head task_list_immed;
#endif	/* PBS_MOM */

extern struct pbs_err_to_txt pbs_err_to_txt[];

#define ERR_MSG_SIZE 127


static void set_err_msg(code, msgbuf)
	int   code;
	char *msgbuf;
{
	char *msg = (char *)0;
	char *msg_tmp;

	/* see if there is an error message associated with the code */

	*msgbuf = '\0';
	if (code == PBSE_SYSTEM) {
	    (void)strcpy(msgbuf, msg_daemonname);
	    (void)strcat(msgbuf, msg_system);
	    msg_tmp = strerror(errno);

	    if (msg_tmp)
		(void)strncat(msgbuf, strerror(errno), 
						ERR_MSG_SIZE - strlen(msgbuf));
	    else
	        strcat(msgbuf, "Unknown error");

		
	} else if ( code > PBSE_ ) {
		msg = pbse_to_txt(code);

	} else {
		msg = strerror(code);
	}

	if (msg) {
		(void)strncpy(msgbuf, msg, ERR_MSG_SIZE);
	}
	msgbuf[ERR_MSG_SIZE] = '\0';
}

static int dis_reply_write(sfds, preply)
	int		    sfds;
	struct batch_reply *preply;
{
	int rc;

	DIS_tcp_setup(sfds);		/* setup for DIS over tcp */
	if ((rc = encode_DIS_reply(sfds, preply)) ||
	    (rc = DIS_tcp_wflush(sfds)) ) {
		(void)sprintf(log_buffer, "DIS reply failure, %d", rc);
		LOG_EVENT(PBSEVENT_SYSTEM, PBS_EVENTCLASS_REQUEST,
			  "dis_reply_write", log_buffer);
		close_conn(sfds);
	}
	return rc;
}

/*
 * reply_send - Send a reply to a batch request, reply either goes to
 *	remote client over the network:
 *	Encode the reply to a "presentation element",
 *	allocate the presenetation stream and attach to socket,
 *	write out reply, and free ps, pe, and isoreply structures.
 *
 *	Or the reply is for a request from the local server:
 *	locate the work task associated with the request and displatch it
 *
 *	The request (and reply) structures are freed.
 */

int reply_send(request)
	struct batch_request *request;
{
	struct work_task   *ptask;
	int		    rc = 0;
	static char	   *id = "reply_send";
	int		    sfds = request->rq_conn;		/* socket */

	/* determine where the reply should go, remote or local */

	if (sfds == PBS_LOCAL_CONNECTION) {

#ifndef PBS_MOM
		/*
		 * reply stays local, find work task and move it to
		 * the immediate list for dispatching.
		 */

		ptask = (struct work_task *)GET_NEXT(task_list_event);
		while (ptask) {
			if ((ptask->wt_type == WORK_Deferred_Local) &&
			    (ptask->wt_parm1 == (void *)request)) {
				delete_link(&ptask->wt_linkall);
				append_link(&task_list_immed,
					    &ptask->wt_linkall, ptask);
				return(0);
			}
			ptask = (struct work_task *)GET_NEXT(ptask->wt_linkall);
		}

		/* Uh Oh, should have found a task and didn't */

		log_err(-1, id, "did not find work task for local request");
#endif	/* PBS_MOM */
		rc = PBSE_SYSTEM;

	} else if (sfds >= 0) {

		/*
		 * Otherwise, the reply is to be sent to a remote client 
	 	*/

		rc = dis_reply_write(sfds, &request->rq_reply);
	}

	free_br(request);
	return (rc);
}


/*
 * reply_ack - Send a normal acknowledgement reply to a request
 *
 *	Always frees the request structure.
 */

void reply_ack(preq)
	struct batch_request *preq;
{

	preq->rq_reply.brp_code    = PBSE_NONE;
	preq->rq_reply.brp_auxcode = 0;
	preq->rq_reply.brp_choice  = BATCH_REPLY_CHOICE_NULL;
	(void)reply_send(preq);
}

/*
 * reply_free - free any sub-struttures that might hang from the basic
 *	batch_reply structure, the reply structure itself IS NOT FREED.
 */

void reply_free(prep)
	struct batch_reply *prep;
{
	struct brp_status  *pstat;
	struct brp_status  *pstatx;
	struct brp_select  *psel;
	struct brp_select  *pselx;

	if (prep->brp_choice == BATCH_REPLY_CHOICE_Text) {
		if (prep->brp_un.brp_txt.brp_str) {
			(void)free(prep->brp_un.brp_txt.brp_str);
			prep->brp_un.brp_txt.brp_str = (char *)0;
			prep->brp_un.brp_txt.brp_txtlen = 0;
		}

	} else if (prep->brp_choice == BATCH_REPLY_CHOICE_Select) {
		psel = prep->brp_un.brp_select;
		while (psel) {
		    pselx = psel->brp_next;
		    (void)free(psel);
		    psel = pselx;
		}
		
	} else if (prep->brp_choice == BATCH_REPLY_CHOICE_Status) {
		pstat = (struct brp_status *)GET_NEXT(prep->brp_un.brp_status);
		while (pstat) {
		    pstatx = (struct brp_status *)GET_NEXT(pstat->brp_stlink);
		    free_attrlist(&pstat->brp_attr);
		    (void)free(pstat);
		    pstat = pstatx;
		}
	} else if (prep->brp_choice == BATCH_REPLY_CHOICE_RescQuery) {
		(void)free(prep->brp_un.brp_rescq.brq_avail);
		(void)free(prep->brp_un.brp_rescq.brq_alloc);
		(void)free(prep->brp_un.brp_rescq.brq_resvd);
		(void)free(prep->brp_un.brp_rescq.brq_down);
	}
	prep->brp_choice = BATCH_REPLY_CHOICE_NULL;
}

/*
 * req_reject - create a reject (error) reply for a request and send it
 *
 *	Always frees the request structure.
 */

void req_reject (code, aux, preq)
	int   code;
	int   aux;
	struct batch_request *preq;
{
	char  msgbuf[ERR_MSG_SIZE+1];

	(void)sprintf(log_buffer,
			"Reject reply code=%d, aux=%d, type=%d, from %s@%s",
			code, aux, preq->rq_type, preq->rq_user, preq->rq_host);
	LOG_EVENT(PBSEVENT_DEBUG, PBS_EVENTCLASS_REQUEST,
			"req_reject", log_buffer);
	set_err_msg(code, msgbuf);
	preq->rq_reply.brp_auxcode = aux;
	(void)reply_text(preq, code, msgbuf);
}

/*
 * reply_badattr - create a reject (error) reply for a request including
 *	the name of the bad attribute/resource
 */

void reply_badattr (code, aux, pal, preq)
	int   code;
	int   aux;
	svrattrl	     *pal;
	struct batch_request *preq;
{
	int   i = 1;
	char  msgbuf[ERR_MSG_SIZE+1];

	set_err_msg(code, msgbuf);
	
	while (pal) {
		if (i == aux) {
		    (void)strcat(msgbuf, " ");
		    (void)strcat(msgbuf, pal->al_name);
		    if (pal->al_resc) {
			(void)strcat(msgbuf, ".");
			(void)strcat(msgbuf, pal->al_resc);
		    }
		    break;
		}
		pal = (svrattrl *)GET_NEXT(pal->al_link);
		++i;
	}
	(void)reply_text(preq, code, msgbuf);
}

/*
 * reply_text - return a reply with a supplied text string
 */

void reply_text(preq, code, text)
	struct batch_request	*preq;
	int			 code;
	char			*text;
{
	if (preq->rq_reply.brp_choice != BATCH_REPLY_CHOICE_NULL)
		/* in case another reply was being built up, clean it out */
		reply_free(&preq->rq_reply);
	
	preq->rq_reply.brp_code    = code;
	preq->rq_reply.brp_auxcode = 0;
	if (text && *text) {
		preq->rq_reply.brp_choice  = BATCH_REPLY_CHOICE_Text;
		preq->rq_reply.brp_un.brp_txt.brp_str = strdup(text);
		preq->rq_reply.brp_un.brp_txt.brp_txtlen = strlen(text);
	} else {
		preq->rq_reply.brp_choice  = BATCH_REPLY_CHOICE_NULL;
	}
	(void)reply_send(preq);
}

/*
 * reply_jobid - return a reply with the job id, used by
 *	req_queuejob(), req_rdytocommit(), and req_commit()
 *
 *	Always frees the request structure.
 */

int reply_jobid(preq, jobid, which)
	struct batch_request	*preq;
	char			*jobid;
	int			 which;
{
	preq->rq_reply.brp_code    = 0;
	preq->rq_reply.brp_auxcode = 0;
	preq->rq_reply.brp_choice  = which;
	(void)strncpy(preq->rq_reply.brp_un.brp_jid, jobid, PBS_MAXSVRJOBID);
	return (reply_send(preq));
}
