/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
/*
 * req_message.c - functions dealing with sending a message 
 *		     to a running job.
 */

#include <pbs_config.h>   /* the master config generated by configure */

#include <sys/types.h>
#include "libpbs.h"
#include <signal.h>
#include "server_limits.h"
#include "list_link.h"
#include "attribute.h"
#include "server.h"
#include "credential.h"
#include "batch_request.h"
#include "job.h"
#include "work_task.h"
#include "pbs_error.h"
#include "log.h"
#include "svrfunc.h"

static char ident[] = "@(#) $RCSfile: req_message.c,v $ $Revision: 2.1 $";

/* Private Function local to this file */

static void post_message_req A_((struct work_task *));

/* Global Data Items: */

extern int   pbs_mom_port;
extern char *msg_messagejob;


/*
 * req_messagejob - service the Message Job Request
 *
 *	This request sends (via MOM) a message to a running job.
 */

void req_messagejob(preq)
	struct batch_request *preq;
{
	job		 *pjob;
	int		  rc;

	if ((pjob = chk_job_request(preq->rq_ind.rq_message.rq_jid, preq)) == 0)
		return;

	/* the job must be running */

	if (pjob->ji_qs.ji_state != JOB_STATE_RUNNING) {
		req_reject(PBSE_BADSTATE, 0, preq);
		return;
	}
	
	/* pass the request on to MOM */

	if (rc = relay_to_mom(pjob->ji_qs.ji_un.ji_exect.ji_momaddr,
			      preq, post_message_req))
		req_reject(rc, 0, preq);	/* unable to get to MOM */

	/* After MOM acts and replies to us, we pick up in post_message_req() */
}

/*
 * post_message_req - complete a Message Job Request
 */

static void post_message_req(pwt)
	struct work_task *pwt;
{
	struct batch_request *preq;

	svr_disconnect(pwt->wt_event);	/* close connection to MOM */
	preq = pwt->wt_parm1;
	preq->rq_conn = preq->rq_orgconn;  /* restore socket to client */

	(void)sprintf(log_buffer, msg_messagejob, preq->rq_reply.brp_code);
	log_event(PBSEVENT_JOB, PBS_EVENTCLASS_JOB, 
		  preq->rq_ind.rq_message.rq_jid, log_buffer);
	if (preq->rq_reply.brp_code)
		req_reject(preq->rq_reply.brp_code, 0, preq);
	else
		reply_ack(preq);
}
