/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
/*
 * req_movejob.c - function to move a job to another queue
 *
 * Included functions are:
 */

#include <pbs_config.h>   /* the master config generated by configure */

#include <stdio.h>
#include <sys/types.h>
#include <sys/param.h>
#include "libpbs.h"
#include <errno.h>

#include "server_limits.h"
#include "list_link.h"
#include "attribute.h"
#include "credential.h"
#include "batch_request.h"
#include "job.h"
#include "log.h"
#include "pbs_error.h"

static char ident[] = "@(#) $RCSfile: req_movejob.c,v $ $Revision: 2.1 $";

/* Global Data Items: */

extern char	*msg_unkjobid;
extern char	*msg_badstate;
extern char	*msg_manager;
extern char	*msg_movejob;
extern int	 pbs_errno;
extern char	*pbs_o_host;

/*
 * req_movejob = move a job to a new destination (local or remote)
 */

void req_movejob(req)
	struct batch_request	*req;
{
	char	*id = "req_movejob";
	job	*jobp;

	jobp = chk_job_request(req->rq_ind.rq_move.rq_jid, req);
	if (jobp == NULL) 
		return;

	if (jobp->ji_qs.ji_state != JOB_STATE_QUEUED &&
	    jobp->ji_qs.ji_state != JOB_STATE_HELD &&
	    jobp->ji_qs.ji_state != JOB_STATE_WAITING) {
#ifndef NDEBUG
		(void)sprintf(log_buffer, "%s %d", msg_badstate,
						   jobp->ji_qs.ji_state);
		(void)strcat(log_buffer, id);
		log_event(PBSEVENT_DEBUG, PBS_EVENTCLASS_JOB,
			  jobp->ji_qs.ji_jobid, log_buffer);
#endif /* NDEBUG */
		req_reject(PBSE_BADSTATE, 0, req);
		return;
	}

	/*
	 * svr_movejob() does the real work, handles both local and
	 * network moves
	 */

	switch (svr_movejob(jobp, req->rq_ind.rq_move.rq_destin, req)) {
	case 0:			/* success */
		(void)strcpy(log_buffer, msg_movejob);
		(void)sprintf(log_buffer+strlen(log_buffer), 
			      msg_manager, req->rq_ind.rq_move.rq_destin,
			      req->rq_user, req->rq_host);
		log_event(PBSEVENT_JOB, PBS_EVENTCLASS_JOB,
			  jobp->ji_qs.ji_jobid, log_buffer);
		reply_ack(req);
		break;
	case -1:
	case 1:			/* fail */
		req_reject(pbs_errno, 0, req);
		break;
	case 2:			/* deferred, will be handled by 	   */
				/* post_movejob() when the child completes */
		break;
	}
	return;
}
/*
 * req_orderjob = reorder the jobs in a queue
 */

void req_orderjob(req)
	struct batch_request	*req;
{
	char	*id = "req_orderjob";
	job	*pjob;
	job	*pjob1;
	job	*pjob2;
	int	 rank;
	int	 rc;
	char	 tmpqn[PBS_MAXQUEUENAME+1];

	if ((pjob1=chk_job_request(req->rq_ind.rq_move.rq_jid, req)) == NULL)
		return;
	if ((pjob2=chk_job_request(req->rq_ind.rq_move.rq_destin,req)) == NULL)
		return;

	if (((pjob = pjob1)->ji_qs.ji_state == JOB_STATE_RUNNING) ||
	    ((pjob = pjob2)->ji_qs.ji_state == JOB_STATE_RUNNING)) {
#ifndef NDEBUG
		(void)sprintf(log_buffer, "%s %d", msg_badstate,
						   pjob->ji_qs.ji_state);
		(void)strcat(log_buffer, id);
		log_event(PBSEVENT_DEBUG, PBS_EVENTCLASS_JOB,
			  pjob->ji_qs.ji_jobid, log_buffer);
#endif	/* NDEBUG */
		req_reject(PBSE_BADSTATE, 0, req);
		return;
	} else if (pjob1->ji_qhdr != pjob2->ji_qhdr) {

		/* Jobs are in different queues */

		if ((rc = svr_chkque(pjob1, pjob2->ji_qhdr,
				     get_variable(pjob1, pbs_o_host),
				     MOVE_TYPE_Order)) ||
		    (rc = svr_chkque(pjob2, pjob1->ji_qhdr,
				     get_variable(pjob2, pbs_o_host),
				     MOVE_TYPE_Order))) {
			req_reject(rc, 0, req);
			return;
		}
	}

	/* now swap the order of the two jobs in the queue lists */

	rank = pjob1->ji_wattr[(int)JOB_ATR_qrank].at_val.at_long;
	pjob1->ji_wattr[(int)JOB_ATR_qrank].at_val.at_long = 
			pjob2->ji_wattr[(int)JOB_ATR_qrank].at_val.at_long;
	pjob2->ji_wattr[(int)JOB_ATR_qrank].at_val.at_long = rank;

	if (pjob1->ji_qhdr != pjob2->ji_qhdr) {
		(void)strcpy(tmpqn, pjob1->ji_qs.ji_queue);
		(void)strcpy(pjob1->ji_qs.ji_queue, pjob2->ji_qs.ji_queue);
		(void)strcpy(pjob2->ji_qs.ji_queue, tmpqn);
		svr_dequejob(pjob1);
		svr_dequejob(pjob2);
		(void)svr_enquejob(pjob1);
		(void)svr_enquejob(pjob2);

	} else {
		swap_link(&pjob1->ji_jobque,  &pjob2->ji_jobque);
		swap_link(&pjob1->ji_alljobs, &pjob2->ji_alljobs);
	}

	/* need to update disk copy of both jobs to save new order */

	(void)job_save(pjob1, SAVEJOB_FULL);
	(void)job_save(pjob2, SAVEJOB_FULL);

	reply_ack(req);
}
