/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
/*
 * req_quejob.c 
 *
 * Functions relating to the Queue Job Batch Request sequence, including
 * Queue Job, Job Script, Ready to Commit, and Commit.
 *
 * Included funtions are:
 *	req_quejob()
 *	req_jobcredential()
 *	req_jobscript()
 *	req_rdycommit()
 *	req_commit()
 */
#include <pbs_config.h>   /* the master config generated by configure */

#include <sys/types.h>
#include <sys/param.h>
#include <errno.h>
#include <fcntl.h>
#include <netinet/in.h>
#include "libpbs.h"
#include "server_limits.h"
#include "list_link.h"
#include "attribute.h"
#include "resource.h"
#include "server.h"
#include "credential.h"
#include "batch_request.h"
#include "job.h"
#include "queue.h"
#include "net_connect.h"
#include "pbs_error.h"
#include "log.h"

#ifdef PBS_MOM
#include <pwd.h>
#include "mom_func.h"
#endif	/* PBS_MOM */

static char ident[] = "@(#) $RCSfile: req_quejob.c,v $ $Revision: 2.4 $";

/*
 * the following funny business is due to the fact that O_SYNC
 * is not currently POSIX
 */
#ifdef O_SYNC
#define O_Sync O_SYNC
#elif _FSYNC
#define O_Sync _FSYNC
#else
#define O_Sync 0
#endif

/* External Functions Called: */

extern int    reply_jid A_((char *jobid));
extern void   start_exec A_((job *));

/* Global Data Items: */

#ifndef PBS_MOM
extern char *path_spool;
extern struct server server;
extern char  server_name[];
extern int   queue_rank;
#endif	/* PBS_MOM */

extern int	 resc_access_perm;
extern list_head svr_alljobs;
extern list_head svr_newjobs;
extern attribute_def job_attr_def[];
extern char *path_jobs;
extern char *pbs_o_host;
extern char *msg_script_open;
extern char *msg_script_write;
extern char *msg_jobnew;
extern time_t time_now;

/* Private Functions in this file */

static job *locate_new_job A_((int sock, char *jobid));


static char *pbs_o_que = "PBS_O_QUEUE=";


/*
 * req_quejob - Queue Job Batch Request processing routine
 */

void req_quejob(preq)
	struct batch_request *preq;	/* ptr to the decoded request   */
{
	char		 basename[PBS_JOBBASE+1];
	char		 buf[256];
	int		 created_here = 0;
	int		 fds;
	int		 i;
	int		 index;
	char		*jid;
	char		 jidbuf[PBS_MAXSVRJOBID+1];
	char		 namebuf[MAXPATHLEN+1];
	char		*pc;
	attribute_def	*pdef;
	job		*pj;
	svrattrl	*psatl;
	pbs_queue	*pque;
	char		*qname;
	int		 rc;
	int		 sock = preq->rq_conn;
	attribute	 tempattr;

	/* set basic (user) level access permission */

	resc_access_perm = ATR_DFLAG_USWR | ATR_DFLAG_Creat;

#ifndef PBS_MOM		/* server server server server */
	/*
	 * if the job id is supplied, the request had better be 
	 * from another server
	 */

	if (preq->rq_fromsvr) {
		/* from another server - accept the extra attributes */
		resc_access_perm |= ATR_DFLAG_MGWR | ATR_DFLAG_SvWR;
		jid = preq->rq_ind.rq_queuejob.rq_jid;

	} else if (preq->rq_ind.rq_queuejob.rq_jid[0] != '\0') {
		/* a job id is not allowed from a client */
		req_reject(PBSE_IVALREQ, 0, preq);
		return;
	} else {
		/* assign it a job id */

		created_here = JOB_SVFLG_HERE;
		(void)sprintf(jidbuf, "%d.", server.sv_qs.sv_jobidnumber);
		(void)strcat(jidbuf, server_name);
		jid = jidbuf;
		
		/* having updated sv_jobidnumber, must save serve struct */

		if (++server.sv_qs.sv_jobidnumber > PBS_SEQNUMTOP)
			server.sv_qs.sv_jobidnumber = 0;	/* wrap it */

		if (svr_save(&server, SVR_SAVE_QUICK)) {
			req_reject(PBSE_INTERNAL, 0, preq);
			return;
		}
	}

#else		/* PBS_MOM mom mom mom mom mom mom*/

	if (preq->rq_fromsvr) {		/* must be from a server */
		/* from another server - accept the extra attributes */
		resc_access_perm |= ATR_DFLAG_MGWR | ATR_DFLAG_SvWR |
				    ATR_DFLAG_MOM;
		jid = preq->rq_ind.rq_queuejob.rq_jid;
	} else {
		req_reject(PBSE_IVALREQ, 0, preq);
		return;
	}
#endif		/* PBS_MOM all all all all all */

	/* does job already exist, check both old and new jobs */

	if ((pj = find_job(jid)) == (job *)0) {
		pj = (job *)GET_NEXT(svr_newjobs);
		while (pj) {
			if (!strcmp(pj->ji_qs.ji_jobid, jid))
				break;
			pj = (job *)GET_NEXT(pj->ji_alljobs);
		}
	}

#ifndef PBS_MOM		/* server server server server server server */
	if (pj != (job *)0) {
		/* server will reject queue request if job already exists */
		req_reject(PBSE_JOBEXIST, 0, preq);
		return;
	}


	/* find requested queue, is it there? */

	qname = preq->rq_ind.rq_queuejob.rq_destin;  
	if ((*qname == '\0') || (*qname == '@')) {  /* use default queue */
		pque = get_dfltque();
		rc   = PBSE_QUENODFLT;
	} else { 		/* else find the named queue */
		pque = find_queuebyname(preq->rq_ind.rq_queuejob.rq_destin);
		rc   = PBSE_UNKQUE;
	}
	if (pque == (pbs_queue *)0) {
		req_reject(rc, 0, preq);	   /* not there   */
		return;
	}

	/*
	 * make up job file name, it is based on the jobid, however the
	 * minimun file name is only 14 character in POSIX, so we may have
	 * to "hash" the name slightly
	 */
	(void)strncpy(basename, jid, PBS_JOBBASE);
	basename[PBS_JOBBASE] = '\0';

	do {
		(void)strcpy(namebuf, path_jobs);
		(void)strcat(namebuf, basename);
		(void)strcat(namebuf, JOB_FILE_SUFFIX);
		fds = open(namebuf, O_CREAT | O_EXCL | O_WRONLY, 0600);
		if (fds < 0) {
			if (errno == EEXIST) {
				pc = basename + strlen(basename) - 1;
				while ( ! isprint((int)*pc) ) {
					pc--;
					if (pc <= basename) {
					    req_reject(PBSE_INTERNAL, 0 ,preq);
					    return;
					}
				}
				(*pc)++;
			} else {
				req_reject(PBSE_SYSTEM, 0, preq);
				return;
			}
		}
	} while (fds < 0);
	(void)close(fds);

	/* create the job structure */

	if ((pj = job_alloc()) == (job *)0) {
		(void)unlink(namebuf);
		req_reject(PBSE_SYSTEM, 0, preq);
		return;
	}

#else                /* PBS_MOM mom mom mom mom*/
	/*
	 * New job ...
  	 *
	 * for MOM - rather than make up a hashname, we use the sent
	 * to us by the server as an attribute.
	 */
	psatl = (svrattrl *)GET_NEXT(preq->rq_ind.rq_queuejob.rq_attr);
	while (psatl) {
		if (!strcmp(psatl->al_name, ATTR_hashname)) {
			(void)strcpy(basename, psatl->al_value);
			break;
		}
		psatl = (svrattrl *)GET_NEXT(psatl->al_link);
	}

	if (pj) {
		if (pj->ji_qs.ji_substate == JOB_SUBSTATE_RUNNING) {
			req_reject(PBSE_JOBEXIST, 0, preq);
			return;
		}

		/* if checkpointed, then keep old and skip rest of process */

		if (pj->ji_qs.ji_svrflags & JOB_SVFLG_CHKPT) {
			pj->ji_qs.ji_substate = JOB_SUBSTATE_TRANSIN;
			if (reply_jobid(preq, pj->ji_qs.ji_jobid,
					BATCH_REPLY_CHOICE_Queue) == 0) {
				delete_link(&pj->ji_alljobs);
				append_link(&svr_newjobs, &pj->ji_alljobs, pj);
				pj->ji_qs.ji_un_type = JOB_UNION_TYPE_NEW;
				pj->ji_qs.ji_un.ji_newt.ji_fromsock = sock;
				pj->ji_qs.ji_un.ji_newt.ji_fromaddr =
							get_connectaddr(sock);
				pj->ji_qs.ji_un.ji_newt.ji_scriptsz = 0;
			} else {
				close_conn(sock);
			}
			return;
		}
		/* unlink job from svr_alljobs since will be place on newjobs */
		delete_link(&pj->ji_alljobs);
	} else {
		/* if not already here, allocate job struct */

		if ((pj = job_alloc()) == (job *)0) {
			req_reject(PBSE_SYSTEM, 0, preq);
			return;
		}
		(void)strcpy(namebuf, path_jobs);      /* job directory path */
		(void)strcat(namebuf, basename);
		(void)strcat(namebuf, JOB_TASKDIR_SUFFIX);
		if (mkdir(namebuf, 0700) == -1) {
			(void)job_purge(pj);
			req_reject(PBSE_SYSTEM, 0, preq);
			return;
		}
	}
#endif          /* PBS_MOM */

	(void)strcpy(pj->ji_qs.ji_jobid, jid);
	(void)strcpy(pj->ji_qs.ji_fileprefix, basename);
	pj->ji_modified = 1;
	pj->ji_qs.ji_svrflags = created_here;
	pj->ji_qs.ji_un_type  = JOB_UNION_TYPE_NEW;


	/* decode attributes from request into job structure */

	psatl = (svrattrl *)GET_NEXT(preq->rq_ind.rq_queuejob.rq_attr);
	while (psatl) {

		/* identify the attribute by name */

		index = find_attr(job_attr_def, psatl->al_name, JOB_ATR_LAST);
		if (index < 0) {

			/* didn`t recognize the name */
#ifndef PBS_MOM
			index = JOB_ATR_UNKN;	/* keep as "unknown" for now */
#else	/* is PBS_MOM */
			reply_badattr(PBSE_NOATTR, 1, psatl, preq);
			return;
#endif	/* PBS_MOM */
		}
		pdef = &job_attr_def[index];

		/* Is attribute not writeable by manager or by a server? */

		if ((pdef->at_flags & resc_access_perm) == 0) {
			(void)job_purge(pj);
			reply_badattr(PBSE_ATTRRO, 1, psatl, preq);
			return;
		}

		/* decode attribute */

		rc = pdef->at_decode(&pj->ji_wattr[index],
		     psatl->al_name, psatl->al_resc, psatl->al_value);
#ifndef PBS_MOM
		if (rc != 0) {
			if (rc == PBSE_UNKRESC) {

				/* unknown resources not allow in Exec queue */

				if (pque->qu_qs.qu_type == QTYPE_Execution) {
					(void)job_purge(pj);
					reply_badattr(rc, 1, psatl, preq);
					return;
				}
			} else {
				/* any other error is fatal */
				(void)job_purge(pj);
				reply_badattr(rc, 1, psatl, preq);
				return;
			}
		}
#else	/* PBS_MOM MOM MOM MOM */
		if (rc != 0) {
			/* all  errors are fatal for MOM */

			(void)job_purge(pj);
			reply_badattr(rc, 1, psatl, preq);
			return;
		}
		if (psatl->al_op == DFLT) {
		    if (psatl->al_resc) {

			resource	*presc;
			resource_def	*prdef;

			prdef = find_resc_def(svr_resc_def, psatl->al_resc,
					      svr_resc_size);
			if (prdef == 0) {
				(void)job_purge(pj);
				reply_badattr(rc, 1, psatl, preq);
				return;
			}
			presc = find_resc_entry(&pj->ji_wattr[index], prdef);
			if (presc)
				presc->rs_value.at_flags |= ATR_VFLAG_DEFLT;
		    } else {
			pj->ji_wattr[index].at_flags |= ATR_VFLAG_DEFLT;
		    }
		}
#endif	/* PBS_MOM */

		psatl = (svrattrl *)GET_NEXT(psatl->al_link);
	}

#ifndef PBS_MOM

	/* perform any at_action routine declared for the attributes */

	for (i=0; i<JOB_ATR_LAST; ++i) {
	    pdef = &job_attr_def[i];
	    if ( (pj->ji_wattr[i].at_flags & ATR_VFLAG_SET) &&
		 (pdef->at_action) ) {
		    rc = pdef->at_action(&pj->ji_wattr[i], pj, ATR_ACTION_NEW);
		    if (rc) {
			(void)job_purge(pj);
			req_reject(rc, i, preq);
			return;
		    }
	    }
	}
	
	/*
	 * Now that the attributes have been decoded, we can setup some
	 * additional parameters and perform a few more checks.
	 *
	 * First, set some items based on who created the job...
	 */

	if (created_here) {	/* created here */

		/* check value of priority */

	        if (pj->ji_wattr[(int)JOB_ATR_priority].at_flags & ATR_VFLAG_SET) {
		    	if ((pj->ji_wattr[(int)JOB_ATR_priority].at_val.at_long < -1024) || (pj->ji_wattr[(int)JOB_ATR_priority].at_val.at_long > 1024)) {

				(void)job_purge(pj);
				req_reject(PBSE_BADATVAL, 0, preq);
				return;
			}
		}

		/* set job owner attribute to user@host */

		job_attr_def[(int)JOB_ATR_job_owner].at_free(
				&pj->ji_wattr[(int)JOB_ATR_job_owner]);	
		(void)strcpy(buf, preq->rq_user);
		(void)strcat(buf, "@");
		(void)strcat(buf, preq->rq_host);
		job_attr_def[(int)JOB_ATR_job_owner].at_decode(
				&pj->ji_wattr[(int)JOB_ATR_job_owner],
				(char *)0, (char *)0, buf);

		/* set create time */

		pj->ji_wattr[(int)JOB_ATR_ctime].at_val.at_long =(long)time_now;
		pj->ji_wattr[(int)JOB_ATR_ctime].at_flags |= ATR_VFLAG_SET;

		/* set hop count = 1 */

		pj->ji_wattr[(int)JOB_ATR_hopcount].at_val.at_long = 1;
		pj->ji_wattr[(int)JOB_ATR_hopcount].at_flags |= ATR_VFLAG_SET;

		/* need to set certain environmental variables per POSIX */

		clear_attr(&tempattr, &job_attr_def[(int)JOB_ATR_variables]);
		(void)strcpy(buf, pbs_o_que);
		(void)strcat(buf, pque->qu_qs.qu_name);
		if (get_variable(pj, pbs_o_host) == (char *)0) {
		    (void)strcat(buf, ",");
		    (void)strcat(buf, pbs_o_host);
		    (void)strcat(buf, "=");
		    (void)strcat(buf, preq->rq_host);
		}
		job_attr_def[(int)JOB_ATR_variables].at_decode(&tempattr,
					(char *)0, (char *)0, buf);
		job_attr_def[(int)JOB_ATR_variables].at_set(
					&pj->ji_wattr[(int)JOB_ATR_variables],
					&tempattr, INCR);
		job_attr_def[(int)JOB_ATR_variables].at_free(&tempattr);

		/* if JOB_ATR_outpath/JOB_ATR_errpath not set, set default */

		if ( !(pj->ji_wattr[(int)JOB_ATR_outpath].at_flags & ATR_VFLAG_SET)) {
		    pj->ji_wattr[(int)JOB_ATR_outpath].at_val.at_str =
					prefix_std_file(pj, (int)'o');
		    pj->ji_wattr[(int)JOB_ATR_outpath].at_flags |=ATR_VFLAG_SET;
		}
		if ( !(pj->ji_wattr[(int)JOB_ATR_errpath].at_flags & ATR_VFLAG_SET)) {
		    pj->ji_wattr[(int)JOB_ATR_errpath].at_val.at_str =
					prefix_std_file(pj, (int)'e');
		    pj->ji_wattr[(int)JOB_ATR_errpath].at_flags |=ATR_VFLAG_SET;
		}

	} else {		/* job was created elsewhere and moved here */
		
		/* make sure job_owner is set, error if not */

		if ( !(pj->ji_wattr[(int)JOB_ATR_job_owner].at_flags & ATR_VFLAG_SET)) {
			(void)job_purge(pj);		
			req_reject(PBSE_IVALREQ, 0, preq);
			return;
		}

		/* increment hop count */

		if (++pj->ji_wattr[(int)JOB_ATR_hopcount].at_val.at_long >
		    PBS_MAX_HOPCOUNT) {
			(void)job_purge(pj);		
			req_reject(PBSE_HOPCOUNT, 0, preq);
			return;
		}

	}

	/* set up at_server attribute for status */

	job_attr_def[(int)JOB_ATR_at_server].at_decode(
			&pj->ji_wattr[(int)JOB_ATR_at_server],
			(char *)0, (char *)0, server_name);

	/*
	 * See if the job is qualified to go into the requested queue.
	 * Note, if an execution queue, then ji_qs.ji_un.ji_exect is set up
	 *
	 * svr_chkque is called way down here because it needs to have the
	 * job structure and attributes already set up.
	 */

	if (rc = svr_chkque(pj, pque, preq->rq_host, MOVE_TYPE_Move)) {
		(void)job_purge(pj);
		req_reject(rc, 0, preq);
		return;
	}
	(void)strcpy(pj->ji_qs.ji_queue, pque->qu_qs.qu_name);

	pj->ji_wattr[(int)JOB_ATR_substate].at_val.at_long = JOB_SUBSTATE_TRANSIN;
	pj->ji_wattr[(int)JOB_ATR_substate].at_flags |= ATR_VFLAG_SET;

#endif		/* PBS_MOM */

	/* set remaining job structure elements			*/

	pj->ji_qs.ji_state =    JOB_STATE_TRANSIT;
	pj->ji_qs.ji_substate = JOB_SUBSTATE_TRANSIN;

	pj->ji_wattr[(int)JOB_ATR_mtime].at_val.at_long = (long)time_now;
	pj->ji_wattr[(int)JOB_ATR_mtime].at_flags |= ATR_VFLAG_SET;

	pj->ji_qs.ji_un_type = JOB_UNION_TYPE_NEW;
	pj->ji_qs.ji_un.ji_newt.ji_fromsock = sock;
	pj->ji_qs.ji_un.ji_newt.ji_fromaddr = get_connectaddr(sock);
	pj->ji_qs.ji_un.ji_newt.ji_scriptsz = 0;

	/* acknowledge the request with the job id */

	if (reply_jobid(preq, pj->ji_qs.ji_jobid,BATCH_REPLY_CHOICE_Queue)==0) {

		/* link job into server's new jobs list request  */

		append_link(&svr_newjobs, &pj->ji_alljobs, pj);
	} else {
		/* reply failed, purge the job and close the connection */

		close_conn(sock);
		(void)job_purge(pj);
	}
}

/*
 * req_jobcredential - receive a set of credentials to be used by the job
 *
 * THIS IS JUST A PLACE HOLDER FOR NOW
 * It does nothing but acknowledge the request 
 */

void req_jobcredential(preq)
	struct batch_request *preq;	/* ptr to the decoded request   */
{
	job *pj;

	pj = locate_new_job(preq->rq_conn, (char *)0);
	if (pj == (job *)0) {
		req_reject(PBSE_IVALREQ, 0, preq);
		return;
	}
#ifndef PBS_MOM
	if (svr_authorize_jobreq(preq, pj) == -1) {
		req_reject(PBSE_PERM, 0, preq);
		return;
	}
#endif		/* PBS_MOM */
	reply_ack(preq);
}

/*
 * req_jobscript - receive job script section
 *
 * Each section is appended to the file
 */

void req_jobscript(preq)
	struct batch_request *preq;	/* ptr to the decoded request   */
{
	int	 fds;
	char	 namebuf[MAXPATHLEN];
	job	*pj;
#ifdef PBS_MOM
	int	 filemode = 0700;
#else	/* server */
	int	 filemode = 0600;
#endif

	pj = locate_new_job(preq->rq_conn, (char *)0);
	if (pj == (job *)0) {
		req_reject(PBSE_IVALREQ, 0, preq);
		return;
	}
	if (pj->ji_qs.ji_substate != JOB_SUBSTATE_TRANSIN) {
		req_reject(PBSE_IVALREQ, 0, preq);
		return;
	}
#ifndef PBS_MOM
	if (svr_authorize_jobreq(preq, pj) == -1) {
		req_reject(PBSE_PERM, 0, preq);
		return;
	}
#else
	/* mom - if job has been checkpointed, discard script,already have it */
	if (pj->ji_qs.ji_svrflags & JOB_SVFLG_CHKPT) {
		/* do nothing, ignore script */
		reply_ack(preq);
		return;
	}
#endif		/* PBS_MOM */

	(void)strcpy(namebuf, path_jobs);  
	(void)strcat(namebuf, pj->ji_qs.ji_fileprefix);
	(void)strcat(namebuf, JOB_SCRIPT_SUFFIX);

	if (pj->ji_qs.ji_un.ji_newt.ji_scriptsz == 0) {
		fds = open(namebuf, O_WRONLY|O_CREAT|O_EXCL|O_Sync, filemode);
	} else {
		fds = open(namebuf, O_WRONLY | O_APPEND | O_Sync, filemode);
	}
	if (fds < 0) {
		log_err(errno, "req_jobscript", msg_script_open);
		req_reject(PBSE_INTERNAL, 0, preq);
		return;
	}
	if (write(fds, preq->rq_ind.rq_jobfile.rq_data, (unsigned)preq->rq_ind.rq_jobfile.rq_size) != preq->rq_ind.rq_jobfile.rq_size) {
		log_err(errno, "req_jobscript", msg_script_write);
		req_reject(PBSE_SYSTEM, 0, preq);
		(void)close(fds);
		return;
	}
	(void)close(fds);
	pj->ji_qs.ji_un.ji_newt.ji_scriptsz += preq->rq_ind.rq_jobfile.rq_size;
	pj->ji_qs.ji_svrflags = (pj->ji_qs.ji_svrflags & ~JOB_SVFLG_CHKPT) |
				JOB_SVFLG_SCRIPT;      /* has a script file */
	reply_ack(preq);
}

#ifndef PBS_MOM	
/* the following is for the server only, MOM has her own version below */

/*
 * req_mvjobfile - receive a job file
 *	This request is used to move a file associated with a job, typically
 *	the standard output or error, between a server and a server or from
 *	a mom back to a server.  For a server, the destination is alway 
 *	within the spool directory.
 */

void req_mvjobfile(preq)
	struct batch_request *preq;	/* ptr to the decoded request   */
{
	int	 fds;
	char	 namebuf[MAXPATHLEN];
	job	*pj;

	pj = locate_new_job(preq->rq_conn, (char *)0);
	if (pj == (job *)0)
		pj = find_job(preq->rq_ind.rq_jobfile.rq_jobid);

	if ( (preq->rq_fromsvr == 0) || (pj == (job *)0) ) {
		req_reject(PBSE_IVALREQ, 0, preq);
		return;
	}


	(void)strcpy(namebuf, path_spool);  
	(void)strcat(namebuf, pj->ji_qs.ji_fileprefix);
	switch ((enum job_file)preq->rq_ind.rq_jobfile.rq_type) {
	    case StdOut:
		(void)strcat(namebuf, JOB_STDOUT_SUFFIX);
		break;

	    case StdErr:
		(void)strcat(namebuf, JOB_STDERR_SUFFIX);
		break;

	    case Chkpt:
		(void)strcat(namebuf, JOB_CKPT_SUFFIX);
		break;

	    default:
		req_reject(PBSE_IVALREQ, 0, preq);
		return;
	}

	if (preq->rq_ind.rq_jobfile.rq_sequence == 0)
		fds = open(namebuf, O_WRONLY|O_CREAT|O_TRUNC|O_Sync, 0600);
	else
		fds = open(namebuf, O_WRONLY|O_APPEND|O_Sync, 0600);
	if (fds < 0) {
		log_err(errno, "req_mvjobfile", msg_script_open);
		req_reject(PBSE_INTERNAL, 0, preq);
		return;
	}
	if (write(fds, preq->rq_ind.rq_jobfile.rq_data, (unsigned)preq->rq_ind.rq_jobfile.rq_size) != preq->rq_ind.rq_jobfile.rq_size) {
		log_err(errno, "req_jobfile", msg_script_write);
		req_reject(PBSE_SYSTEM, 0, preq);
		(void)close(fds);
		return;
	}
	(void)close(fds);
	reply_ack(preq);
}
#else	/* PBS_MOM - MOM MOM MOM */
/*
 * req_mvjobfile - move the specifled job standard files 
 *	This is MOM's version.  The files are owned by the user and placed
 *	in either the spool area or the user's home directory depending
 *	on the compile option, see std_file_name().
 */

void req_mvjobfile(preq)
	struct batch_request *preq;
{
	int	      fds;
	enum job_file jft;
	int	      oflag;
	job	     *pj;
	struct passwd *pwd;


	jft = (enum job_file)preq->rq_ind.rq_jobfile.rq_type;
	if (preq->rq_ind.rq_jobfile.rq_sequence == 0)
		oflag = O_CREAT | O_WRONLY | O_TRUNC;
	else
		oflag = O_CREAT | O_WRONLY | O_APPEND;

	pj = locate_new_job(preq->rq_conn, (char *)0);
	if (pj == (job *)0)
		pj = find_job(preq->rq_ind.rq_jobfile.rq_jobid);

	if (pj == (job *)0) {
		req_reject(PBSE_UNKJOBID, 0, preq);
		return;
	}
	if (((pwd = getpwnam(pj->ji_wattr[(int)JOB_ATR_euser].at_val.at_str)) == 0) || ((fds =open_std_file(pj, jft, oflag, pwd->pw_gid))<0)) {
		req_reject(PBSE_MOMREJECT, 0, preq);
		return;
	}

	if (write(fds, preq->rq_ind.rq_jobfile.rq_data, preq->rq_ind.rq_jobfile.rq_size) != preq->rq_ind.rq_jobfile.rq_size)
		req_reject(PBSE_SYSTEM, 0, preq);
	else
		reply_ack(preq);	
	(void)close(fds);
}
#endif /* PBS_MOM */


/*
 * req_rdytocommit - Ready To Commit Batch Request
 *
 *	Set substate to JOB_SUBSTATE_TRANSICM and
 *	record job to permanent storage, i.e. writen to the job save file
 */

void req_rdytocommit(preq)
	struct batch_request *preq;
{
	job *pj;
	int  sock = preq->rq_conn;

	pj = locate_new_job(sock, preq->rq_ind.rq_rdytocommit);
	if (pj == (job *)0) {
		req_reject(PBSE_UNKJOBID, 0, preq);
		return;
	}
	if (pj->ji_qs.ji_substate != JOB_SUBSTATE_TRANSIN) {
		req_reject(PBSE_IVALREQ, 0, preq);
		return;
	}
#ifndef PBS_MOM
	if (svr_authorize_jobreq(preq, pj) == -1) {
		req_reject(PBSE_PERM, 0, preq);
		return;
	}
#endif		/* PBS_MOM */

	pj->ji_qs.ji_state    = JOB_STATE_TRANSIT;
	pj->ji_qs.ji_substate = JOB_SUBSTATE_TRANSICM;
	pj->ji_wattr[(int)JOB_ATR_state].at_val.at_char = 'T';
	pj->ji_wattr[(int)JOB_ATR_state].at_flags |= ATR_VFLAG_SET;

	if (job_save(pj, SAVEJOB_NEW) == -1) {
		req_reject(PBSE_SYSTEM, 0, preq);
		return;
	}
	/* acknowledge the request with the job id */

	if (reply_jobid(preq, pj->ji_qs.ji_jobid, BATCH_REPLY_CHOICE_RdytoCom) != 0) {

		/* reply failed, purge the job and close the connection */

		close_conn(sock);
		(void)job_purge(pj);
	}
}

/*
 * req_commit - commit ownership of job
 *
 *	Set state of job to JOB_STATE_QUEUED (or Held or Waiting) and
 *	enqueue the job into its destination queue.
 */

void req_commit(preq)
	struct batch_request *preq;
{
	int	   newstate;
	int	   newsub;
	job	  *pj;
	pbs_queue *pque;
	int	   rc;

	pj = locate_new_job(preq->rq_conn, preq->rq_ind.rq_commit);
	if (pj == (job *)0) {
		req_reject(PBSE_UNKJOBID, 0, preq);
		return;
	}
	if (pj->ji_qs.ji_substate != JOB_SUBSTATE_TRANSICM) {
		req_reject(PBSE_IVALREQ, 0, preq);
		return;
	}

#ifdef PBS_MOM	/* MOM only */

	/* move job from new job list to "all" job list, set to running state */

	delete_link(&pj->ji_alljobs);
	append_link(&svr_alljobs, &pj->ji_alljobs, pj);
	/*
	** Set JOB_SVFLG_HERE to indicate that this is Mother Superior.
	*/
	pj->ji_qs.ji_svrflags |= JOB_SVFLG_HERE;

	pj->ji_qs.ji_state = JOB_STATE_RUNNING;
	pj->ji_qs.ji_substate = JOB_SUBSTATE_PRERUN;
	pj->ji_qs.ji_un_type = JOB_UNION_TYPE_MOM;
	pj->ji_qs.ji_un.ji_momt.ji_svraddr = get_connectaddr(preq->rq_conn);
	pj->ji_qs.ji_un.ji_momt.ji_exitstat = 0;

	/* For MOM - start up the job */

	start_exec(pj);
	(void)reply_jobid(preq, pj->ji_qs.ji_jobid, BATCH_REPLY_CHOICE_Commit);
	(void)job_save(pj, SAVEJOB_FULL);
	/* set modify for returning these to server once, see stat_job.c */
	pj->ji_wattr[(int)JOB_ATR_errpath].at_flags |= ATR_VFLAG_MODIFY;
	pj->ji_wattr[(int)JOB_ATR_outpath].at_flags |= ATR_VFLAG_MODIFY;
	pj->ji_wattr[(int)JOB_ATR_session_id].at_flags |= ATR_VFLAG_MODIFY;
	pj->ji_wattr[(int)JOB_ATR_altid].at_flags |= ATR_VFLAG_MODIFY;

#else	/* PBS_SERVER */
	if (svr_authorize_jobreq(preq, pj) == -1) {
		req_reject(PBSE_PERM, 0, preq);
		return;
	}

	/* remove job for the server new job list, set state, and enqueue it */

	delete_link(&pj->ji_alljobs);

	svr_evaljobstate(pj, &newstate, &newsub, 1);
	(void)svr_setjobstate(pj, newstate, newsub);

	/* set the queue rank attribute */

	pj->ji_wattr[(int)JOB_ATR_qrank].at_val.at_long = ++queue_rank;
	pj->ji_wattr[(int)JOB_ATR_qrank].at_flags |= ATR_VFLAG_SET;

	if (rc = svr_enquejob(pj)) {
		(void)job_purge(pj);
		req_reject(rc, 0, preq);
		return;
	}

	if (job_save(pj, SAVEJOB_FULL)) {
		(void)job_purge(pj);
		req_reject(PBSE_SYSTEM, 0, preq);
		return;
	}

	/*
	 * if the job went into a Route (push) queue that has been started,
	 * try once to route it to give immediate feedback as a courtsey
	 * to the user.
	 */

	pque = pj->ji_qhdr;
	if ((preq->rq_fromsvr == 0) &&
            (pque->qu_qs.qu_type == QTYPE_RoutePush) &&
	    (pque->qu_attr[(int)QA_ATR_Started].at_val.at_long != 0)) {
		if (rc = job_route(pj)) {
			(void)job_purge(pj);
			req_reject(rc, 0, preq);
			return;
		}
	}

	/* need to format message first, before request goes away */

	(void)sprintf(log_buffer, msg_jobnew, 
		      preq->rq_user, preq->rq_host,
		      pj->ji_wattr[(int)JOB_ATR_job_owner].at_val.at_str,
		      pj->ji_wattr[(int)JOB_ATR_jobname].at_val.at_str,
		      pj->ji_qhdr->qu_qs.qu_name);

	/* acknowledge the request with the job id */

	(void)reply_jobid(preq, pj->ji_qs.ji_jobid, BATCH_REPLY_CHOICE_Commit);
	LOG_EVENT(PBSEVENT_JOB, PBS_EVENTCLASS_JOB, pj->ji_qs.ji_jobid,
		  log_buffer);

	if ((pj->ji_qs.ji_svrflags & JOB_SVFLG_HERE) == 0)
		issue_track(pj);	/* notify creator where job is */
#endif		/* PBS_SERVER */
}

/*
 * locate_new_job - locate a "new" job which has been set up req_quejob on
 *	the servers new job list.
 *	
 *	This function is used by the sub-requests which make up the global
 *	"Queue Job Request" to locate the job structure.
 *
 *	If the jobid is specified (will be for rdytocommit and commit, but not
 *	for script), we search for a matching jobid.  
 *	
 *	The job must (also) match the socket specified and the host associated
 *	with the socket unless ji_fromsock == -1, then its a recovery situation.
 */

static job *locate_new_job(sock, jobid)
	int    sock;
	char *jobid;
{
	job *pj;

	pj = (job *)GET_NEXT(svr_newjobs);
	while (pj) {

	    if ((pj->ji_qs.ji_un.ji_newt.ji_fromsock == -1) ||
		((pj->ji_qs.ji_un.ji_newt.ji_fromsock == sock) &&
		(pj->ji_qs.ji_un.ji_newt.ji_fromaddr==get_connectaddr(sock)))) {

	    	if (jobid != (char *)0) {
		    if ( !strncmp(pj->ji_qs.ji_jobid, jobid, PBS_MAXSVRJOBID))
	                break;
		} else
			break;
	    }

	    pj = (job *)GET_NEXT(pj->ji_alljobs);
	}
	return (pj);
}
