/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
/*
 * req_track.c
 *
 * Functions relation to the Track Job Request and job tracking.
 */
#include <pbs_config.h>   /* the master config generated by configure */

#include <errno.h>
#include <sys/types.h>
#include "libpbs.h"
#include <fcntl.h>
#include <signal.h>
#include "server_limits.h"
#include "list_link.h"
#include "attribute.h"
#include "server.h"
#include "credential.h"
#include "batch_request.h"
#include "job.h"
#include "pbs_error.h"
#include "work_task.h"
#include "tracking.h"
#include "svrfunc.h"

static char ident[] = "@(#) $RCSfile: req_track.c,v $ $Revision: 2.1 $";

/* External functions */

extern int issue_to_svr A_((char *svr, struct batch_request *, void (*func)(struct work_task *)));

/* Global Data Items: */

extern char         *path_track;
extern struct server server;
extern time_t        time_now;
extern char 	     server_name[];

/*
 * req_track - record job tracking information
 */

void req_track(preq)
	struct batch_request	*preq;
{
	struct tracking *empty = (struct tracking *)0;
	int		 i;
	int		 need;
	struct tracking	*new;
	struct tracking *ptk;
	struct rq_track *prqt;

	/*  make sure request is from a server */

	if ( !preq->rq_fromsvr ) {
		req_reject(PBSE_IVALREQ, 0, preq);
		return;
	}

	/* attempt to locate tracking record for this job    */
	/* also remember first empty slot in case its needed */

	prqt = &preq->rq_ind.rq_track;

	ptk = server.sv_track;
	for (i=0; i<server.sv_tracksize; i++) {
	    if ( (ptk+i)->tk_mtime ) {
		if ( !strcmp((ptk+i)->tk_jobid, prqt->rq_jid) ) {

		    /*
		     * found record, discard it if state == exiting,
		     * otherwise, update it if older
		     */

		    if ( *prqt->rq_state == 'E') {
			(ptk+i)->tk_mtime = 0;
		    } else if ( (ptk+i)->tk_hopcount < prqt->rq_hopcount) {
			(ptk+i)->tk_hopcount = prqt->rq_hopcount;
			(void)strcpy((ptk+i)->tk_location, prqt->rq_location);
			(ptk+i)->tk_state = *prqt->rq_state;
			(ptk+i)->tk_mtime = time_now;
		    }
		    server.sv_trackmodifed = 1;
		    reply_ack(preq);
		    return;
		}
	    } else if (empty == (struct tracking *)0) {
		empty = ptk + i;
	    }
	}

	/* if we got here, didn't find it... */

	if (*prqt->rq_state != 'E') {

		 /* and need to add it */

		if (empty == (struct tracking *)0) {

	    		/* need to make room for more */

	    		need = server.sv_tracksize * 3 / 2;
	    		new  = (struct tracking *)realloc(server.sv_track,
					       need * sizeof (struct tracking));
	    		if (new == (struct tracking *)0) {
				log_err(errno, "req_track", "malloc failed");
				req_reject(PBSE_SYSTEM, 0, preq);
				return;
	    		}
			empty = new + server.sv_tracksize; /* first new slot */
			for (i = server.sv_tracksize; i < need; i++)
				(new + i)->tk_mtime = 0;
			server.sv_tracksize = need;
			server.sv_track     = new;
		}

		empty->tk_mtime = time_now;
		empty->tk_hopcount = prqt->rq_hopcount;
		(void)strcpy(empty->tk_jobid, prqt->rq_jid);
		(void)strcpy(empty->tk_location, prqt->rq_location);
		empty->tk_state = *prqt->rq_state;
		server.sv_trackmodifed = 1;
	}
	reply_ack(preq);
	return;
}

/*
 * track_save - save the tracking records to a file
 *
 *	This routine is invoked periodically by a timed work task entry.
 *	The first entry is created at server initialization time and then
 *	recreated on each entry.
 *
 *	On server shutdown, track_save is called with a null work task pointer.
 */

void track_save(pwt)
	struct work_task *pwt;	/* unused */
{
	int fd;
	char *myid = "save_track";

	/* set task for next round trip */

	if (pwt) { 	/* set up another work task for next time period */
		if ( !set_task(WORK_Timed, (long)time_now + PBS_SAVE_TRACK_TM,
			       track_save,0))
			log_err(errno, myid, "Unable to set task for save");
	}
		
	if (server.sv_trackmodifed == 0)
		return;			/* nothing to do this time */

	fd = open(path_track, O_WRONLY, 0);
	if (fd < 0) {
		log_err(errno, myid, "Unable to open tracking file");
		return;
	}
	(void)write(fd, (char *)server.sv_track, server.sv_tracksize * sizeof(struct tracking));
	(void)close(fd);
	server.sv_trackmodifed = 0;
	return;
}

/*
 * issue_track - issue a Track Job Request to another server
 */

void issue_track(pjob)
	job *pjob;
{
	struct batch_request   *preq;
	char		       *pc;

	preq = alloc_br(PBS_BATCH_TrackJob);
	if (preq == (struct batch_request *)0)
		return;

	preq->rq_ind.rq_track.rq_hopcount = pjob->ji_wattr[(int)JOB_ATR_hopcount].at_val.at_long;
	(void)strcpy(preq->rq_ind.rq_track.rq_jid, pjob->ji_qs.ji_jobid);
	(void)strcpy(preq->rq_ind.rq_track.rq_location, server_name);
	preq->rq_ind.rq_track.rq_state[0] = pjob->ji_wattr[(int)JOB_ATR_state].at_val.at_char;
	
	pc = pjob->ji_qs.ji_jobid;
	while (*pc != '.')
		pc++;
	(void)issue_to_svr(++pc, preq, release_req);
}
