/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
#include <pbs_config.h>   /* the master config generated by configure */

/*
 * stat_job.c 
 *
 * Functions which support the Status Job Batch Request.
 *
 * Included funtions are:
 *	status_jobs()
 *	status_attrib()
 */
#include <sys/types.h>
#include "libpbs.h"
#include <ctype.h>
#include "server_limits.h"
#include "list_link.h"
#include "attribute.h"
#include "server.h"
#include "queue.h"
#include "credential.h"
#include "batch_request.h"
#include "job.h"
#include "work_task.h"
#include "pbs_error.h"
#include "svrfunc.h"

static char ident[] = "@(#) $RCSfile: stat_job.c,v $ $Revision: 2.2 $";

/* Global Data Items: */

extern attribute_def job_attr_def[];
extern int	     resc_access_perm; /* see encode_resc() in attr_fn_resc.c */
extern struct server server;

/*
 * status_job - Build the status reply for a single job.
 */

int status_job(pjob, preq, pal, pstathd, bad)
	job	  *pjob;	/* ptr to job to status */
	struct batch_request *preq;
	svrattrl  *pal;		/* specific attributes to status */
	list_head *pstathd;	/* RETURN: head of list to append status to */
	int       *bad;		/* RETURN: index of first bad attribute */
{
	struct brp_status *pstat;

	/* see if the client is authorized to status this job */

	if ( ! server.sv_attr[(int)SRV_ATR_query_others].at_val.at_long)
		if (svr_authorize_jobreq(preq, pjob))
			return (PBSE_PERM);


	/* allocate reply structure and fill in header portion */

	pstat = (struct brp_status *)malloc(sizeof (struct brp_status));
	if (pstat == (struct brp_status *)0)
		return (PBSE_SYSTEM);
	CLEAR_LINK(pstat->brp_stlink);
	pstat->brp_objtype = MGR_OBJ_JOB;
	(void)strcpy(pstat->brp_objname, pjob->ji_qs.ji_jobid);
	CLEAR_HEAD(pstat->brp_attr);
	append_link(pstathd, &pstat->brp_stlink, pstat);

	/* add attributes to the status reply */

	*bad = 0;
	if (status_attrib(pal, job_attr_def, pjob->ji_wattr, JOB_ATR_LAST,
		          preq->rq_perm, &pstat->brp_attr, bad) )
		return (PBSE_NOATTR);

	return (0);
}

/*
 * status_attrib - add each requested or all attributes to the status reply 
 *
 *	Returns: 0 on success
 *		-1 on error (bad attribute), "bad" set to ordinal of attribute
 */

int status_attrib(pal, padef, pattr, limit, priv, phead, bad)
	svrattrl	*pal;
	attribute_def   *padef;
	attribute	*pattr;
	int		 limit;
	int		 priv;
	list_head	*phead;
	int		*bad;
{
	int   index;
	int   nth = 0;
	
	priv &= ATR_DFLAG_RDACC;  /* user-client privilege		*/
	resc_access_perm = priv;  /* pass privilege to encode_resc()	*/

	/* for each attribute asked for or for all attributes, add to reply */

	if (pal) {		/* client specified certain attributes */
	    while (pal) {
		++nth;
		index = find_attr(padef, pal->al_name, limit);
		if (index < 0) {
			*bad = nth;
			return (-1);
		}
		if ((padef+index)->at_flags & priv) {
			(void)(padef+index)->at_encode(pattr+index, phead,
					(padef+index)->at_name,
					(char *)0, ATR_ENCODE_CLIENT);
		}
		pal = (svrattrl *)GET_NEXT(pal->al_link);
	    }
		
	} else {	/* non specified, return all readable attributes */

		for (index = 0; index < limit; index++) {
		    if (((padef+index)->at_flags & priv) &&
			!((padef+index)->at_flags & ATR_DFLAG_NOSTAT)) {
			(void)(padef+index)->at_encode(pattr+index, phead,
					(padef+index)->at_name,
					(char *)0, ATR_ENCODE_CLIENT);
		    }
		}
	}
	return (0);
}
