/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
/*
 * svr_connect.c - contains routines to tie the structures used by
 *	net_client and net_server together with those used by the
 *	various PBS_*() routines in the API.
 *
 *	svr_connect() opens a connection with can be used with the
 *		API routines and still be selected in wait_request().
 *
 *		Returns a connection handle ( >= 0 ).  Note that a value
 *		of PBS_LOCAL_CONNECTION is special, it means the server
 *		is talking to itself.
 *	
 *		It is called by the server whenever we need to send
 *		a request to another server, or talk to MOM.
 *
 *		On an error, PBS_NET_RC_FATAL (-1) is retuned if the
 *				error is believed to be permanent,
 *			     PBS_NET_RC_RETRY (-2) if the error is
 *				believed to be temporary, ie retry.
 *
 *	svr_disconnect() closes the above connection.
 */

#include <pbs_config.h>   /* the master config generated by configure */

#include <sys/types.h>
#include "libpbs.h"
#include "server_limits.h"
#include "net_connect.h"
#include "svrfunc.h"

static char ident[] = "@(#) $RCSfile: svr_connect.c,v $ $Revision: 2.3 $";

/* global data */

struct connect_handle connection[PBS_NET_MAX_CONNECTIONS]; /* used by API */

extern int		 errno;

extern int		 pbs_errno;
extern unsigned int	 pbs_server_port_dis;
extern struct connection svr_conn[];
extern pbs_net_t	 pbs_server_addr;

int svr_connect(hostaddr, port, func, cntype)
	pbs_net_t    hostaddr;			/* host order */
	unsigned int port;			/* host order */
	void	  (*func) A_((int));
	enum conn_type cntype;
{
	int handle;
	int sock;

	/* First, determine if the request is to another server or ourselves */

	if ((hostaddr == pbs_server_addr) && (port == pbs_server_port_dis))
		return (PBS_LOCAL_CONNECTION);	/* special value for local */

	/* obtain the connection to the other server */

	sock = client_to_svr(hostaddr, port, 1);
	if (sock < 0) {
		pbs_errno = errno;
		return (sock);	/* PBS_NET_RC_RETRY or PBS_NET_RC_FATAL */
	}

	/* add the connection to the server connection table and select list */

	if (func) {
		if (cntype == ToServerASN)
			abort();	/* ASN kludge */	
		else
			add_conn(sock, ToServerDIS, hostaddr, port, func);
	}
	svr_conn[sock].cn_authen = PBS_NET_CONN_AUTHENTICATED;

	/* find a connect_handle entry we can use and pass to the PBS_*() */

	handle = socket_to_handle(sock);
	if (handle == -1) {
		close_conn(sock);
		return (PBS_NET_RC_RETRY);
	}

	return (handle);
}

/*
 * svr_disconnect - close a connection made with svr_connect()
 *
 *	In addition to closing the actual connection, both the
 *	server's connection table and the handle table used by
 *	the API routines must be cleaned-up.
 */

void svr_disconnect(handle)
	int handle;
{
	if ((handle >= 0) && (handle < PBS_LOCAL_CONNECTION)) {
		shutdown(connection[handle].ch_socket, 2);
		close_conn(connection[handle].ch_socket);
		if (connection[handle].ch_errtxt != (char *)0) {
			free(connection[handle].ch_errtxt);
			connection[handle].ch_errtxt = (char *)0;
		}
		connection[handle].ch_errno = 0;
		connection[handle].ch_inuse = 0;
	}
}

/*
 * socket_to_handle() - turn a socket into a connection handle
 *	as used by the libpbs.a routines.
 *
 *	Returns: >=0 connection handle if successful, or
 *		  -1 if error, error number set in pbs_errno.
 */

int socket_to_handle(sock)
	int sock;	/* opened socket */
{
	int	i;

	for (i=0; i<PBS_NET_MAX_CONNECTIONS; i++) {
		if (connection[i].ch_inuse == 0) {

			connection[i].ch_stream = 0;
			connection[i].ch_inuse = 1;
			connection[i].ch_errno = 0;
			connection[i].ch_socket= sock;
			connection[i].ch_errtxt = 0;

			/* save handle for later close */

			svr_conn[sock].cn_handle = i;
			return (i);
		}
	}
	pbs_errno = PBSE_NOCONNECTS;
	return (-1);
}

/*
 * parse_servername - parse a server/mom name in the form:
 *	hostname[:service_port][/#][+hostname...]
 *
 *	Returns ptr to the host name as the function value and the service_port
 *	number (int) into service if :port is found, otherwise port is unchanged
 *	host name is also terminated by a '+' or '/' in string
 *
 *	Warning: as written, Not reentrient/thread safe
 */

char *parse_servername(name, service)
	char  		*name;		/* server name in form name[:port] */
	unsigned int	*service;	/* RETURN: service_port if :port */
{
	static char  buf[PBS_MAXSERVERNAME + PBS_MAXPORTNUM + 2];
	int   i = 0;
	char *pc;

	/* look for a ':', '+' or '/' in the string */
	
	pc = name;
	while (*pc && (i < PBS_MAXSERVERNAME+PBS_MAXPORTNUM+2)) {
		if ((*pc == '+') || (*pc == '/')) {
			break;
		} else if (*pc == ':') {
			*service = (unsigned int)atoi(pc + 1);
			break;
		} else {
			buf[i++] = *pc++;
		}
	}
	buf[i] = '\0';
	return (buf);
}
