/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
/*
 * svr_func.c - miscellaneous server functions
 */
#include <pbs_config.h>   /* the master config generated by configure */

#include "portability.h"
#include <assert.h>
#include <sys/types.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "server_limits.h"
#include "list_link.h"
#include "attribute.h"
#include "resource.h"
#include "job.h"
#include "queue.h"
#include "server.h"
#include "pbs_error.h"
#include "svrfunc.h"
#include "sched_cmds.h"

static char ident[] = "@(#) $RCSfile: svr_func.c,v $ $Revision: 2.1 $";

extern int    scheduler_sock;
extern int    svr_do_schedule;
extern time_t time_now;

/*
 * the following array of strings is used in decoding/encoding the server state
 */
static char *svr_idle   = "Idle";
static char *svr_sched  = "Scheduling";
static char *svr_state_names[] = {
	"",			/* SV_STATE_DOWN */
	"",			/* SV_STATE_INIT */
	"Hot_Start",		/* SV_STATE_HOT  */
	"Active",		/* SV_STATE_RUN  */
	"Terminating_Delay",	/* SV_STATE_SHUTDEL */
	"Terminating",		/* SV_STATE_SHUTIMM */
	"Terminating"		/* SV_STATE_SHUTSIG */
};

/*
 * encode_svrstate - encode the current server state from the internal
 *	integer to a state name string.
 */

int encode_svrstate(pattr, phead, atname, rsname, mode)
	attribute       *pattr;    /* ptr to attribute */
	list_head       *phead;   /* head of attrlist list */
	char            *atname;  /* attribute name */
	char            *rsname;  /* null */
	int             mode;     /* encode mode */
{
	svrattrl *pal;
	char *psname;

	if (!pattr)
		return (-1);
	if ( (mode == ATR_ENCODE_SAVE)			||
	     (pattr->at_val.at_long <= SV_STATE_DOWN)	||
	     (pattr->at_val.at_long > SV_STATE_SHUTSIG) )
		return (0);		/* don't bother to encode it */

	
	psname = svr_state_names[pattr->at_val.at_long];
	if (pattr->at_val.at_long == SV_STATE_RUN) {
		if (server.sv_attr[(int)SRV_ATR_scheduling].at_val.at_long == 0)
			psname = svr_idle;
		else if (scheduler_sock != -1)
		psname = svr_sched;
	}

	pal = attrlist_create(atname, rsname, strlen(psname) + 1);
	if (pal == (svrattrl *)0)
		return (-1);
	(void)strcpy(pal->al_value, psname);
	pal->al_flags = pattr->at_flags;
	append_link(phead, &pal->al_link, pal);
	return (1);
}

/*
 * set_resc_assigned - set the resources used by a job in the server and
 *	queue resources_used attribute
 */
void set_resc_assigned(pjob, op)
	job	*pjob;
	enum batch_op op;
{
	resource     *jobrsc;
	resource     *pr;
	attribute    *queru;
	resource_def *rscdef;
	attribute    *sysru;

	assert(pjob->ji_qhdr->qu_qs.qu_type == QTYPE_Execution);

	if (pjob->ji_qs.ji_state != JOB_STATE_RUNNING)
		return;
	sysru = &server.sv_attr[(int)SRV_ATR_resource_assn];
	queru = &pjob->ji_qhdr->qu_attr[(int)QE_ATR_ResourceAssn];
	jobrsc = (resource *)GET_NEXT(pjob->ji_wattr[(int)JOB_ATR_resource].at_val.at_list);

	while (jobrsc) {
		rscdef = jobrsc->rs_defin;

		/* if resource usage is to be tracked */

		if ( (rscdef->rs_flags & ATR_DFLAG_RASSN) && 
		     (jobrsc->rs_value.at_flags & ATR_VFLAG_SET) ) {

			/* update system attribute of resources assigned */

			pr = find_resc_entry(sysru, rscdef);
			if (pr == (resource *)0) {
				pr = add_resource_entry(sysru, rscdef);
				if (pr == (resource *)0)
					return;
			}
			rscdef->rs_set(&pr->rs_value, &jobrsc->rs_value, op);

			/* update queue attribute of resources assigned */

			pr = find_resc_entry(queru, rscdef);
			if (pr == (resource *)0) {
				pr = add_resource_entry(queru, rscdef);
				if (pr == (resource *)0)
					return;
			}
			rscdef->rs_set(&pr->rs_value, &jobrsc->rs_value, op);
		}
		jobrsc = (resource *)GET_NEXT(jobrsc->rs_link);
	}
}


/*
 * ck_chkpnt - check validity of job checkpoint attribute value
 */
int ck_chkpnt(pattr, pobject, mode)
	attribute *pattr;
	void	  *pobject;	/* not used */
	int	   mode;	/* not used */
{
	char *val;

	val = pattr->at_val.at_str;
	if (val == (char *)0)
		return (0);

	if ((*val == 'n') || (*val == 's') || (*val == 'u')) {
		if (*(val+1) != '\0')
			return (PBSE_BADATVAL);
	}
	else if (*val++ == 'c') {
		if (*val != '\0') {
			if (*val++ != '=')
				return (PBSE_BADATVAL);
			if (atoi(val) <= 0)
				return (PBSE_BADATVAL);
		}
	} else
		return (PBSE_BADATVAL);

	return (0);
}

/*
 * decode_null - Null attribute decode routine for Read Only (server
 *	and queue ) attributes.  It just returns 0.
 */

int decode_null(patr, name, rn, val)
	attribute *patr;
	char      *name;
	char      *rn;
	char      *val;
{
	return 0;
}

/*
 * set_null - Null set routine for Read Only attributes. 
 */

int set_null(pattr, new, op)
	attribute *pattr;
	attribute *new;
	enum batch_op op;
{
	return 0;
}

/*
 * poke_scheduler - action routine for the server's "scheduling" attribute.
 *	Call the scheduler whenever the attribute is set (or reset) to true.
 */

int poke_scheduler(pattr, pobj, actmode)
	attribute *pattr;
	void      *pobj;
	int	   actmode;
{
	if (actmode == ATR_ACTION_ALTER) {
		if (pattr->at_val.at_long)
			svr_do_schedule = SCH_SCHEDULE_CMD;
	}
	return 0;
}
