/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
/*
 * svr_mail.c - send mail to mail list or owner of job on
 *	job begin, job end, and/or job abort
 */

#include <pbs_config.h>   /* the master config generated by configure */

#include "pbs_ifl.h"
#include <sys/types.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "list_link.h"
#include "attribute.h"
#include "server_limits.h"
#include "job.h"
#include "server.h"

static char ident[] = "@(#) $RCSfile: svr_mail.c,v $ $Revision: 2.1 $";

/* External Functions Called */

extern void net_close A_((int));

/* Globol Data */

extern struct server server;
extern char *msg_job_abort;
extern char *msg_job_start;
extern char *msg_job_end;
extern char *msg_job_stageinfail;


void svr_mailowner(pjob, mailpoint, force, text)
	job	*pjob;
	int 	 mailpoint;	/* note, single character  */
	int	 force;		/* if set, force mail delivery */
	char	*text;		/* additional message text */
{
	char	 *cmdbuf;
	int	 i;
	char	*mailfrom;
	char	 mailto[512];
	FILE	*outmail;
	struct array_strings *pas;
	char	*stdmessage = (char *)0;


	/* if force is true, force the mail out regardless of mailpoint */

	if (force != MAIL_FORCE) {

	    /* see if user specified mail of this type */

	    if (pjob->ji_wattr[(int)JOB_ATR_mailpnts].at_flags &ATR_VFLAG_SET) {
		if (strchr(pjob->ji_wattr[(int)JOB_ATR_mailpnts].at_val.at_str,
		    mailpoint) == (char *)0)
			return;
	    } else if (mailpoint != MAIL_ABORT)	/* not set, default to abort */
			return;
	}

	/*
	 * ok, now we will fork a process to do the mailing to not
 	 * hold up the server's other work.
	 */


	if (fork()) 
		return;		/* its all up to the child now */
	
/*
 * From here on, we are a child process of the server.
 * Fix up file descriptors and signal handlers.
 */

	rpp_terminate();
	net_close(-1);

	/* Who is mail from, if SVR_ATR_mailfrom not set use default */

	if ((mailfrom = server.sv_attr[(int)SRV_ATR_mailfrom].at_val.at_str)==0)
		mailfrom = PBS_DEFAULT_MAIL;

	/* Who does the mail go to?  If mail-list, them; else owner */

	*mailto = '\0';
	if (pjob->ji_wattr[(int)JOB_ATR_mailuser].at_flags & ATR_VFLAG_SET) {

		/* has mail user list, send to them rather than owner */

		pas = pjob->ji_wattr[(int)JOB_ATR_mailuser].at_val.at_arst;
		if (pas != (struct array_strings *)0)	{
		    for (i = 0; i < pas->as_usedptr; i++ ) {
		    	if ((strlen(mailto) + strlen(pas->as_string[i]) + 2) <
			    sizeof(mailto) ) {
		    		(void)strcat(mailto, pas->as_string[i]);
		    		(void)strcat(mailto, " ");
			}		
		    }
		}

	} else {

		/* no mail user list, just send to owner */

		(void)strcpy(mailto, pjob->ji_wattr[(int)JOB_ATR_job_owner].at_val.at_str);
	}

	/* setup sendmail command line with -f from_whom */

	i = strlen(SENDMAIL_CMD) + strlen(mailfrom) + strlen(mailto) + 6;
	if ((cmdbuf = malloc(i)) == 0)
		exit(1);

	(void)sprintf(cmdbuf, "%s -f %s %s", SENDMAIL_CMD, mailfrom, mailto);
		
	outmail = (FILE *) popen(cmdbuf, "w");
	if (outmail == (FILE *)0)
		exit(1);
	
	/* Pipe in mail headers: To: and Subject: */

	fprintf(outmail, "To: %s\n", mailto);
	fprintf(outmail, "Subject: PBS JOB %s\n\n", pjob->ji_qs.ji_jobid);

	/* Now pipe in "standard" message */

	switch (mailpoint) {

		case MAIL_ABORT:
			stdmessage = msg_job_abort;
			break;

		case MAIL_BEGIN:
			stdmessage = msg_job_start;
			break;

		case MAIL_END:
			stdmessage = msg_job_end;
			break;

		case MAIL_STAGEIN:
			stdmessage = msg_job_stageinfail;
			break;
	}

	fprintf(outmail, "PBS Job Id: %s\n", pjob->ji_qs.ji_jobid);
	fprintf(outmail, "Job Name:   %s\n",
		          pjob->ji_wattr[(int)JOB_ATR_jobname].at_val.at_str);
	if (stdmessage)
		fprintf(outmail, "%s\n", stdmessage);
	if (text != (char *)0)
		fprintf(outmail, "%s\n",text);
	fclose(outmail);

	exit(0);
}
