/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
/*
 * svr_recov.c - contains functions to save server state and recover
 *
 * Included functions are:
 *	svr_recov()
 *	svr_save()
 */
#include <pbs_config.h>   /* the master config generated by configure */

#include <sys/types.h>
#include <sys/param.h>
#include "pbs_ifl.h"
#include <sys/stat.h>
#include <errno.h>
#include <fcntl.h>
#include <stdlib.h>
#include "server_limits.h"
#include "list_link.h"
#include "attribute.h"
#include "queue.h"
#include "server.h"
#include "svrfunc.h"
#include "log.h"

static char ident[] = "@(#) $RCSfile: svr_recov.c,v $ $Revision: 2.1 $";

/*
 * the following funny business is due to the fact that O_SYNC
 * is not currently POSIX
 */
#ifdef O_SYNC
#define O_Sync O_SYNC
#elif _FSYNC
#define O_Sync _FSYNC
#else
#define O_Sync 0
#endif

/* Global Data Items: */

extern struct server server;
extern list_head svr_queues;
extern attribute_def svr_attr_def[];
extern char	    *path_svrdb;
extern char	    *path_svrdb_new;
extern char	    *path_priv;
extern time_t	     time_now;
extern char	    *msg_svdbopen;
extern char	    *msg_svdbnosv;

/*
 * svr_recov() - recover server state from server database
 */

int svr_recov(svrfile)
	char *svrfile;
{
	int i;
	int sdb;
	void recov_acl A_((attribute *, attribute_def *, char *, char *));

	sdb = open(svrfile, O_RDONLY, 0);
	if (sdb < 0) {
		log_err(errno, "svr_recov", msg_svdbopen);
		return (-1);
	}

	/* read in server structure */

	i = read(sdb, (char *)&server.sv_qs, sizeof (struct server_qs));
	if (i != sizeof (struct server_qs)) {
		if (i < 0) 
			log_err(errno, "svr_recov", "read of serverdb failed");
		else
			log_err(errno, "svr_recov", "short read of serverdb");
		(void)close(sdb);
		return (-1);
	}
	
	/* read in server attributes */

	if (recov_attr(sdb, &server, svr_attr_def, server.sv_attr,
	    (int)SRV_ATR_LAST,0) != 0 ) {
		log_err(errno, "svr_recov", "error on recovering server attr");
		(void)close(sdb);
		return (-1);
	}
	(void)close(sdb);
	
	/* recover the server various acls  from their own files */

	for (i=0; i<SRV_ATR_LAST; i++) {
		if (server.sv_attr[i].at_type == ATR_TYPE_ACL) {
			recov_acl(&server.sv_attr[i], &svr_attr_def[i],
				  PBS_SVRACL, svr_attr_def[i].at_name);
			if (svr_attr_def[i].at_action != (int (*)())0)
				svr_attr_def[i].at_action(&server.sv_attr[i],
							  &server,
							  ATR_ACTION_RECOV);
		}
	}

	return (0);
}

/*
 * svr_save - save the state of the server (server structure)
 */

int svr_save(ps, mode)
	struct server  *ps;
	int		mode;
{
	int i;
	int sdb;
	int save_acl A_((attribute *, attribute_def *, char *, char *));
	char *my_id = "svr_save";

	if (mode == SVR_SAVE_QUICK) {
		sdb = open(path_svrdb, O_WRONLY | O_CREAT | O_Sync, 0600);
		if (sdb < 0) {
			log_err(errno, "svr_recov", msg_svdbopen);
			return (-1);
		}
		while ((i=write(sdb, &ps->sv_qs, sizeof (struct server_qs))) !=
		    sizeof (struct server_qs)) {
			if ((i == -1) && (errno == EINTR))
				continue;
			log_err(errno, my_id, msg_svdbnosv);
			return (-1);
		}
		(void)close(sdb);

	} else {	/* SVR_SAVE_FULL Save */

		sdb = open(path_svrdb_new, O_WRONLY | O_CREAT | O_Sync, 0600);
		if (sdb < 0) {
			log_err(errno, "svr_recov", msg_svdbopen);
			return (-1);
		}

		ps->sv_qs.sv_savetm = time_now;

		save_setup(sdb);
		if (save_struct((char *)&ps->sv_qs, sizeof (struct server_qs)) != 0) {
			(void)close(sdb);
			return (-1);
		}

		if (save_attr(svr_attr_def,ps->sv_attr,(int)SRV_ATR_LAST) !=0) {
			(void)close(sdb);
			return (-1);
		}

		if (save_flush() != 0) {
			(void)close(sdb);
			return (-1);
		}

		(void)close(sdb);
		(void)unlink(path_svrdb);
		(void)link(path_svrdb_new, path_svrdb);
		(void)unlink(path_svrdb_new);

		/* save the server acls to their own files:	*/
		/* 	priv/svracl/(attr name)			*/

		for (i=0; i<SRV_ATR_LAST; i++) {
			if (ps->sv_attr[i].at_type == ATR_TYPE_ACL)
				save_acl(&ps->sv_attr[i], &svr_attr_def[i],
				 	PBS_SVRACL, svr_attr_def[i].at_name);
		}
	}
	return (0);
}


/*
 * save_acl() - save an Access Control List to its file 
 */

int save_acl(attr, pdef, subdir, name)
	attribute *attr;	/* acl attribute */
	attribute_def *pdef;	/* attribute def structure */
	char	  *subdir;	/* sub-directory path */
	char	  *name;	/* parent object name = file name */
{
	int  fds;
	char filename1[MAXPATHLEN];
	char filename2[MAXPATHLEN];
	list_head    head;
	int	     i;
	svrattrl    *pentry;

	if ((attr->at_flags & ATR_VFLAG_MODIFY) == 0)	
		return(0);  	/* Not modified, don't bother */

	attr->at_flags &= ~ATR_VFLAG_MODIFY;

	(void)strcpy(filename1, path_priv);
	(void)strcat(filename1, subdir);
	(void)strcat(filename1, "/");
	(void)strcat(filename1, name);

	if ((attr->at_flags & ATR_VFLAG_SET) == 0) {

		/* has been unset, delete the file */

		(void)unlink(filename1);
		return(0);
	}

	(void)strcpy(filename2, filename1);
	(void)strcat(filename2, ".new");

	fds = open(filename2, O_WRONLY|O_CREAT|O_TRUNC|O_Sync, 0600);
	if (fds < 0) {
		log_err(errno, "save_acl", "unable to open acl file");
		return (-1);
	}
	
	CLEAR_HEAD(head);
	i = pdef->at_encode(attr,&head,pdef->at_name,(char *)0,ATR_ENCODE_SAVE);
	if (i < 0) {
		log_err(-1, "save_acl", "unable to encode acl");
		(void)close(fds);
		(void)unlink(filename2);
		return (-1);
	}
	
	pentry = (svrattrl *)GET_NEXT(head);
	if (pentry != (svrattrl *)0) {
		/* write entry, but without terminating null */
		while ((i = write(fds, pentry->al_value, pentry->al_valln - 1))
		      != pentry->al_valln - 1) {
			if ((i == -1) && (errno == EINTR))
				continue;
			log_err(errno,"save_acl","wrote incorrect amount");	
			(void)close(fds);
			(void)unlink(filename2);
			return (-1);
		}
		(void)free(pentry);
	}

	(void)close(fds);
	(void)unlink(filename1);
	if (link(filename2, filename1) < 0) {
		log_err(errno, "save_acl", "unable to relink file");
		return (-1);
	}
	(void)unlink(filename2);
	attr->at_flags &= ~ATR_VFLAG_MODIFY;	/* clear modified flag */
	return (0);
}

/*
 * recov_acl() - reload an Access Control List from its file 
 */

void recov_acl(pattr, pdef, subdir, name)
	attribute *pattr;	/* acl attribute */
	attribute_def *pdef;	/* attribute def structure */
	char	  *subdir;	/* directory path */
	char	  *name;	/* parent object name = file name */
{
	char *buf;
	int   fds;
	char  filename1[MAXPATHLEN];
	struct stat sb;
	attribute tempat;

	errno = 0;
	(void)strcpy(filename1, path_priv);
	if (subdir != (char *)0) {
		(void)strcat(filename1, subdir);
		(void)strcat(filename1, "/");
	}
	strcat(filename1, name);

	fds = open(filename1, O_RDONLY, 0600);
	if (fds < 0) {
		if (errno != ENOENT) {
			(void)sprintf(log_buffer, "unable to open acl file %s",
					filename1);
			log_err(errno, "recov_acl", log_buffer);
		}
		return;
	}

	if (fstat(fds, &sb) < 0) {
		return;
	}
	if (sb.st_size == 0)
		return;		/* no data */
	buf = malloc((size_t)sb.st_size + 1);	/* 1 extra for added null */
	if (buf == (char *)0) {
		(void)close(fds);
		return;
	}
	if (read(fds, buf, (unsigned int)sb.st_size) != (int)sb.st_size) {
		log_err(errno, "recov_acl", "unable to read acl file");
		(void)close(fds);
		(void)free(buf);
		return;
	}
	(void)close(fds);
	*(buf + sb.st_size) = '\0';

	clear_attr(&tempat, pdef);

	if (pdef->at_decode(&tempat, pdef->at_name, (char *)0, buf) < 0) {
		(void)sprintf(log_buffer,  "decode of acl %s failed",
				pdef->at_name);
		log_err(errno, "recov_acl", log_buffer);
	} else if (pdef->at_set(pattr, &tempat, SET) != 0) {
		(void)sprintf(log_buffer, "set of acl %s failed",
				pdef->at_name);
		log_err(errno, "recov_acl", log_buffer);
	}
	pdef->at_free(&tempat);
	(void)free(buf);
	return;
}
