/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
#include <pbs_config.h>   /* the master config generated by configure */

#include <assert.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/types.h>
#include "pbs_ifl.h"
#include "list_link.h"
#include "attribute.h"
#include "resource.h"
#include "pbs_error.h"
#include "server_limits.h"
#include "server.h"
#include "job.h"

static char ident[] = "@(#) $RCSfile: svr_resccost.c,v $ $Revision: 2.1 $";

/*
 * This file contains the functions for manipulating the server
 * attribute "resource cost", which is of type ATR_TYPE_LIST
 *
 * It contains functions for:
 *	Decoding the value string to the machine representation, a long
 *		integer within the resource cost structure.
 *	Encoding the long integer value to external form
 *	Setting the value by =, + or - operators.
 *	Freeing the storage space used by the list.
 */
/*
 * note - it was my orginal intent to have the cost be an integer recorded
 *	in the resource_defination structure itself.  It seemed logical, one
 *	value per definition, why not.  But "the old atomic set" destroys that
 *	idea.  Have to be able to have temporary attributes with their own
 *	values...  Hence it came down to another linked-list of values.
 */

/*
 * Resource_cost entry, one per resource type which has been set.
 * The list is headed in the resource_cost attribute.
 */

struct resource_cost {
	list_link	rc_link;
	resource_def   *rc_def;
	long		rc_cost;
};

/*
 * add_cost_entry - add a new cost entry to the resource_cost list
 */

static struct resource_cost *add_cost_entry(patr, prdef)
	attribute    *patr;
	resource_def *prdef;
{
	struct resource_cost *pcost;

	pcost = malloc(sizeof (struct resource_cost));
	if (pcost) {
		CLEAR_LINK(pcost->rc_link);
		pcost->rc_def = prdef;
		pcost->rc_cost = 0;
		append_link(&patr->at_val.at_list, &pcost->rc_link, pcost);
	}
	return (pcost);
}

/*
 * decode_rcost - decode string into resource cost value
 *
 *	Returns: 0 if ok
 *		>0 error number if error
 *		*patr elements set
 */

int decode_rcost(patr, name, rescn, val)
	struct attribute *patr;
	char *name;		/* attribute name */
	char *rescn;		/* resource name, unused here */
	char *val;		/* attribute value */
{
	resource_def *prdef;
	struct resource_cost *pcost;
	void free_rcost A_((attribute *));
	

	if ((val == (char *)0) || (rescn == (char *)0)) {
		patr->at_flags = (patr->at_flags & ~ATR_VFLAG_SET) |
				 ATR_VFLAG_MODIFY;
		return (0);
	}
	if (patr->at_flags & ATR_VFLAG_SET) {
		free_rcost(patr);
	}

	prdef = find_resc_def(svr_resc_def, rescn, svr_resc_size);
	if (prdef == (resource_def *)0)
		return (PBSE_UNKRESC);
	pcost = (struct resource_cost *)GET_NEXT(patr->at_val.at_list);
	while (pcost) {
		if (pcost->rc_def == prdef)
			break;	/* have entry in attr already */
		pcost = (struct resource_cost *)GET_NEXT(pcost->rc_link);
	}
	if (pcost == (struct resource_cost *)0) {	/* add entry */
		if ((pcost=add_cost_entry(patr,prdef)) == (struct resource_cost *)0)
			return (PBSE_SYSTEM);
	}
	pcost->rc_cost = atol(val);
	patr->at_flags |= ATR_VFLAG_SET | ATR_VFLAG_MODIFY;
	return (0);
}

/*
 * encode_rcost - encode attribute of type long into attr_extern
 *
 *	Returns: >0 if ok
 *		 =0 if no value, no attrlist link added
 *		 <0 if error
 */
/*ARGSUSED*/


int encode_rcost(attr, phead, atname, rsname, mode)
	attribute	*attr;	  /* ptr to attribute */
	list_head	*phead;	  /* head of attrlist list */
	char		*atname;  /* attribute name */
	char		*rsname;  /* resource name or null */
	int		mode;	  /* encode mode, unused here */
{
	svrattrl *pal;
	struct resource_cost *pcost;

	if ( !attr )
		return (-1);
	if ( !(attr->at_flags & ATR_VFLAG_SET))
		return (0);

	pcost = (struct resource_cost *)GET_NEXT(attr->at_val.at_list);
	while (pcost) {
		rsname = pcost->rc_def->rs_name;
		if ((pal = attrlist_create(atname,rsname,23)) == (svrattrl *)0)
			return (-1);

		(void)sprintf(pal->al_value, "%ld", pcost->rc_cost);
		pal->al_flags = attr->at_flags;
		append_link(phead, &pal->al_link, pal);

		pcost = (struct resource_cost *)GET_NEXT(pcost->rc_link);
	}
	
	return (1);
}

/*
 * set_rcost - set attribute A to attribute B,
 *	either A=B, A += B, or A -= B
 *
 *	Returns: 0 if ok
 *		>0 if error
 */

int set_rcost(old, new, op)
	struct attribute *old;
	struct attribute *new;
	enum batch_op op;
{
	struct resource_cost *pcnew;
	struct resource_cost *pcold;

	assert(old && new && (new->at_flags & ATR_VFLAG_SET));

	pcnew = (struct resource_cost *)GET_NEXT(new->at_val.at_list);
	while (pcnew) {
	    pcold = (struct resource_cost *)GET_NEXT(old->at_val.at_list);
	    while (pcold) {
		if (pcnew->rc_def == pcold->rc_def)
		    break;
		pcold = (struct resource_cost *)GET_NEXT(pcold->rc_link);
	    }
	    if (pcold == (struct resource_cost *)0)
		if ((pcold = add_cost_entry(old, pcnew->rc_def)) == (struct resource_cost *)0)
			return (PBSE_SYSTEM);

	    switch (op) {
		case SET:   pcold->rc_cost = pcnew->rc_cost;
			    break;

		case INCR:  pcold->rc_cost += pcnew->rc_cost;
			    break;

		case DECR:  pcold->rc_cost -= pcnew->rc_cost;
			    break;

		default:    return (PBSE_INTERNAL);
	    }
	    pcnew = (struct resource_cost *)GET_NEXT(pcnew->rc_link);
	}
	old->at_flags |= (ATR_VFLAG_SET | ATR_VFLAG_MODIFY);
	return (0);
}


/*
 * free_rcost - free space used by resource cost attribute
 */

void free_rcost(pattr)
	attribute *pattr;
{
	struct resource_cost *pcost;

	while (pcost=(struct resource_cost *)GET_NEXT(pattr->at_val.at_list)) {
		delete_link(&pcost->rc_link);
		(void)free(pcost);
	}
	pattr->at_flags &= ~ATR_VFLAG_SET;
}

/*
 * calc_job_cost - Compute the "job cost":
 *	jc = sum(each resource cost * amount of resource) + per system cost
 *
 *	If a resource is of type "size", the product is likely to be large,
 *	so for those resources types, the cost is per MB.  All others, the
 *	cost is per unit.
 */

long calc_job_cost(pjob)
	job *pjob;
{
	long      amt;
	long      cost = 0;
	struct resource_cost *pcost;
	resource *pjobr;
	int	  shiftct;

	pcost = (struct resource_cost *)GET_NEXT(server.sv_attr[(int)SRV_ATR_resource_cost].at_val.at_list);
	while (pcost) {
		pjobr = find_resc_entry(&pjob->ji_wattr[(int)JOB_ATR_resource],
					pcost->rc_def);
		if (pjobr) {
		    switch (pcost->rc_def->rs_type) {
			case ATR_TYPE_LONG:
			    cost += pjobr->rs_value.at_val.at_long * pcost->rc_cost;
			    break;

			case ATR_TYPE_SIZE:
			    amt = pjobr->rs_value.at_val.at_size.atsv_num;
			    shiftct = pjobr->rs_value.at_val.at_size.atsv_shift - 20;
			    if (shiftct < 0)
				amt = amt >> -shiftct;
			    else
				amt = amt << shiftct;

			    if (pjobr->rs_value.at_val.at_size.atsv_units)
				amt *= sizeof (int);

			    cost += amt * pcost->rc_cost;		    
			    break;
		    }
		}
			
		pcost = (struct resource_cost *)GET_NEXT(pcost->rc_link);
	}
		
	cost += server.sv_attr[(int)SVR_ATR_sys_cost].at_val.at_long;
	return (cost);
}
