/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
/*
 * svr_task.c - contains functions to deal with the server's task list
 *
 * Functions included are:
 */
#include <pbs_config.h>   /* the master config generated by configure */

#include "portability.h"
#include <stdlib.h>
#include <time.h>
#include <sys/param.h>
#include <sys/types.h>
#include <sys/wait.h>
#include "server_limits.h"
#include "list_link.h"
#include "work_task.h"

static char ident[] = "@(#) $RCSfile: svr_task.c,v $ $Revision: 2.1 $";

/* Global Data Items: */

extern list_head task_list_immed;
extern list_head task_list_timed;
extern list_head task_list_event;

/*
 * set_task - add the job entry to the task list
 *
 * Task time depends on the type of task.  The list is time ordered.
 */
struct work_task *set_task(type, event_id, func, parm)
	enum work_type	 type;
	long		 event_id;
	void	       (*func)();
	void		*parm;
{
	struct work_task *pnew;
	struct work_task *pold;

	pnew = (struct work_task *)malloc(sizeof (struct work_task));
	if (pnew == (struct work_task *)0)
		return ((struct work_task *)0);
	CLEAR_LINK(pnew->wt_linkall);
	CLEAR_LINK(pnew->wt_linkobj);
	pnew->wt_event = event_id;
	pnew->wt_type  = type;
	pnew->wt_func  = func;
	pnew->wt_parm1 = parm;
	pnew->wt_parm2 = NULL;
	pnew->wt_aux   = 0;

	if (type == WORK_Immed)
		append_link(&task_list_immed, &pnew->wt_linkall, pnew);
	else if (type == WORK_Timed) {
		pold = (struct work_task *)GET_NEXT(task_list_timed);
		while (pold) {
			if (pold->wt_event > pnew->wt_event)
				break;
			pold = (struct work_task *)GET_NEXT(pold->wt_linkall);
		}
		if (pold)
			insert_link(&pold->wt_linkall, &pnew->wt_linkall, pnew,
				    LINK_INSET_BEFORE);
		else
			append_link(&task_list_timed, &pnew->wt_linkall, pnew);
	} else
		append_link(&task_list_event, &pnew->wt_linkall, pnew);
	return (pnew);
}

/*
 * dispatch_task - dispatch a work task found on a work list
 *
 *	delinks the work task entry, calls the associated function with
 *	the parameters from the work task entry, and then frees the entry.
 */

void dispatch_task(ptask)
	struct work_task *ptask;
{
	delete_link(&ptask->wt_linkall);
	delete_link(&ptask->wt_linkobj);
	if (ptask->wt_func)
		ptask->wt_func(ptask);		/* dispatch process function */
	(void)free(ptask);
}

/*
 * delete_task - unlink and free a work_task structure.
 */

void delete_task(ptask)
	struct work_task *ptask;
{
	delete_link(&ptask->wt_linkobj);
	delete_link(&ptask->wt_linkall);
	(void)free(ptask);
}
