/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/

/*
 * pbs_demux - handle I/O from multiple node job
 *
 *	Standard Out and Standard Error of each task is bound to
 *	stream sockets connected to pbs_demux which inputs from the
 *	various streams and writes to the JOB's out and error.
 */

#include <pbs_config.h>   /* the master config generated by configure */

#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#if defined(FD_SET_IN_SYS_SELECT_H)
#  include <sys/select.h>
#endif


enum rwhere {invalid, new_out, new_err, old_out, old_err};
struct routem {
	enum rwhere	r_where;
	short		r_nl;
	short		r_first;
};
fd_set readset;
char   *cookie = 0;


void readit(int sock, struct routem *prm)
{
	int   amt;
	char  buf[256];
	FILE *fil;
	int   i;
	char *pc;

	if (prm->r_where == old_out)
		fil = stdout;
	else
		fil = stderr;

	i = 0;
	if ((amt = read(sock, buf, 256)) > 0) {
	    if (prm->r_first == 1) {

		/* first data on connection must be the cookie to validate it */

		i = strlen(cookie);
		if (strncmp(buf, cookie, i) != 0) {
		    (void)close(sock);
		    prm->r_where = invalid;
		    FD_CLR(sock, &readset);
		}
		prm->r_first = 0;
	    }
	    for (pc = buf+i; pc < buf+amt; ++pc) {
#ifdef DEBUG
		if (prm->r_nl) {
			fprintf(fil, "socket %d: ", sock);
			prm->r_nl = 0;
		}
#endif /* DEBUG */
		putc(*pc, fil);
		if (*pc == '\n') {
			prm->r_nl = 1;
			fflush(fil);
		}
	    }
	} else {
		close(sock);
		prm->r_where = invalid;
		FD_CLR(sock, &readset);
	}
	return;
}
	
	
main(int argc, char *argv[])
{
	struct timeval timeout;
	int	i;
	int	maxfd;
	int	main_sock_out = 3;
	int	main_sock_err = 4;
	int	n;
	int	newsock;
	pid_t	parent;
	fd_set selset;
	struct routem *routem;


	parent = getppid();
	cookie = getenv("PBS_JOBCOOKIE");
	if (cookie == 0) {
		fprintf(stderr, "%s: no PBS_JOBCOOKIE found in the env\n",
			argv[0]);
		exit(3);
	}
#ifdef DEBUG
	printf("Cookie found in environment: %s\n", cookie);
#endif

	maxfd = sysconf(_SC_OPEN_MAX);
	routem = (struct routem *)malloc(maxfd*sizeof(struct routem));
	for (i=0; i<maxfd; ++i) {
		(routem+i)->r_where = invalid;
		(routem+i)->r_nl    = 1;
		(routem+i)->r_first = 0;
	}
	(routem+main_sock_out)->r_where = new_out;
	(routem+main_sock_err)->r_where = new_err;
	

	FD_ZERO(&readset);
	FD_SET(main_sock_out, &readset);
	FD_SET(main_sock_err, &readset);

	if (listen(main_sock_out, 5) < 0) {
		perror("listen on out");
		exit(5);
	}

	if (listen(main_sock_err, 5) < 0) {
		perror("listen on err");
		exit(5);
	}

	while (1) {

	    selset = readset;
	    timeout.tv_usec = 0;
	    timeout.tv_sec  = 10;

	    n = select(FD_SETSIZE, &selset, (fd_set *)0, (fd_set *)0, &timeout);
	    if (n == -1) {
		if (errno == EINTR) {
			n = 0;
		} else {
			fprintf(stderr,"%s: select failed\n", argv[0]);
			exit(1);
		}
	    } else if (n == 0) {
		if (getppid() == 1) {
#ifdef DEBUG
			fprintf(stderr, "%s: Parent has gone, and so do I\n",
				argv[0]);
#endif	/* DEBUG */
			break;
		}
	    }

	    for (i=0; n && i<maxfd; ++i) {
		if (FD_ISSET(i, &selset)) {     /* this socket has data */
		    n--;
		    switch ((routem+i)->r_where) {
			case new_out:
			case new_err:
				newsock = accept(i, 0, 0);
				(routem+newsock)->r_where =  (routem+i)->r_where== new_out ? old_out : old_err;
				FD_SET(newsock, &readset);
				(routem+newsock)->r_first = 1;
				break;
			case old_out:
			case old_err:
				readit(i, routem+i);
				break;
			default:
				fprintf(stderr, "%s: internal error\n",argv[0]);
				exit(2);
		    }
		}
	    }
	}
	return 0;
}
