/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
/*
 * qterm
 *  The qterm command terminates the batch server.
 *
 * Synopsis:
 *  qterm [-t type] [server ...]
 *
 * Options:
 *  -t  delay   Jobs are (1) checkpointed if possible; otherwise, (2) jobs are
 *		rerun (requeued) if possible; otherwise, (3) jobs are left to
 *              run.
 *
 *      immediate
 *              Jobs are (1) checkpointed if possible; otherwise, (2) jobs are
 *		rerun if possible; otherwise, (3) jobs are aborted.
 *
 *	quick
 *		The server will save state and exit leaving running jobs
 *		still running.  Good for shutting down when you wish to
 *		quickly restart the server.
 *
 * Arguments:
 *  server ...
 *      A list of servers to terminate.
 *
 * Written by:
 *  Bruce Kelly
 *  National Energy Research Supercomputer Center
 *  Livermore, CA
 *  May, 1993
 */

#include "cmds.h"
#include <pbs_config.h>   /* the master config generated by configure */

static char ident[] = "@(#) $RCSfile: qterm.c,v $ $Revision: 2.1 $";

int exitstatus = 0; /* Exit Status */

static void execute ();

main ( argc, argv )
int argc;
char **argv;
{
/*
 *  This routine sends a Server Shutdown request to the batch server.  If the
 * batch request is accepted, and the type is IMMEDIATE, then no more jobs
 * are accepted and all jobs are checkpointed or killed.  If the type is
 * DELAY, then only privileged users can submit jobs, and jobs will be
 * checkpointed if available.
 */

    static char opts[] = "t:";     /* See man getopt */
    int s;                  /* The execute line option */
    static char usage[] = "Usage: qterm [-t immediate|delay|quick] [server ...]\n";
    char *type = NULL;      /* Pointer to the type of termination */
    int manner;             /* The type of termination */
    int errflg = 0;         /* Error flag */
    
    /* Command line options */
    while ((s = getopt(argc, argv, opts)) != EOF)
        switch (s) {
        case 't':
            type = optarg;
            break;
        case '?':
        default:
            errflg++;
            break;
        }
    if ( errflg ) {
        fprintf(stderr, usage);
        exit(1);
    } else if ( type != NULL ) {
        if ( strcmp(type, "delay") == 0 )
            manner = SHUT_DELAY;
        else if ( strcmp(type, "immediate") == 0 )
            manner = SHUT_IMMEDIATE;
	else if ( strcmp(type, "quick") == 0)
	    manner = SHUT_QUICK;
        else {
            fprintf(stderr, usage);
            exit(1);
        }
    } else
        manner = SHUT_IMMEDIATE;

    if ( optind < argc )
        for ( ; optind < argc; optind++ )
            execute(manner, argv[optind]);
    else
	execute(manner, "");

    exit (exitstatus);
}


/*
 * void execute( int manner, char *server )
 *
 * manner   The manner in which to terminate the server.
 * server   The name of the server to terminate.
 *
 * Returns:
 *  None
 *
 * File Variables:
 *  exitstatus  Set to two if an error occurs.
 */
static void
execute( manner, server )
int manner;
char *server;
{
    int ct;         /* Connection to the server */
    int err;        /* Error return from pbs_terminate */
    char *errmsg;   /* Error message from pbs_terminate */
    
    if ( (ct = cnt2server(server)) > 0 ) {
        err = pbs_terminate(ct, manner, NULL);
        if ( err != 0 ) {
            errmsg = pbs_geterrmsg(ct);
            if ( errmsg != NULL ) {
                fprintf(stderr, "qterm: %s ", errmsg);
            } else {
                fprintf(stderr, "qterm: Error (%d) terminating server ", pbs_errno);
            }
            fprintf(stderr, "%s\n", server);
            exitstatus = 2;
        }
        pbs_disconnect(ct);
    } else {
        fprintf(stderr, "qterm: could not connect to server %s (%d)\n", server, pbs_errno);
        exitstatus = 2;
    }
}
