/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
#include <pbs_config.h>   /* the master config generated by configure */

#include <ctype.h>
#include <memory.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "portability.h"
#include "pbs_ifl.h"
#include "list_link.h"
#include "attribute.h"
#include "resource.h"
#include "pbs_error.h"

static char ident[] = "@(#) $RCSfile: attr_atomic.c,v $ $Revision: 2.1 $";

/*
 * This file contains general functions for manipulating an attribute array.
 * Included is:
 *	attr_atomic_set()
 *	attr_atomic_node_set()
 *	attr_atomic_kill()
 *
 * The prototypes are declared in "attr_func.h"
 */

/* Global Variables */

extern int resc_access_perm;	/* see lib/Libattr/attr_fn_resc.c */

/*
 * attr_atomic_set - atomically set a attribute array with values from
 * an svrattrl
*/

int attr_atomic_set(plist, old, new, pdef, limit, unkn, privil, badattr)
	struct svrattrl *plist;
	attribute	*old;
	attribute	*new;
	attribute_def	*pdef;
	int		 limit;
	int		 unkn;
	int		 privil;
	int		*badattr;
{
	int	    acc;
	int	    index;
	int	    listidx;
	resource   *prc;
	int	    rc;
	attribute   temp;

	for (index=0; index<limit; index++)
		clear_attr(new+index, pdef+index);

	resc_access_perm = privil;	/* set privilege for decode_resc() */

	listidx = 0;
	while (plist) {
		listidx++;

		if ((index = find_attr(pdef, plist->al_name, limit)) < 0) {
			if (unkn < 0) {	          /*unknown attr isn't allowed*/
				rc =  PBSE_NOATTR;
				break;
			} else
				index = unkn;  /* if unknown attr are allowed */
		}
	
		/* have we privilege to set the attribute ? */

		acc = (pdef+index)->at_flags & ATR_DFLAG_ACCESS;
		if ((acc & privil & ATR_DFLAG_WRACC ) == 0) {
			if (privil & ATR_DFLAG_SvWR) {
			    /*from a daemon, just ignore this attribute*/
			    plist = (struct svrattrl *)GET_NEXT(plist->al_link);
			    continue;
			} else {
			    /*from user, error if can't write attribute*/
			    rc = PBSE_ATTRRO;
			    break;
			}
		}

		/* decode new value */

		clear_attr(&temp, pdef+index);
		if (rc = (pdef+index)->at_decode(&temp, plist->al_name,
		      plist->al_resc, plist->al_value)) {
			if ((rc == PBSE_UNKRESC) && (unkn > 0))
				rc = 0;	/* ignore the "error" */
			else
				break;
		}

		/* duplicate current value, if set AND not already dup-ed */

		if (((old+index)->at_flags & ATR_VFLAG_SET) &&
		    !((new+index)->at_flags & ATR_VFLAG_SET)) {
			if (rc = (pdef+index)->at_set(new+index,old+index,SET))
				break;
			/*
			 * we need to know if the value is changed during
			 * the next step, so clear MODIFY here; including
			 * within resources.
			 */
			(new+index)->at_flags &= ~ATR_VFLAG_MODIFY;
			if ((new+index)->at_type == ATR_TYPE_RESC) {
			    prc = (resource *)GET_NEXT((new+index)->at_val.at_list);
			    while (prc) {
				prc->rs_value.at_flags &= ~ATR_VFLAG_MODIFY;
				prc = (resource *)GET_NEXT(prc->rs_link);
			    }
			}
		}

		/* update new copy with temp, MODIFY is set on ones changed */

		if ((plist->al_op != INCR) && (plist->al_op != DECR) &&
		    (plist->al_op != SET))
			plist->al_op = SET;

		if (temp.at_flags & ATR_VFLAG_SET) {
		    if (rc=(pdef+index)->at_set(new+index,&temp,plist->al_op)) {
			(pdef+index)->at_free(&temp);
			break;
		    }
		} else if (temp.at_flags & ATR_VFLAG_MODIFY) {
		    (pdef+index)->at_free(new+index);
		    (new+index)->at_flags |= ATR_VFLAG_MODIFY;
		}

		(pdef+index)->at_free(&temp);
		plist = (struct svrattrl *)GET_NEXT(plist->al_link);
	}

	if (rc != 0) {
		*badattr = listidx;
		for (index=0; index<limit; index++)
			(pdef+index)->at_free(new+index);
		
		return (rc);
	} else 
		return (0);
}




/*
 * attr_atomic_node_set - atomically set an attribute array with
 * values from an svrattrl
*/

int attr_atomic_node_set(plist, old, new, pdef, limit, unkn, privil, badattr)
	struct svrattrl *plist;		/*list of attribute modif structs*/
	attribute	*old;		/*unused*/
	attribute	*new;		/*new attribute array begins here*/
	attribute_def	*pdef;		/*begin array  definition structs*/
	int		 limit;		/*number elts in definition array*/
	int		 unkn;		/*<0 unknown attrib not permitted*/
	int		 privil;	/*requester's access privileges  */
	int		*badattr;	/*return list position wher bad  */ 
{
	int	    acc;
	int	    index;
	int	    listidx;
	resource   *prc;
	int	    rc;
	attribute   temp;



	listidx = 0;
	while (plist) { /*Traverse loop for each client entered attribute*/
		listidx++;

		if ((index = find_attr(pdef, plist->al_name, limit)) < 0) {
			if (unkn < 0) {	      /*if unknown attr not allowed*/
				rc =  PBSE_NOATTR;
				break;
			} else
				index = unkn;  /*if unknown attr are allowed*/
		}
	

		/* The name of the attribute is in the definitions list*/
		/* Now, have we privilege to set the attribute ?       */
		/* Check access capabilities specified in the attrdef  */
		/* against the requestor's privilege level	       */

		acc = (pdef+index)->at_flags & ATR_DFLAG_ACCESS;
		if ((acc & privil & ATR_DFLAG_WRACC ) == 0) {
			if (privil & ATR_DFLAG_SvWR) {
			    /*  from a daemon, just ignore this attribute */
			    plist = (struct svrattrl *)GET_NEXT(plist->al_link);
			    continue;
			} else {
			    /*from user, no write access to attribute     */
			    rc = PBSE_ATTRRO;
			    break;
			}
		}


		/*decode new value*/
		clear_attr(&temp, pdef+index);

		if (rc = (pdef+index)->at_decode(&temp, plist->al_name,
		      plist->al_resc, plist->al_value)) {
			if ((rc == PBSE_UNKRESC) && (unkn > 0))
				rc = 0;              /*ignore the "error"*/
			else
				break;
		}



		/*update "new" with "temp", MODIFY is set on "new" if changed*/

		(new+index)->at_flags &= ~ATR_VFLAG_MODIFY;

		if ((plist->al_op != INCR) && (plist->al_op != DECR) &&
		    (plist->al_op != SET))
			plist->al_op = SET;


		if (temp.at_flags & ATR_VFLAG_SET) {
		    /* "temp" has a data value, use it to update "new" */

		    if (rc=(pdef+index)->at_set(new+index,&temp,plist->al_op)) {
			(pdef+index)->at_free(&temp);
			break;
		    }
		} else if (temp.at_flags & ATR_VFLAG_MODIFY) {

		    (pdef+index)->at_free(new+index);
		    (new+index)->at_flags |= ATR_VFLAG_MODIFY;
		}

		(pdef+index)->at_free(&temp);
		plist = (struct svrattrl *)GET_NEXT(plist->al_link);
	}

	if (rc != 0) {

		/*"at_free" functions get invoked by upstream caller*/
		/*invoking attr_atomic_kill() on the array of       */
		/*node-attribute structs-- any hanging structs are  */
		/*freed and then the node-attribute array is freed  */

		*badattr = listidx;	  /*the svrattrl that gave a problem*/
	}
	return (rc);
}



/*
 * attr_atomic_kill - kill (free) a temporary attribute array which
 *	was set up by attr_atomic_set().
 *
 *	at_free() is called on each element on the array, then
 *	the array itself is freed.
 */

void attr_atomic_kill(temp, pdef, limit)
	attribute	*temp;
	attribute_def	*pdef;
	int		 limit;
{
	int i;
 
	for (i=0; i<limit; i++)
		(pdef+i)->at_free(temp+i);
	(void)free(temp);
}
