/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
#include <pbs_config.h>   /* the master config generated by configure */

#include <limits.h>
#include <assert.h>
#include <ctype.h>
#include <memory.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pbs_ifl.h>
#include "list_link.h"
#include "attribute.h"
#include "pbs_error.h"

static char ident[] = "@(#) $RCSfile: attr_fn_time.c,v $ $Revision: 2.1 $";

/*
 * This file contains functions for manipulating attributes of type
 *	time:	[[hh:]mm:]ss[.sss]
 *
 * Each set has functions for:
 *	Decoding the value string to the machine representation.
 *	Encoding the internal attribute to external form
 *	Setting the value by =, + or - operators.
 *	Comparing a (decoded) value with the attribute value.
 *
 * Some or all of the functions for an attribute type may be shared with
 * other attribute types.
 * 
 * The prototypes are declared in "attribute.h"
 *
 * --------------------------------------------------
 * The Set of Attribute Functions for attributes with
 * value type "long"
 * --------------------------------------------------
 */


/*
 * decode_time - decode time into into attribute structure of type ATR_TYPE_LONG
 *
 *	Returns: 0 if ok
 *		>0 error number if error
 *		*patr elements set
 */

#define PBS_MAX_TIME (LONG_MAX - 1)
int decode_time(patr, name, rescn, val)
	struct attribute *patr;
	char *name;		/* attribute name */
	char *rescn;		/* resource name, unused here */
	char *val;		/* attribute value */
{
	int   i;
	char  msec[4];
	int   ncolon = 0;
	char *pc;
	long  rv = 0;
	char *workval;
	char *workvalsv;

	if ((val == (char *)0) || (strlen(val) == 0)) {
		patr->at_flags = (patr->at_flags & ~ATR_VFLAG_SET) |
				 ATR_VFLAG_MODIFY;
		patr->at_val.at_long = 0;
		return (0);
	}

	workval = strdup(val);
	workvalsv = workval;

	for (i = 0; i < 3; ++i)
		msec[i] = '0';
	msec[i] = '\0';

	for (pc = workval; *pc; ++pc) {

		if (*pc == ':') {
			if (++ncolon > 2)
				goto badval;
			*pc = '\0';
			rv = (rv * 60) + atoi(workval);
			workval = pc + 1;

		} else if (*pc == '.') {
			*pc++ = '\0';
			for (i = 0; (i < 3) && *pc; ++i)
				msec[i] = *pc++;
			break;
		} else if ( !isdigit((int)*pc) ) { 
			goto badval;	/* bad value */
		}
	}
	rv = (rv * 60) + atoi(workval);
	if (rv > PBS_MAX_TIME)
		goto badval;
	if (atoi(msec) >= 500)
		rv++;
	patr->at_val.at_long = rv;
	patr->at_flags |= ATR_VFLAG_SET | ATR_VFLAG_MODIFY;
	(void)free(workvalsv);
	return (0);

badval:	(void)free(workvalsv);
	return (PBSE_BADATVAL);
}

/*
 * encode_time - encode attribute of type long into attr_extern
 *	with value in form of [[hh:]mm:]ss
 *
 *	Returns: >0 if ok
 *		 =0 if no value, no attrlist link added
 *		 <0 if error
 */
/*ARGSUSED*/

#define CVNBUFSZ 21

int encode_time(attr, phead, atname, rsname, mode)
	attribute	*attr;	  /* ptr to attribute */
	list_head	*phead;	  /* head of attrlist list */
	char		*atname;  /* attribute name */
	char		*rsname;  /* resource name or null */
	int		mode;	  /* encode mode, unused here */
{
	size_t	  ct;
	char	  cvnbuf[CVNBUFSZ];
	int 	  hr;
	int	  min;
	long	  n;
	svrattrl *pal;
	int	  sec;
	char	 *pv;

	if ( !attr )
		return (-1);
	if ( !(attr->at_flags & ATR_VFLAG_SET))
		return (0);

	n   = attr->at_val.at_long;
	hr  = n / 3600;
	n   = n % 3600;
	min = n / 60;
	n   = n % 60;
	sec = n;

	pv = cvnbuf;
	(void)sprintf(pv, "%02d:%02d:%02d", hr, min, sec);
	pv += strlen(pv);

	ct = strlen(cvnbuf) + 1;

	pal = attrlist_create(atname, rsname, ct);
	if (pal == (svrattrl *)0)
		return (-1);

	(void)memcpy(pal->al_value, cvnbuf, ct);
	pal->al_flags = attr->at_flags;
	append_link(phead, &pal->al_link, pal);
	
	return (1);
}

/*
 * set_time  - use the function set_l()
 *
 * comp_time - use the funttion comp_l()
 *
 * free_l - use free_null to (not) free space
 */
