/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/

#include <pbs_config.h>   /* the master config generated by configure */

#include <sys/types.h>
#include <ctype.h>
#include <memory.h>
#ifndef NDEBUG
#include <stdio.h>
#endif
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include "pbs_ifl.h"
#include "list_link.h"
#include "attribute.h"
#include "server_limits.h"
#include "net_connect.h"
#include "job.h"
#include "pbs_nodes.h"
#include "pbs_error.h"

static char ident[] = "@(#) $RCSfile: attr_node_func.c,v $ $Revision: 2.5 $";

/*
 * This file contains functions for deriving attribute values from a pbsnode
 * and for updating the "state" (inuse), "node type" (ntype) or "properties"
 * list using the "value" carried in an attribute.
 *
 * Included are:
 *
 * global:
 * decode_state()		"functions for at_decode func pointer"
 * decode_ntype()
 * decode_props()
 *
 * encode_state()		"functions for at_encode func pointer"
 * encode_ntype()
 * encode_props()
 * encode_jobs()
 *
 * set_node_state()		"functions for at_set func pointer"
 * set_node_ntype()
 *
 * node_state()			"functions for at_action func pointer"
 * node_ntype()
 * node_prop_list()
 *
 * free_prop_list()
 * free_prop_attr()		"function for at_free func pointer"
 *
 * local:
 * load_prop()
 * set_nodeflag()
 *
 * The prototypes are declared in "attr_func.h"
 */


/*
 * Set of forward declarations for functions used before defined
 * keeps the compiler happy
*/
static	int	set_nodeflag A_(( char*, short*));
static	int	load_prop A_(( char*, struct prop* ));


/*
 * encode_state
 * Once the node's "inuse" field is converted to an attribute,
 * the attribute can be passed to this function for encoding into
 * an svrattrl structure
 * Returns  <0       an error encountered; value is negative of an error code
 *           0       ok, encode happened and svrattrl created and linked in,
 *		     or nothing to encode
 */
 
int	encode_state(pattr, ph, aname, rname, mode)
	attribute	 *pattr;	/*struct attribute being encoded  */
	list_head	 *ph;		/*head of a list of  "svrattrl"   */
					/*structs which are to be returned*/

	char		 *aname;	/*attribute's name		  */
	char		 *rname;	/*resource's name (null if none)  */
	int		 mode;		/*mode code, unused here	  */

{
	int	  i;
        svrattrl *pal;
	short	  state;
	static struct node_state {
		short	 bit;
		char	*name;
	} ns[] = {	{INUSE_UNKNOWN,	ND_state_unknown},
			{INUSE_DOWN,    ND_down},
			{INUSE_OFFLINE, ND_offline},
			{INUSE_RESERVE, ND_reserve},
			{INUSE_JOB,	ND_job_exclusive},
			{INUSE_JOBSHARE,ND_job_sharing},
			{INUSE_BUSY,	ND_busy},
			{0,		(char *)0} };
		

	
	static   char	state_str[ MAX_ENCODE_BFR ];

        if ( !pattr )
		return -(PBSE_INTERNAL);

        if ( !(pattr->at_flags & ATR_VFLAG_SET) )
                return (0);      /*nothing to report back*/
	
	state = pattr->at_val.at_short & (INUSE_SUBNODE_MASK | INUSE_UNKNOWN);
	if (!state)
	    strcpy( state_str, ND_free );

	else {
	   state_str[0] = '\0';
	   for (i=0; ns[i].name; i++) {
		if (state & ns[i].bit) {
		    if (state_str[0] != '\0')
			(void)strcat(state_str, ",");
		    (void)strcat(state_str, ns[i].name);
		}
	   }
	}
	
        pal = attrlist_create(aname,rname,(int)strlen(state_str)+1);
        if (pal == (svrattrl *)0)
                return -(PBSE_SYSTEM);

        (void)strcpy(pal->al_value, state_str);
        pal->al_flags = ATR_VFLAG_SET;
        append_link(ph, &pal->al_link, pal);

        return (0);             /*success*/
}


/*
 * encode_ntype
 * Once the node's "ntype" field is converted to an attribute,
 * the attribute can be passed to this function for encoding into
 * an svrattrl structure
 * Returns  <0       an error encountered; value is negative of an error code
 *           0       ok, encode happened and svrattrl created and linked in,
 *		     or nothing to encode
 */
 
int	encode_ntype(pattr, ph, aname, rname, mode)
	attribute	 *pattr;	/*struct attribute being encoded  */
	list_head	 *ph;		/*head of a list of  "svrattrl"   */
					/*structs which are to be returned*/

	char		 *aname;	/*attribute's name		  */
	char		 *rname;	/*resource's name (null if none)  */
	int		 mode;		/*mode code, unused here	  */

{
        svrattrl *pal;
	short	 ntype;
	static	 char	*nt[] = { ND_cluster, ND_timeshared };

	static   char	ntype_str[ MAX_ENCODE_BFR ];

        if ( !pattr )
                return -(PBSE_INTERNAL);

        if ( !(pattr->at_flags & ATR_VFLAG_SET) )
                return (0);      /*nothing to report back*/
	
	ntype = pattr->at_val.at_short & PBSNODE_NTYPE_MASK;
	if (!ntype)
	    strcpy( ntype_str, nt[0] );

	else
	    strcpy( ntype_str, nt[1] );
	
        pal = attrlist_create(aname,rname,(int)strlen(ntype_str)+1);
        if (pal == (svrattrl *)0)
                return -(PBSE_SYSTEM);

        (void)strcpy(pal->al_value, ntype_str);
        pal->al_flags = ATR_VFLAG_SET;
        append_link(ph, &pal->al_link, pal);

        return (0);             /*success*/
}


/*
 * encode_jobs
 * Once the node's struct jobinfo pointer is put in the data area of 
 * temporary attribute containing a pointer to the parent node, this
 * function will walk the list of jobs and generate the comma separated 
 * list to send back via an svrattrl structure.
 * Returns  <0       an error encountered; value is negative of an error code
 *           0       ok, encode happened and svrattrl created and linked in,
 *		     or nothing to encode
 */
 
int	encode_jobs(pattr, ph, aname, rname, mode)
	attribute	 *pattr;	/*struct attribute being encoded  */
	list_head	 *ph;		/*head of a  list of "svrattrl"   */
					/*structs which are to be returned*/

	char		 *aname;	/*attribute's name		  */
	char		 *rname;	/*resource's name (null if none)  */
	int		 mode;		/*mode code, unused here	  */

{
        svrattrl	*pal;
	struct jobinfo  *jip;
	struct pbsnode	*pnode;
	struct pbssubn 	*psubn;
	int		i;
	int		jobcnt;		/*number of jobs using the node     */
	int		strsize;	/*computed string size		    */
	char		*job_str;	/*holds comma separated list of jobs*/

        if ( !pattr )
                return (-1);
        if ( !(pattr->at_flags & ATR_VFLAG_SET) || !pattr->at_val.at_jinfo )
                return (0);                  /*nothing to report back   */
	
	/*cnt number of jobs and estimate size of string buffer required*/
	jobcnt = 0;
	strsize = 1;			     /*allow for terminating null char*/
	pnode = pattr->at_val.at_jinfo;
	for (psubn = pnode->nd_psn; psubn; psubn = psubn->next) {
	    for ( jip = psubn->jobs; jip; jip = jip->next) {
		jobcnt++;
		strsize += strlen( jip->job->ji_qs.ji_jobid ) + 4;
	    }
	}

	if( jobcnt == 0 )
	      return (0);		     /*no jobs currently on this node*/

	else if ( !(job_str = (char *)malloc( strsize )) )
	      return -(PBSE_SYSTEM);

	job_str[0] = '\0';
	i = 0;
	for (psubn = pnode->nd_psn; psubn; psubn = psubn->next) {
	    for ( jip = psubn->jobs; jip; jip = jip->next) {
	        if (i != 0 ) 
		    strcat(job_str, ", ");
		else
		    i++;
		
		sprintf(job_str + strlen(job_str), "%d/%s", psubn->index,
			jip->job->ji_qs.ji_jobid);	
	    }
	}


        pal = attrlist_create(aname, rname, (int)strlen(job_str) + 1  );
        if ( pal == (svrattrl *)0 ) {
		free(job_str);
                return -(PBSE_SYSTEM);
	}

        (void)strcpy(pal->al_value, job_str);
        pal->al_flags = ATR_VFLAG_SET;
	free( job_str );

        append_link(ph, &pal->al_link, pal);

        return (0);                  /*success*/
}


/*
 * decode_state 
 * In this case, the two arguments that get  used are
 * pattr-- it points to an attribute whose value is a short,
 * and the argument "val".
 * Once the "value" argument, val, is decoded from its form
 * as a string of comma separated substrings, the component
 * values are used to set the appropriate bits in the attribute's
 * value field.
 */
 
int	decode_state(pattr, name, rescn, val)
        attribute *pattr;
        char *name;             /* attribute name */
        char *rescn;            /* resource name, unused here */
        char *val;              /* attribute value */

{
	int	rc = 0;		/*return code; 0==success*/
	short	flag, currflag;
	char    *str;

	char	strbuf[512];	/*should handle most vals*/
	char	*sbufp;
	int	slen;


	if(val == (char *)0)
		return (PBSE_BADNDATVAL);


        /*
         * determine string storage requirement and copy the string "val"
         * to a work buffer area
         */

	slen = strlen( val );	/*bufr either on stack or heap*/
	if( slen - 512 < 0 )
	    sbufp = strbuf;
	else {
	    if( !(sbufp = (char *)malloc( slen + 1 )) )
		return (PBSE_SYSTEM);
	}

	strcpy( sbufp, val );

	if( (str = parse_comma_string( sbufp )) == (char *)0 ) {
	     if( slen >= 512 )
		 free( sbufp );
	     return rc;
	}

	flag = 0;
	if( rc = set_nodeflag( str, &flag ) ) {
	    if( slen >= 512 )
		 free( sbufp );
	    return rc;
	}
	currflag = flag;


	/*calling parse_comma_string with a null ptr continues where*/
	/*last call left off.  The initial comma separated string   */
	/*copy pointed to by sbufp is modified with each func call  */

	while( str = parse_comma_string( (char *)0 ) ) {
	    if( rc = set_nodeflag( str, &flag ) )
		break;

	    if( (currflag == 0 && flag) || (currflag && flag == 0) ) {
		rc = PBSE_MUTUALEX;	/*free is mutually exclusive*/
		break;
	    }
	    currflag = flag;
	}

	if( !rc )
	{
	    pattr->at_val.at_short = flag;
	    pattr->at_flags |= ATR_VFLAG_SET|ATR_VFLAG_MODIFY;
	}

	if( slen >= 512 )		/*buffer on heap, not stack*/
	    free( sbufp );

	return rc;
}


/*
 * decode_ntype 
 * In this case, the two arguments that get  used are
 * pattr-- it points to an attribute whose value is a short,
 * and the argument "val".
 * Although we have only "time-shared" and "cluster" at this
 * point in PBS's evolution, there may come a time when other
 * ntype values are needed.  The one thing that is assumed is
 * that the types are going to be mutually exclusive
 */
 
int	decode_ntype(pattr, name, rescn, val)
        attribute *pattr;
        char *name;             /* attribute name */
        char *rescn;            /* resource name, unused here */
        char *val;              /* attribute value */

{
	int	rc = 0;		/*return code; 0==success*/
	short	tmp = 0;


	if( val == (char *)0 )
	   rc = (PBSE_BADNDATVAL);
	else if( !strcmp(val, ND_timeshared) )
	   tmp = NTYPE_TIMESHARED;
	else if( !strcmp(val, ND_cluster) )
	   tmp = NTYPE_CLUSTER;
	else
	   rc = (PBSE_BADNDATVAL);


	if( !rc )
	{
	    pattr->at_val.at_short = tmp;
	    pattr->at_flags |= ATR_VFLAG_SET|ATR_VFLAG_MODIFY;
	}

	return rc;
}


/*
 * free_prop_list
 * For each element of a null terminated prop list call free
 * to clean up any string buffer that hangs from the element.
 * After this, call free to remove the struct prop.
 */

void	free_prop_list( prop )
	struct prop	*prop;
{
	struct prop	*pp;

	while( prop ) {
	   pp = prop->next;
	   free(prop);
	   prop = pp;
	}
}


/*
 * set_node_state - the information in the "short" attribute, *new, is used to
 *		    update the information in the "short" attribute, *pattr.
 *		    the mode of the update is goverened by the argument "op"
 *		    (SET,INCR,DECR)
 */

int	set_node_state( pattr, new, op)
	attribute	*pattr;		/*attribute gets modified    */
	attribute	*new;		/*carries new, modifying info*/
					/*that got decoded into 'new"*/
	enum batch_op	 op;
{
	int	rc = 0;

	assert( pattr && new && (new->at_flags & ATR_VFLAG_SET) );

	switch( op ) {

	    case SET:
	    pattr->at_val.at_short = new->at_val.at_short;
	    break;

	    case INCR:
	    if(pattr->at_val.at_short && new->at_val.at_short == 0) {

			rc = PBSE_BADNDATVAL;  /*"free" mutually exclusive*/
			break;
	    }

	    pattr->at_val.at_short |= new->at_val.at_short;
	    break;

	    case DECR:
	    if( pattr->at_val.at_short && new->at_val.at_short == 0 ) {

		       rc = PBSE_BADNDATVAL;  /*"free" mutually exclusive*/
		       break;
	    }
	    if( pattr->at_val.at_short & new->at_val.at_short
		!= new->at_val.at_short ) {

			rc = PBSE_BADNDATVAL;  /*not subset of existing*/
			break;
	    }

	    pattr->at_val.at_short &= ~new->at_val.at_short;
		   break;
		   default:	rc = PBSE_INTERNAL;
	}

	if ( !rc )
	     pattr->at_flags |= ATR_VFLAG_SET | ATR_VFLAG_MODIFY;
	return rc;
}


/*
 * set_node_ntype - the value entry in attribute "new" is a short.  It was
 *		    generated by the decode routine is used to update the
 *		    value portion of the attribute *pattr
 *		    the mode of the update is goverened by the argument "op"
 *		    (SET,INCR,DECR)
 */

int	set_node_ntype ( pattr, new, op)
	attribute	*pattr;		/*attribute gets modified    */
	attribute	*new;		/*carries new, modifying info*/
					/*that got decoded into 'new"*/
	enum batch_op	 op;
{
	int	rc = 0;

	assert( pattr && new && (new->at_flags & ATR_VFLAG_SET) );

	switch( op ) {

	    case SET:
	    pattr->at_val.at_short = new->at_val.at_short;
	    break;

	    case INCR:
	    if( pattr->at_val.at_short != new->at_val.at_short ) {

			rc = PBSE_MUTUALEX;  /*types are mutually exclusive*/
	    }
	    break;

	    case DECR:
	    if( pattr->at_val.at_short != new->at_val.at_short )
			rc = PBSE_MUTUALEX;  /*types are mutually exclusive*/

	    else if( pattr->at_val.at_short == NTYPE_TIMESHARED )
			pattr->at_val.at_short = NTYPE_CLUSTER;

	    else
			pattr->at_val.at_short = NTYPE_TIMESHARED;

	    break;

	    default:	rc = PBSE_INTERNAL;
	}

	if ( !rc )
	     pattr->at_flags |= ATR_VFLAG_SET | ATR_VFLAG_MODIFY;
	return rc;
}


/*
 * set_nodeflag - Use the input string's value to set a bit in
 *		  the "flags" variable pointed to by pflags
 *		  Each call sets one more bit in the flags
 *		  variable or it clears the flags variable
 *		  in the case where *str is the value "free"
 */

static int	set_nodeflag( str, pflag)
		char	*str;
		short	*pflag;
{
	int	rc = 0;

	if( *str == '\0' )
	    return (PBSE_BADNDATVAL);

	if( !strcmp(str, ND_free) )
	    *pflag = 0;
	else if( !strcmp(str, ND_offline) )
	    *pflag = *pflag | INUSE_OFFLINE ;
	else if( !strcmp(str, ND_down) )
	    *pflag = *pflag | INUSE_DOWN ;
	else if( !strcmp(str, ND_reserve) )
	    *pflag = *pflag | INUSE_RESERVE ;
	else {
	     rc = PBSE_BADNDATVAL;
	}
		
	return rc;
}


/*
 * node_state - Either derive a "state" attribute from the node
 *		or update node's "inuse" field using the "state"
 *		attribute.
 */

int	node_state( new, pnode, actmode)
	attribute	*new;		/*derive state into this attribute*/
	void		*pnode;		/*pointer to a pbsnode struct    */
	int		 actmode;	/*action mode; "NEW" or "ALTER"   */
{
	int rc = 0;
	struct pbsnode* np;


	np = (struct pbsnode*)pnode;	/*because of def of at_action  args*/
	switch( actmode ) {

	    case ATR_ACTION_NEW:  /*derive attribute*/
	    new->at_val.at_short = np->nd_state;	
	    break;

	    case ATR_ACTION_ALTER:
	    np->nd_state = new->at_val.at_short;
	    break;

	    default: rc = PBSE_INTERNAL;
	}
	return rc;
}


/*
 * node_ntype - Either derive an "ntype" attribute from the node
 *		or update node's "ntype" field using the 
 *		attribute's data
 */

int	node_ntype ( new, pnode, actmode)
	attribute	*new;		/*derive ntype into this attribute*/
	void		*pnode;		/*pointer to a pbsnode struct     */
	int		 actmode;	/*action mode; "NEW" or "ALTER"   */
{
	int rc = 0;
	struct pbsnode* np;


	np = (struct pbsnode*)pnode;	/*because of def of at_action  args*/
	switch( actmode ) {

	    case ATR_ACTION_NEW:  /*derive attribute*/
	    new->at_val.at_short = np->nd_ntype;	
	    break;

	    case ATR_ACTION_ALTER:
	    np->nd_ntype = new->at_val.at_short;
	    break;

	    default: rc = PBSE_INTERNAL;
	}
	return rc;
}

/*
 * node_prop_list - Either derive a "prop list" attribute from the node
 *		    or update node's prop list from attribute's prop list.
 */

int	node_prop_list( new, pnode, actmode)
	attribute	*new;		/*derive props into this attribute*/
	void		*pnode;		/*pointer to a pbsnode struct     */
	int		 actmode;	/*action mode; "NEW" or "ALTER"   */
{
	int		 rc = 0, i;
	struct pbsnode  *np;
	attribute	 temp;

	

	np = (struct pbsnode*)pnode;	/*because of at_action arg type*/

	switch( actmode ) {

	    case ATR_ACTION_NEW:

		/* if node has a property list, then copy array_strings    */
		/* into temp to use to setup a copy, otherwise setup empty */

		if (np->nd_prop) {

			/* setup temporary attribute with the array_strings */
			/* from the node				    */

			temp.at_val.at_arst = np->nd_prop;
			temp.at_flags = ATR_VFLAG_SET;
			temp.at_type  = ATR_TYPE_ARST;
			rc = set_arst(new, &temp, SET);
		} else {

			/* Node has no properties, setup empty attribute */

			new->at_val.at_arst = 0;
			new->at_flags       = 0;
			new->at_type        = ATR_TYPE_ARST;
		}
	    	break;
				

	    case ATR_ACTION_ALTER:

		/* update node with new attr_strings */

		np->nd_prop = new->at_val.at_arst;

		/* update number of properties listed in node */
		/* does not include name and subnode property */

		if (np->nd_prop)
			np->nd_nprops = np->nd_prop->as_usedptr;
		else
			np->nd_nprops = 0;
		break;

	    default:	rc = PBSE_INTERNAL;
	}

	return rc;
}
