/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
/*
 * get_server
 *
 * ------------------------------------
 * As specified in section 5 of the ERS:
 *
 *  5.1.2.  Directing Requests to Correct Server
 *
 *  A  command  shall  perform  its  function  by  sending   the
 *  corresponding  request  for  service  to the a batch server.
 *  The choice of batch servers to which to send the request  is
 *  governed by the following ordered set of rules:
 *
 *  1. For those commands which require or accept a job identif-
 *     ier  operand, if the server is specified in the job iden-
 *     tifier operand as @server, then the batch  requests  will
 *     be sent to the server named by server.
 *
 *  2. For those commands which require or accept a job identif-
 *     ier  operand  and  the @server is not specified, then the
 *     command will attempt to determine the current location of
 *     the  job  by  sending  a  Locate Job batch request to the
 *     server which created the job.
 *  
 *  3. If a server component of a destination  is  supplied  via
 *     the  -q  option,  such  as  on  qsub and qselect, but not
 *     qalter, then the server request is sent to that server.
 *  
 *  4. The server request is sent to the  server  identified  as
 *     the default server, see section 2.6.3.
 *     [pbs_connect() implements this]
 *  
 *  2.6.3.  Default Server
 *  
 *  When a server is not specified to a client, the client  will
 *  send  batch requests to the server identified as the default
 *  server.  A client identifies the default server by  (a)  the
 *  setting  of  the environment variable PBS_DEFAULT which con-
 *  tains a destination, or (b) the  destination  in  the  batch
 *  administrator established file {PBS_DIR}/default_destn.
 * ------------------------------------
 *  
 * Takes a job_id_in string as input, calls parse_jobid to separate 
 * the pieces, then applies the above rules in order
 * If things go OK, the function value is set to 0, 
 * if errors, it is set to 1.
 *
 * Full legal syntax is:
 *  seq_number[.parent_server[:port]][@current_server[:port]]  
 *
 */

#include <pbs_config.h>   /* the master config generated by configure */

#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <netdb.h>
#include "pbs_ifl.h"
#include "net_connect.h"

static char ident[] = "@(#) $RCSfile: get_server.c,v $ $Revision: 2.1 $";

#define notNULL(x) (((x)!=NULL) && (strlen(x)>(size_t)0))

extern int parse_jobid A_((char *jodid, char **seq, char **parent, char **current));

int
get_server(job_id_in, job_id_out, server_out)
char *job_id_in;    /* read only */
char *job_id_out;   /* write only */
char *server_out;   /* write only */
{
    char *seq_number;
    char *parent_server;
    char *current_server;
    char def_server[PBS_MAXSERVERNAME+1];
    char host_server[PBS_MAXSERVERNAME+1];
    char *c;

    /* parse the job_id_in into components */

    if ( parse_jobid(job_id_in, &seq_number, &parent_server, &current_server) )
        return (1);

    /* Apply the above rules, in order, except for the locate job request.
       That request is only sent if the job is not found on the local server.
     */

    if ( notNULL(current_server) ) {		/* @server found */
        strcpy(server_out, current_server);
    } else if ( notNULL(parent_server) ) {	/* .server found */
        strcpy(server_out, parent_server);
    } else {  /* can't locate a server, so return a NULL 
               to tell pbs_connect to use default */
        server_out[0] = '\0';
    }
    
    /* Make a fully quaified name of the job id. */

    strcpy(job_id_out, seq_number);
    strcat(job_id_out, ".");
    if ( notNULL(parent_server) ) {
        if (get_fullhostname(parent_server,host_server,PBS_MAXSERVERNAME) ==0) {
            strcat(job_id_out, host_server);
	    if (c = strchr(parent_server, (int)':')) {
		if (*(c-1) == '\\') c--;
		strcat(job_id_out, c);
	    }
        } else
            return 1;
    } else {
	parent_server = pbs_default();
	if( (parent_server == (char *)NULL) || (*parent_server == '\0') ) {
            return 1;
	} else {
	    strncpy(def_server, parent_server, PBS_MAXSERVERNAME);
	}
        c = def_server;
        while ( *c != '\n' && *c != '\0' ) c++;
        *c = '\0';
        if (get_fullhostname(def_server, host_server, PBS_MAXSERVERNAME) == 0) {
            strcat(job_id_out, host_server);
	    if (c = strchr(def_server, (int)':')) {
		if (*(c-1) == '\\') c--;
		strcat(job_id_out, c);
	    }
        } else
            return 1;
    }
    return 0;
}
